// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_DOMAIN_TRAITS_LOC_H
#define POOMA_DOMAIN_DOMAIN_TRAITS_LOC_H

//-----------------------------------------------------------------------------
// Class:
// DomainTraits<Loc<N>>
// DomainChangeDim<Loc<N>,Dim>
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview:
// DomainTraits<Loc<N>> is a specialization of the general DomainTraits
// class, for the case of Loc domain objects.  It defines the general
// behavior of Loc, including its typedefed and enumerated characteristics,
// how to store data for a Loc, etc.  It is used by the Domain base class
// of Loc to implement most of the public interface.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Domain/DomainTraits.int.h"
#include "Utilities/UninitializedVector.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<int Dim> class Loc;
template<int Dim> class Interval;
template<int Dim> class Range;


//-----------------------------------------------------------------------------
//
// Full Description:
//
// DomainTraits<Loc<Dim>> stores the characteristics and much of the
// implementation details for Loc domain objects.  A Loc acts like a
// single integer point in N-dimensional space, so it is a single-valued,
// unit-stride domain.
//
// A general version of DomainTraits<Loc<Dim>> is defined here, which
// only includes the basic information to make Loc<Dim> look like an
// array of Loc<1> objects.  DomainTraits<Loc<1>> is a more specific
// specialization which provides most of the necessary interface information
// for items which need to know about Loc.  Since most of the interface
// for a domain object is only available for 1D versions of that domain
// object, the Loc<1> specialization defines more interface functions than
// the Loc<Dim> case.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// DomainTraits<Loc<Dim>>:
// The specialization of DomainTraits for Loc, for dimensions greater than
// one.
//-----------------------------------------------------------------------------

template<int Dim>
struct DomainTraits< Loc<Dim> >
  : public DomainTraitsDomain<Loc<Dim>, int, Dim>
{
  // necessary typedefs
  typedef typename DomainTraitsDomain<Loc<Dim>, int, Dim>::Element_t Element_t;
  typedef typename DomainTraitsDomain<Loc<Dim>, int, Dim>::Domain_t Domain_t;
  typedef typename DomainTraitsDomain<Loc<Dim>, int, Dim>::NewDomain1_t 
    NewDomain1_t;

  // necessary enums
  enum { domain = DomainTraitsDomain<Loc<Dim>, int, Dim>::domain };
  enum { dimensions = DomainTraitsDomain<Loc<Dim>, int, Dim>::dimensions };

  // other necessary typedefs
  typedef Loc<1>        OneDomain_t;
  typedef Loc<1>        PointDomain_t;
  typedef Interval<Dim> BlockDomain_t;
  typedef Loc<Dim>      AskDomain_t;
  typedef Loc<Dim>      AddResult_t;
  //typedef Range<Dim>    MultResult_t;
  typedef Loc<Dim>      MultResult_t;

  // type for storage of this domain's data
  typedef UninitializedVector<OneDomain_t,Dim,Element_t> Storage_t;

  // other necessary enums
  enum { sliceDimensions = 0 };
  enum { loopAware = 0 };
  enum { singleValued = 1 };
  enum { unitStride = 1 };
  enum { wildcard = 0 };

  // get the Nth element of the domain, and return a OneDomain_t
  // object with it (here, as a copy).
  inline
  static OneDomain_t &getDomain(Domain_t &d, int n) {
    return d[n];
  }
  inline
  static const OneDomain_t &getDomain(const Domain_t &d, int n) {
    return d[n];
  }

  // convert from the Nth element of the domain to a single point, and
  // return a PointDomain_t.
  inline
  static PointDomain_t &getPointDomain(Domain_t &d, int n) {
    return d[n];
  }
  inline
  static const PointDomain_t &getPointDomain(const Domain_t &d, int n) {
    return d[n];
  }

  // Domains get the chance to do special initialization.
  static void initializeStorage(Storage_t &dom) { dom.initialize(); }


  // addAccum means dom += newdom
  template<class T>
  inline
  static void addAccum(Storage_t &dom, const T &newdom) 
  {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     (DomainTraits<T>::dimensions == 1 || 
	      DomainTraits<T>::dimensions == dimensions ) );

    if (DomainTraits<T>::dimensions > 1)
      for (int i = 0;i< DomainTraits<T>::dimensions ; ++i)
	dom[i] += DomainTraits<T>::getFirst(newdom[i]);
    else
      for (int i = 0;i< dimensions ; ++i)
	dom[i] += DomainTraits<T>::getFirst(newdom[0]);
  }

  // subtractAccum means dom -= newdom
  template<class T>
  inline
  static void subtractAccum(Storage_t &dom, const T &newdom) 
  {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     (DomainTraits<T>::dimensions == 1 || 
	      DomainTraits<T>::dimensions == dimensions ) );
    if (DomainTraits<T>::dimensions > 1)
      for (int i = 0;i< DomainTraits<T>::dimensions ; ++i)
	dom[i] -= DomainTraits<T>::getFirst(newdom[i]);
    else
      for (int i = 0;i< dimensions ; ++i)
	dom[i] -= DomainTraits<T>::getFirst(newdom);
  }

  template<class T>
  static void multiplyAccum(Storage_t &dom, const T &newdom) 
  {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     (DomainTraits<T>::dimensions == 1 || 
	      DomainTraits<T>::dimensions == dimensions ) );
    if (DomainTraits<T>::dimensions > 1)
      for (int i = 0;i< DomainTraits<T>::dimensions ; ++i)
	dom[i] *= DomainTraits<T>::getFirst(newdom[i]);
    else
      for (int i = 0;i< dimensions ; ++i)
	dom[i] *= DomainTraits<T>::getFirst(newdom);

  }
  
  template<class T>
  static void divideAccum(Storage_t &dom, const T &newdom) 
  {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     (DomainTraits<T>::dimensions == 1 || 
	      DomainTraits<T>::dimensions == dimensions ) );
    if (DomainTraits<T>::dimensions > 1)
      for (int i = 0;i< DomainTraits<T>::dimensions ; ++i)
	dom[i] /= DomainTraits<T>::getFirst(newdom[i]);
    else
      for (int i = 0;i< dimensions ; ++i)
	dom[i] /= DomainTraits<T>::getFirst(newdom);

  }

};


//-----------------------------------------------------------------------------
// DomainTraits<Loc<1>>:
// The specialization of DomainTraits for Loc, for dimension == 1.
//-----------------------------------------------------------------------------

template<>
struct DomainTraits< Loc<1> >
  : public DomainTraitsDomain<Loc<1>, int, 1>
{
  // necessary typedefs
  //typedef DomainTraitsDomain<Loc<1>, int, 1>::Element_t Element_t;
  //typedef DomainTraitsDomain<Loc<1>, int, 1>::Domain_t Domain_t;

  // other necessary typedefs
  typedef Loc<1>      OneDomain_t;
  typedef Loc<1>      PointDomain_t;
  typedef Interval<1> BlockDomain_t;
  typedef Loc<1>      AskDomain_t;
  typedef Loc<1>      AddResult_t;
  typedef Loc<1>      MultResult_t;

  // 1D necessary typedefs.  Loc's store just a single integer, which is the
  // point.  They cannot represent empty domains, and always have length == 1,
  // stride == 1.
  typedef Element_t   Storage_t;

  // other necessary enums
  enum { sliceDimensions = 0 };
  enum { loopAware = 0 };
  enum { singleValued = 1 };
  enum { unitStride = 1 };
  enum { wildcard = 0 };

  // return size, endpoint, stride, and loop information from the storage
  // data for this domain
  inline
  static Element_t first(Storage_t d)    { return d; }
  inline
  static Element_t last(Storage_t d)     { return d; }
  inline
  static Element_t stride(Storage_t)     { return 1; }
  inline
  static Element_t length(Storage_t)     { return 1; }
  inline
  static Element_t min(Storage_t d)      { return d; }
  inline
  static Element_t max(Storage_t d)      { return d; }
  inline
  static bool      empty(Storage_t)      { return false; }
  inline
  static int       loop(Storage_t)       { return 0; }

  // get the Nth value of the domain, where value # 0 is first(), etc.
  inline
  static Element_t elem(Storage_t d, int) { return d; }

  // get the Nth element of the domain, and return a OneDomain_t
  // object with it (here, as a copy).
  inline
  static OneDomain_t &getDomain(Domain_t &d, int) {
    return d;
  }
  inline
  static const OneDomain_t &getDomain(const Domain_t &d, int) {
    return d;
  }

  // convert from the Nth element of the domain to a single point, and
  // return a PointDomain_t.
  inline
  static PointDomain_t &getPointDomain(Domain_t &d, int) {
    return d;
  }
  inline
  static const PointDomain_t &getPointDomain(const Domain_t &d, int) {
    return d;
  }

  // Domains get the chance to do special initialization.
  // 1D Loc's are initialized to zero.
  inline
  static void initializeStorage(Storage_t &dom) {
    dom = 0;
  }

  // change this domain object to the given one.  If things do not
  // match properly, assert a compile-time or run-time error.
  // For Loc, we must have:
  // 1) the same dimensions==1
  // 2) length() == 1 for the new domain
  // NOTE: this could be changed to a more rigorous compile-time check
  // that the copied domain is singleValued, if that is considered important
  // for performance.
  template<class T>
  inline
  static void setDomain(Storage_t &dom, const T &newdom) {
    CTAssert(DomainTraits<T>::dimensions == 1);
    PAssert(DomainTraits<T>::getLength(newdom) == 1);
    dom = DomainTraits<T>::getFirst(newdom);
  }

  // change the loop variable for this object.  For Loc, this is a no-op.
  inline
  static void setLoop(Storage_t &, int) { }

  // change the value of this 1D domain given a user-supplied reference
  // domain and a wildcard.
  template<class UT, class T>
  inline
  static void setWildcardDomain(Storage_t &dom, const UT &u, const T &newdom) {
    CTAssert(DomainTraits<T>::wildcard == 1);
    CTAssert(DomainTraits<T>::dimensions == 1);
    CTAssert(DomainTraits<UT>::dimensions == 1);
    dom = newdom.first(u);	// uses wildcard version of first()
  }

  //
  // compare this domain type to the given domain.  For the comparisons
  // to be meaningful for Loc, we must have:
  // 1) the same dimensions==1
  // 2) length() == 1 for the new domain
  // NOTE: this could be changed to a more rigorous compile-time check
  // that the copied domain is singleValued, if that is considered important
  // for performance.
  //

  // 'isLessThan' returns true if dom < newdom
  template<class T>
  static bool isLessThan(const Storage_t &dom, const T &newdom) {
    CTAssert(DomainTraits<T>::dimensions == 1);
    PAssert(DomainTraits<T>::getLength(newdom) == 1);
    return (dom < DomainTraits<T>::getFirst(newdom));
  }

  // 'isEqualTo' returns true if dom == newdom
  template<class T>
  static bool isEqualTo(const Storage_t &dom, const T &newdom) {
    CTAssert(DomainTraits<T>::dimensions == 1);
    PAssert(DomainTraits<T>::getLength(newdom) == 1);
    return (dom == DomainTraits<T>::getFirst(newdom));
  }

  //
  // arithmetic accumulation operators.  These only work with
  // other domain objects with the following characteristics:
  // 1) they are singleValue'd
  // 2) they have dimensions == 1 -or- the have dimension 
  // equal to the dimension of *this
  //
  // Note that for Locs, we do NOT allow *= or /=.  You
  // must convert a Loc to a Range before doing multiplicative operations.
  //

  // addAccum means dom += newdom
  template<class T>
  inline
  static void addAccum(Storage_t &dom, const T &newdom) {
   
    CTAssert(DomainTraits<T>::singleValued == 1 &&
    	     DomainTraits<T>::dimensions == 1);
   dom += DomainTraits<T>::getFirst(newdom);

  }

  // subtractAccum means dom -= newdom
  template<class T>
  inline
  static void subtractAccum(Storage_t &dom, const T &newdom) {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     DomainTraits<T>::dimensions == 1);
    dom -= DomainTraits<T>::getFirst(newdom);
  }

  template<class T>
  static void multiplyAccum(Storage_t &dom, const T &newdom) {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     DomainTraits<T>::dimensions == 1);
    dom *= DomainTraits<T>::getFirst(newdom);
  }
  
 template<class T>
  static void divideAccum(Storage_t &dom, const T &newdom) {
    CTAssert(DomainTraits<T>::singleValued == 1 &&
	     DomainTraits<T>::dimensions == 1);
    dom /= DomainTraits<T>::getFirst(newdom);
  }

};


//-----------------------------------------------------------------------------
//
// Full Description:
//
// DomainChangeDim<T, int> is used to convert from a domain of one dimension
// to another dimension (the second template parameter).
// For Loc<Dim1>, it changes from Dim1 to Dim2.
//
//-----------------------------------------------------------------------------

template<int Dim1, int Dim2>
struct DomainChangeDim<Loc<Dim1>, Dim2>
{
  // the type of the old and new domain
  typedef Loc<Dim1> OldType_t;
  typedef Loc<Dim2> NewType_t;

  // enumerations with old and new dimensions
  enum { oldDim = Dim1 };
  enum { newDim = Dim2 };
};


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_DOMAIN_TRAITS_LOC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DomainTraits.Loc.h,v $   $Author: sa_smith $
// $Revision: 1.28 $   $Date: 2000/05/25 20:15:50 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
