// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_ENGINE_ENGINE_H
#define POOMA_ENGINE_ENGINE_H

//-----------------------------------------------------------------------------
// Classes:
//   Engine<Dim,T,EngineTag>
//   NewEngine<Engine,SubDomain>
//   NewEngineEngine<Engine,SubDomain>
//   NewEngineDomain<Engine,SubDomain>
//   EngineConstructTag
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace Pooma {

//-----------------------------------------------------------------------------
// Overview:
//
// Engine<Dim,T,EngineTag>
//  - General Engine template.
// NewEngine<Engine,SubDomain>
//  - Traits class used for taking views.
// NewEngineEngine<Engine,SubDomain>
// NewEngineDomain<Engine,SubDomain>
//  - Optional functors that can be used to simplify view construction.
// EngineConstructTag
//  - tag class to disambiguate certain constructors.
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Full Description:
// Engine<Dim,T,EngineTag>
//
// General Engine template. All concrete Engine classes specialize this
// template for a particular Engine tag.
//-----------------------------------------------------------------------------

template <int Dim, class T, class EngineTag>
class Engine;


//-----------------------------------------------------------------------------
//
// Full Description:
// NewEngine<Engine,SubDomain>
//
// Traits class for determining the type obtained by subsetting a 
// particular Engine with a particular SubDomain.
//
// Concrete Engine classes will make specializations of this class
// for the pairs that can result in that particular Engine being created.
//
//-----------------------------------------------------------------------------

template <class Engine, class SubDomain>
struct NewEngine
{
};

//-----------------------------------------------------------------------------
//
// Full Description:
// NewEngineEngine<Engine,SubDomain>
// NewEngineDomain<Engine,SubDomain>
//
// These two traits classes allow you to modify the engine and domain
// that get passed to the view engine that results from a subset
// operation.  This indirection allows you to define view operations
// without requiring that the view engine knows about the engine you
// start with (for example, BrickView shouldn't have to know about
// patch engines that contain it).
//
// The natural location for these functors is inside the NewEngine
// traits class, but defining them separately allows us to provide
// the default behaviour: just forward the engine and domain through.
//
//-----------------------------------------------------------------------------

template <class Engine, class SubDomain>
struct NewEngineEngine
{
  typedef Engine Type_t;
  static inline
  const Engine &apply(const Engine &e, const SubDomain &)
  {
    return e;
  }
} ;

template <class Engine, class SubDomain>
struct NewEngineDomain
{
  typedef SubDomain Type_t;
  typedef const SubDomain &Return_t;
  static inline
  Return_t apply(const Engine &, const SubDomain &i)
  {
    return i;
  }
} ;


template<class Eng, class Dom>
inline typename NewEngineEngine<Eng, Dom>::Type_t
newEngineEngine(const Eng &e, const Dom &dom)
{
  return NewEngineEngine<Eng, Dom>::apply(e, dom);
}


template<class Eng, class Dom>
inline typename NewEngineDomain<Eng, Dom>::Type_t
newEngineDomain(const Eng &e, const Dom &dom)
{
  return NewEngineDomain<Eng, Dom>::apply(e, dom);
}


//-----------------------------------------------------------------------------
// EngineConstructTag is used by Array to disambiguate
// engine-based constructor calls.  (also used by some engines)
//-----------------------------------------------------------------------------

struct EngineConstructTag 
{
  EngineConstructTag() { };
  ~EngineConstructTag() { };
  EngineConstructTag(const EngineConstructTag &) { };
  EngineConstructTag &operator=(const EngineConstructTag &) { return *this; }
};

// } // namespace Pooma
///////////////////////////////////////////////////////////////////////////////

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Engine.h,v $   $Author: swhaney $
// $Revision: 1.25 $   $Date: 2000/07/20 15:39:26 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
