// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// BrickEngine test code
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/PAssert.h"
#include "Utilities/Tester.h"
#include "Domain/Interval.h"
#include "Domain/Range.h"
#include "Engine/BrickEngine.h"

#include <iomanip>

typedef Engine<1,double,Brick> Array_t;
typedef Engine<2,double,Brick> Array2_t;
typedef Engine<1,double,BrickView>  View1_t;
typedef Engine<1,double,BrickView> View2_t;

void print(const Array2_t &, Pooma::Tester &tester);

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);
#if POOMA_EXCEPTIONS
  try {
#endif
    tester.out() << "\nTesting BrickEngine with Fortran-like offsets."
              << std::endl;

    Interval<1> I(1,4);
    Interval<2> II(I,I);

    Array2_t a(II), b(II), c(II);

    for (int i = 1; i <=4; i++)
      for (int j = 1; j <= 4; j++)
	{
	  a(i,j) = i;
	  b(i,j) = j;
	}

    for (int i = 1; i <=4; i++)
      for (int j = 1; j <= 4; j++)
	c(i,j) = 0.0;

    for (int i = 1; i <=4; i++)
      for (int j = 1; j <= 4; j++)
	for (int k = 1; k <= 4; k++)
	  {
	    c(i,j) += a(i,k)*b(k,j);
	  }

    tester.out() << "\na = " << std::endl;

    print(a,tester);

    tester.out() << "\nb = " << std::endl;

    print(b,tester);

    tester.out() << "\nc = matmul(a,b) = " << std::endl;

    print(c,tester);
#if POOMA_EXCEPTIONS
  }
  catch(const char *err) 
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
  catch(const Pooma::Assertion &err)
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
#endif    
  int ret = tester.results("brick_test4");
  Pooma::finalize();
  return ret;
}

void print(const Array2_t &a, Pooma::Tester &tester)
{
  for (int i = 1; i <= 4; i++)
    {
      for (int j = 1; j <=4; j++)
	{
	  tester.out() << std::setw(4) << a(i,j) << " ";
	}
      tester.out() << std::endl;
    }
}

	     
// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: brick_test4.cpp,v $   $Author: sa_smith $
// $Revision: 1.13 $   $Date: 2000/07/11 23:06:54 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
