// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
//   ReverseBC
//-----------------------------------------------------------------------------

#ifndef POOMA_PARTICLES_REVERSEBC_H
#define POOMA_PARTICLES_REVERSEBC_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//-----------------------------------------------------------------------------
// 
// A reversing boundary condition for Particles.
// When subject Attribute value goes outside boundary by some amount,
// ReverseBC sets value to be inside boundary by that amount.
// In addition, it flips the sign of the object Attribute.
// The typical use for this would be to bounce a particle off of
// a wall by replacing it inside the wall and reversing its velocity.
//
//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Particles/ParticleBC.h"
#include <iostream>

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
//-----------------------------------------------------------------------------
//
// Here is an example of adding a new particle boundary condition.
// First, add a BCType that inherits from ParticleBCType, describes 
// a boundary condition, and stores any data that is needed.
// Then, we specialize ParticleBC for this BCType.
//
//-----------------------------------------------------------------------------


template <class T>
class ReverseBC : public ParticleBCType< ReverseBC<T> >
{
public:

  // Constructors.
  ReverseBC(T min, T max)
    : MinVal_m(min), MaxVal_m(max) {}
  ReverseBC(const ReverseBC<T>& model)
    : MinVal_m(model.min()), MaxVal_m(model.max()) {}

  // Destructor.
  ~ReverseBC() {}

  // Methods.

  // Assignment operator does deep copy.
  ReverseBC<T>& operator=(const ReverseBC<T>& rhs)
  {
    MinVal_m = rhs.min();
    MaxVal_m = rhs.max();
    return *this;
  }

  // Accessor functions
  T min() const { return MinVal_m; }
  T max() const { return MaxVal_m; }
  T& min() { return MinVal_m; }
  T& max() { return MaxVal_m; }

private:

  // extents of ReverseBC region
  T MinVal_m, MaxVal_m;
};



template <class Subject, class Object, class T>
class ParticleBC< Subject, Object, ReverseBC<T> >
  : public ParticleBCItem
{
public:

  // Typedefs.
  typedef Subject Subject_t;
  typedef Object Object_t;
  typedef ReverseBC<T> BCType_t;
  typedef ParticleBC<Subject_t,Object_t,BCType_t> This_t;

  // Constructors.
  ParticleBC(const Subject_t& s, const Object_t& o,
             const BCType_t& bc)
    : subject_m(s), object_m(o), bc_m(bc) { }
  ParticleBC(const This_t& model)
    : subject_m(model.subject()),
      object_m(model.object()),
      bc_m(model.bc()) { }

  // Destructor.
  ~ParticleBC() { }

  // Methods.

  // Accessors for subject and object.

  const Subject_t& subject() const { return subject_m; }
  const Object_t& object() const { return object_m; }

  // Access boundary condition.
  const BCType_t& bc() const { return bc_m; }

  // Apply boundary condition.
  void applyBoundaryCondition(int pid);

  // Print out to an ostream.
  void print(std::ostream& o) const
    {
      o << "BC Type: Reverse, Range: (" << bc_m.min()
        << "," << bc_m.max() << ")" << std::endl;
    }

private:

  // Can't call default constructor.
  ParticleBC() { }

  // Subject of the boundary condition. 
  Subject_t subject_m;

  // Object of the boundary condition.
  Object_t object_m;

  // Our boundary condition.
  const BCType_t& bc_m;
};

#include "Particles/ReverseBC.cpp"

#endif     // POOMA_PARTICLES_REVERSEBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ReverseBC.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/03/07 13:17:51 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
