// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
//   ReduceOverContexts<T, Op>
//-----------------------------------------------------------------------------

#ifndef POOMA_CHEETAH_REDUCEOVERCONTEXTS_H
#define POOMA_CHEETAH_REDUCEOVERCONTEXTS_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Tulip/Messaging.h"
#include "Tulip/RemoteProxy.h"


//-----------------------------------------------------------------------------
//
// ReductionValue<T>
//
// This class associates a value with a flag that indicates whether or not
// it is valid. It takes special care to not read the value if it is invalid.
//
//-----------------------------------------------------------------------------

template<class T>
class ReductionValue
{
public:

  ReductionValue(bool valid, const T &val)
    : valid_m(valid)
    {
      if (valid_m)
	val_m = val;
    }

  ReductionValue(const ReductionValue<T> &model)
    {
      valid_m = model.valid();
      if (valid_m)
	val_m = model.value();
    }

  ReductionValue<T> &operator=(const ReductionValue<T> &rhs)
    {
      if (&rhs != this)
	{
	  valid_m = rhs.valid();
	  if (valid_m)
	    val_m = rhs.value();
	}

      return *this;
    }

  bool valid() const { return valid_m; }
  const T &value() const { PAssert(valid()); return val_m; }

private:

  bool valid_m;
  T val_m;
};


//-----------------------------------------------------------------------------
//
// Cheetah::Serialize<CHEETAH, ReductionValue<T> >
//
// This class is used to serialize ReductionValue<T> objects, taking care
// not to send invalid values.
//
//-----------------------------------------------------------------------------

#if POOMA_CHEETAH

namespace Cheetah {

template<class T>
class Serialize<CHEETAH, ReductionValue<T> >
{
public:

  static inline int size(const ReductionValue<T> &v)
  {
    int nBytes = Serialize<CHEETAH, bool>::size(v.valid());
    if (v.valid())
      nBytes += Serialize<CHEETAH, T>::size(v.value());

    return nBytes;
  }

  static inline int pack(const ReductionValue<T> &v, char *buffer)
  {
    int nBytes = Serialize<CHEETAH, bool>::pack(v.valid(), buffer); 

    if (v.valid())
      {
	nBytes += Serialize<CHEETAH, T>::pack(v.value(), buffer + nBytes); 
      }

    return nBytes;
  }

  static inline int unpack(ReductionValue<T>* &vp, char *buffer)
  {
    bool *pvalid;
    T val, *pval = &val;

    int nBytes = Serialize<CHEETAH, bool>::unpack(pvalid, buffer);

    if (*pvalid)
      {
	nBytes += Serialize<CHEETAH, T>::unpack(pval, buffer + nBytes); 
      }

    vp = new ReductionValue<T>(*pvalid, *pval);

    return nBytes;
  }
  
  static inline void cleanup(ReductionValue<T> *vp)
  {
    delete vp;
  }
};

} // namespace Cheetah

#endif

//-----------------------------------------------------------------------------
//
// ReduceOverContextsBase
//
// This struct holds a few static quantities that are shared by all
// instantiations of ReduceOverContexts<T>. In particular, we want to 
// maintain a running sequence of tags across all instantiations. 
//
//-----------------------------------------------------------------------------

struct ReduceOverContextsBase
{
  // We use this as a counter to generate tags.

  static int tagBase_m;
};


//-----------------------------------------------------------------------------
//
// ReduceOverContexts<T>
//
// This class is used to implement the final reduction over contexts used
// in Reduction<RemoteMultiPatchTag>::evaluate().
//
//-----------------------------------------------------------------------------

template<class T, class ReductionOp>
class ReduceOverContexts : public ReduceOverContextsBase
{
  typedef ReduceOverContexts<T, ReductionOp> This_t;

public:

  // All the work happens in the constructor. If we're on the "to" context,
  // we set up to receive messages from all of the other contexts. The 
  // receive() handler performs the reduction over contexts incrementally as
  // we get the messages in. We poll until everything shows up. If we're
  // not on the "to" context, we send our value to there.

  // Things are slightly more complicated by the fact that we don't want
  // to read 'val' unless it is a valid value. Values don't have to
  // valid because not all contexts necessarily contribute to the reduction.

#if POOMA_CHEETAH

  ReduceOverContexts(const T &val, int toContext = 0, bool valid = true)
    : toContext_m(toContext)
    {
      if (valid)
        value_m = val;
        
      int tagBase = tagBase_m;
      tagBase_m += Pooma::contexts();

      if (Pooma::context() == toContext)
	{
	  toReceive_m = Pooma::contexts();
	  int fromContext;
	  for (fromContext = 0; fromContext < Pooma::contexts(); ++fromContext)
	    {
	      if (fromContext != toContext)
		{
		  Pooma::reductionHandler()->
		    request(fromContext, tagBase + fromContext, receive, this);
		}
	      else
		{
		  ReductionValue<T> v(valid, val);
		  receive(this, v);
		}
	    }
	  
	  while (toReceive_m != 0)
	    {
	      Pooma::poll();
	    }
	}
      else
	{
	  ReductionValue<T> v(valid, val);

	  Pooma::reductionHandler()->
	    send(toContext, tagBase + Pooma::context(), v);
	}
    }

  void broadcast(T &val)
    {
      RemoteProxy<T> broadcast(value_m, toContext_m);
      val = broadcast;
    }
      
#else

  ReduceOverContexts(const T &val, int = 0, bool valid = true)
    {
      PAssert(valid);
      value_m = val;
    }
    
  void broadcast(T &val)
    {
      val = value_m;
    }

#endif // POOMA_CHEETAH

  inline operator T() const { return value_m; }

private:

  // Handler function for cheetah.

  static void receive(This_t *me, ReductionValue<T> &v)
  {
    if (v.valid())
      {
	if (me->toReceive_m == Pooma::contexts())
	  {
	    me->value_m = v.value();
	  }
	else
	  {
	    ReductionOp()(me->value_m, v.value());
	  }
      }

    me->toReceive_m--;
  }
 
 // The actual value we're reducing.

  T value_m;

  // The number of messages we're receiving.

  int toReceive_m;
  
  // The context we're reducing on.
  
  int toContext_m;
};

#endif     // POOMA_CHEETAH_REDUCEOVERCONTEXTS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ReduceOverContexts.h,v $   $Author: sa_smith $
// $Revision: 1.8 $   $Date: 2000/06/08 22:16:59 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
