// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// Tester
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Utilities/Tester.h"
#include "Utilities/Inform.h"
#include "Utilities/PAssert.h"
#include <string>
#include <iostream>


//-----------------------------------------------------------------------------
// Create a default Tester object
//-----------------------------------------------------------------------------

Pooma::Tester::Tester()
  : ok_m(true), quiet_m(false), inform_m("Pooma"), 
    verbose_m(false), abort_m(false)
{
}


//-----------------------------------------------------------------------------
// Create a Tester by specifying argc,argv values.
//-----------------------------------------------------------------------------

Pooma::Tester::Tester(int argc, char **argv)
  : ok_m(true), quiet_m(false), inform_m("Pooma"), 
    verbose_m(false), abort_m(false)
{

  // Parse the command-line paramters

  parse(argc, argv);
}


//-----------------------------------------------------------------------------
// Destructor.
//-----------------------------------------------------------------------------

Pooma::Tester::~Tester()
{
}


//-----------------------------------------------------------------------------
// Print out a message to cout about the current status
//-----------------------------------------------------------------------------

int Pooma::Tester::results(const char *msg) const
{
  // Only print out a message if we're not supposed to be quiet

  if (!quiet_m)
    {
      Inform coutmsg;
      coutmsg << (ok_m ? "PASSED" : "FAILED");
      if (msg != 0)
	coutmsg << " ... " << msg;
      coutmsg << std::endl;
    }

  // Return the current value for the error code

  return returnValue();
}


//-----------------------------------------------------------------------------
// Print notification of exception.
//-----------------------------------------------------------------------------

void Pooma::Tester::exceptionHandler(const char *msg)
{
  if (verbose_m)
  {
    Inform exout("EXCEPTION");

    exout << "### Exception handled by Tester. ###\n"
	  << "### Exception message:\n";
  
    if (msg)
      exout << msg;
    else
      exout << "[none]";
  
    exout << std::endl;
  }
}

void Pooma::Tester::exceptionHandler(const Pooma::Assertion &asrt)
{
  if (verbose_m)
  {
    Inform exout("EXCEPTION");

    exout << "### POOMA Assertion Failure ###\n"
	  << "### " << asrt.what() << '\n'
	  << "### File " << asrt.file() << "; Line " << asrt.line()
	  << std::endl;
  }
}


//-----------------------------------------------------------------------------
// Parse the given command-line arguments in argc,argv, and use them to
// change the tester's behavior.  This will not modify argc or argv.
//
// Tester arguments:
//   -v       : turn on verbose output from test program
//   -p <str> : change prefix of test program messages to <str>
//   -q       : do not print out anything at all, just have program
//              return 0 or 1
//-----------------------------------------------------------------------------

void Pooma::Tester::parse(int argc, char **argv)
{
  int i = 1;
  while (i < argc)
    {
      std::string word(argv[i]);

      // Check the word for one we recognize, and set the proper variable.

      if (word == "-v")
	{
	  verbose_m = true;
	  quiet_m = false;
	}
      else if (word == "-p")
	{
	  if (i < (argc + 1))
	    out().setPrefix(argv[++i]);
	}
      else if (word == "-q")
	{
	  verbose_m = false;
	  quiet_m = true;
	}
      else if (word == "-abort")
        {
          abort_m = true;
        }

      // Move on to the next argument.

      ++i;
    }

  // Use the values from the flags to set up our state.

  // If we're not verbose, turn off the inform stream.

  if (!verbose_m || quiet_m)
    out().setOutputLevel(Inform::off);
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Tester.cmpl.cpp,v $   $Author: jac $
// $Revision: 1.13 $   $Date: 2000/06/28 19:50:31 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
