/*
 * Author: Heinz Mauelshagen, Germany
 *
 * February-November 1997
 * April-May 1998
 *
 * lvm is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * lvm is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/09/1997 - added chr ioctls VG_STATUS_GET_COUNT
 *                 and VG_STATUS_GET_NAMELIST
 *    01/18/1997 - change chr_open/close lock handling
 *    04/30/1998 - changed LV_STATUS ioctl to LV_STATUS_BYNAME and
 *               - added   LV_STATUS_BYINDEX ioctl
 *               - used lvm_status_byname_req_t and
 *                      lvm_status_byindex_req_t vars
 *    05/04/1998 - added multiple device support
 *    05/08/1998 - added support to set/clear extendable flag in volume group
 *    05/09/1998 - changed output of lvm_proc_get_info() because of
 *                 support for free (egg. longer) logical volume names
 *    05/12/1998 - added spin_locks (thanks to Pascal van Dam)
 *    05/25/1998 - fixed handling of locked PEs in lvm_map() and lvm_chr_ioctl()
 *    05/26/1998 - reactivated verify_area by access_ok
 *
 */

/*
 * TODO
 *
 *   - add code for LV specific read_ahead sectors?
 *   - implement own handling of unavailable physical volumes
 *
 */

#include <linux/config.h>

#ifdef MODVERSIONS
#  undef MODULE
#  define MODULE
#  include <linux/modversions.h>
#endif

#ifdef MODULE
#  include <linux/module.h>
#endif

#include <linux/version.h>
#include <linux/malloc.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/hdreg.h>
#include <linux/stat.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/blkdev.h>
#include <linux/genhd.h>
#include <asm/ioctl.h>
#include <asm/segment.h>

#if ( OS_Major > 1) && ( OS_Minor > 0)
#  include <asm/uaccess.h>
#  define	memcpy_fromfs		copy_from_user
#  define	memcpy_tofs		copy_to_user
#  define	proc_register_dynamic	proc_register
#  define	verify_area		!access_ok
#endif

#ifdef CONFIG_KERNELD
#include <linux/kerneld.h>
#endif
#include <linux/errno.h>

#include <lvm.h>
#include <lvm_kernel.h>

#ifdef MODULE
#   define MODULE_NAME	"lvm"
#endif


/* Volume group structure pointers */
static vg_t *vg[ABS_MAX_VG];
static pv_t *pvp = NULL;
static lv_t *lvp = NULL;
static pe_t *pep = NULL;

/* Request structures (lvm_chr_ioctl) */
static pv_change_req_t pv_change_req;
static pe_lock_req_t   pe_lock_req;
static le_remap_req_t  le_remap_req;

static char pv_name[NAME_LEN];
static char lv_name[NAME_LEN];
#if defined LVM_CONFIG_PROC_FS && defined CONFIG_PROC_FS
static char dummy_buf[160]; /* sized for 2 lines in lvm_proc_get_info */
#endif

static uint lv_open = 0;

static const char* const lvm_name = "lvm";
static int lock = 0;
static int loadtime = 0;
static uint vg_count = 0;
static long lvm_chr_open_count = 0;
static ushort lvm_iop_version = LVM_IOP_VERSION;
struct wait_queue *lvm_wait = NULL;
struct wait_queue *lvm_map_wait = NULL;

#if defined LVM_CONFIG_PROC_FS && defined CONFIG_PROC_FS
static struct proc_dir_entry lvm_proc_entry = {
   0, 3, "lvm", S_IFREG | S_IRUGO,
   1, 0, 0, 0,
   NULL,
   lvm_proc_get_info,
   NULL, NULL, NULL, NULL, NULL,
};
#endif

static struct file_operations lvm_chr_fops = {
   NULL,                /* No lseek              */
   NULL,		/* No read               */
   NULL,		/* No write              */
   NULL,		/* No readdir            */
   NULL,		/* No select             */
   lvm_chr_ioctl,
   NULL,		/* No mmap               */
   lvm_chr_open,
   lvm_chr_close,
   NULL,		/* No fsync              */
   NULL,		/* No fasync             */
   NULL,		/* No check_media_change */
   NULL		        /* No revalidate         */
};

static struct file_operations lvm_blk_fops = {
   NULL,           /* No lseek				 */
   block_read,     /* generic read          */
   block_write,    /* genric write          */
   NULL,           /* No readdir            */
   NULL,           /* No select             */
   lvm_blk_ioctl,
   NULL,           /* No mmap               */
   lvm_blk_open,
   lvm_blk_close,
   NULL,           /* No fsync              */
   NULL,           /* No fasync             */
   NULL,           /* No check_media_change */
   NULL            /* No revalidate         */
};


/* gendisk structures */
static struct hd_struct lvm_hd_struct[MAX_VG*(MAX_LV+1)];
static int lvm_blocksizes[MAX_VG*(MAX_LV+1)] = { 0, };
static int lvm_size[MAX_VG*(MAX_LV+1)] = { 0, };
static struct gendisk lvm_gendisk = {
   LVM_MAJOR,
   "lvm",
   0,
   1,
   MAX_VG * ( MAX_LV + 1),
   lvm_geninit,
   lvm_hd_struct,
   lvm_size,
   MAX_VG * ( MAX_LV + 1),
   NULL,
   NULL
};


#ifdef MODULE
/*
 * Module initialization...
 */
static int init_module ( void)
#else
/*
 * Driver initialization...
 */
static int lvm_init ( void)
#endif /* #ifdef MODULE */
{
   lvm_init_vars ();

   if ( register_chrdev ( lvm_MajorNumber, lvm_name, &lvm_chr_fops) < 0) {
      printk ( "%s -- register_chrdev failed\n", lvm_name);
      return -EIO;
   }
   if ( register_blkdev ( lvm_MajorNumber, lvm_name, &lvm_blk_fops) < 0) {
      printk ( "%s -- register_blkdev failed\n", lvm_name);
      if ( unregister_chrdev ( lvm_MajorNumber, lvm_name) < 0)
         printk ( "%s -- unregister_chrdev failed\n", lvm_name);
      return -EIO;
   }

#if defined LVM_CONFIG_PROC_FS && defined CONFIG_PROC_FS
    proc_register_dynamic ( &proc_root, &lvm_proc_entry);
#endif

#ifdef DEBUG
   printk ( "%s -- registered\n", lvm_name);
#endif

   blk_dev[LVM_MAJOR].request_fn = DEVICE_REQUEST;
   blk_dev[LVM_MAJOR].current_request = NULL;
   /* read ahead per LV later? */
   read_ahead[LVM_MAJOR] = LVM_MAX_READ_AHEAD;

   /* insert our gendisk */
   lvm_geninit ( &lvm_gendisk);
   lvm_gendisk.next = gendisk_head;
   gendisk_head = &lvm_gendisk;

   /* reference from ll_rw_blk */
   lvm_map_ptr = lvm_map;

   printk ( "%s%s -- ", lvm_version, lvm_name);
#ifdef MODULE
   printk ( "Module");
#else
   printk ( "Driver");
#endif
   printk ( " successfully initialized\n");

   return 0;
}


#ifdef MODULE
/*
 * Module cleanup...
 */
void cleanup_module ( void)
{
   /* DANGER: another one might be in chain beyond us */
   if ( gendisk_head != &lvm_gendisk) {
      printk ( "%s -- ERROR: another one is in chain "
               "beyond me --> can't cleanup\n",
               lvm_name);
      MOD_INC_USE_COUNT;
      return;
   }

   /* delete our gendisk from chain */
   gendisk_head = lvm_gendisk.next;
   blk_size[LVM_MAJOR] = NULL;
   blksize_size[LVM_MAJOR] = NULL;

#if defined LVM_CONFIG_PROC_FS && defined CONFIG_PROC_FS
   proc_unregister ( &proc_root, lvm_proc_entry.low_ino);
#endif

   if ( unregister_chrdev ( lvm_MajorNumber, lvm_name) < 0) {
      printk ( "%s -- unregister_chrdev failed\n", lvm_name);
   }
   if ( unregister_blkdev ( lvm_MajorNumber, lvm_name) < 0) {
      printk ( "%s -- unregister_blkdev failed\n", lvm_name);
   }

   /* reference from ll_rw_blk */
   lvm_map_ptr = NULL;

   blk_dev[LVM_MAJOR].request_fn = NULL;
   blk_dev[LVM_MAJOR].current_request = NULL;

   printk ( "%s -- Module succesfully deactivated\n", lvm_name);

   return;
} /* void cleanup_module () */
#endif /* #ifdef MODULE */


/*
 * support function to initialize lvm variables
 */
static void lvm_init_vars ( void)
{
   int v;

#ifdef MODULE
   /* just to fool the compiler to be quiet */
   if ( 0) init_module ();
   if ( 0) cleanup_module ();
#endif

   loadtime = CURRENT_TIME;

   pe_lock_req.lock = UNLOCK_PE;
   pe_lock_req.data.lv_dev = \
   pe_lock_req.data.pv_dev = \
   pe_lock_req.data.pv_offset = 0;

   /* Initialize VG pointers */
   for ( v = 0; v < MAX_VG; v++) vg[v] = NULL;

   return;
}


/********************************************************************
 *
 * Character device functions
 *
 ********************************************************************/

/*
 * character device open routine
 */
static int lvm_chr_open ( struct inode *inode,
                          struct file *file) {
   int minor = MINOR ( inode->i_rdev);
   ulong flags = 0;
   spinlock_t lvm_lock_dummy, lvm_lock = SPIN_LOCK_UNLOCKED;

#ifdef DEBUG
   printk ( "%s -- lvm_chr_open MINOR: %d  VG#: %d  LV#: %d  lock: %d\n",
            lvm_name, minor, VG(minor), LV(minor), lock);
#endif

   /* Just for gcc */
   lvm_lock_dummy = lvm_lock;

   /* super user validation */
   if ( ! suser()) return -EACCES;

   /* Group special file open */
   if ( VG(minor) > MAX_VG || LV(minor) != 0) return -ENXIO;

   MOD_INC_USE_COUNT;

   spin_lock_irqsave ( &lvm_lock, flags);
   if ( lock == 0) {
       lock = current->pid;
   } else if ( lock != current->pid) {
sleep_on:
       spin_lock_irqsave ( &lvm_lock, flags);
       while ( lock != 0) interruptible_sleep_on ( &lvm_wait);
       spin_lock_irqsave ( &lvm_lock, flags);
       lock = current->pid;
       if ( lock != current->pid) goto sleep_on;
   }
   spin_unlock_irqrestore( &lvm_lock, flags);

   lvm_chr_open_count++;

   return 0;
}


/*
 * character device i/o-control routine
 */
static int lvm_chr_ioctl ( struct inode *inode, struct file *file,
                           uint command, ulong a) {
   int minor = MINOR ( inode->i_rdev);
   int error;
   int extendable;
#if defined(VERIFY_READ) || defined(VERIFY_WRITE)
   int   ret;
#endif
   ulong  l, le, p, v;
   ulong size;
   void  *arg = ( void*) a;
   lv_t  lv;
#ifdef LVM_GET_INODE
   struct inode *inode_sav;
#endif
   lv_status_byname_req_t lv_status_byname_req;
   lv_status_byindex_req_t lv_status_byindex_req;


#ifdef DEBUG_IOCTL
   printk ( "%s -- lvm_chr_ioctl: command: 0x%X  MINOR: %d  "
            "VG#: %d  LV#: %d  mode: 0x%X\n",
            lvm_name, command, minor, VG(minor), LV(minor), file->f_mode);
#endif

   switch ( command) {

      /* check lvm version to ensure driver/application interoperability */
      case LVM_GET_IOP_VERSION:
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( ushort*) arg,
                sizeof ( ushort))) != 0) return ret;
#endif
         memcpy_tofs ( arg, &lvm_iop_version, sizeof ( ushort));
         return 0;

#ifdef LVM_TOTAL_RESET
      /* test reset function */
      case LVM_RESET:
         for ( v = 0; v < ABS_MAX_VG; v++) {
            if ( vg[v] != NULL) {
               for ( l = 0; l < vg[v]->lv_max; l++) {
                  if ( vg[v]->lv[l] != NULL) {
#ifdef DEBUG
                     printk ( "%s -- fsync_dev and invalidate_buffers "
                              "for %s [%s]\n",
                              lvm_name,
                              vg[v]->lv[l]->lv_name,
                              kdevname ( vg[v]->lv[l]->lv_dev));
#endif
                     fsync_dev ( vg[v]->lv[l]->lv_dev);
                     vg[v]->vg_status &= ~VG_ACTIVE;
                     invalidate_buffers ( vg[v]->lv[l]->lv_dev);
                     if ( vg[v]->lv[l]->lv_current_pe != NULL) {
#ifdef DEBUG_KFREE
                        printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
                        kfree ( vg[v]->lv[l]->lv_current_pe);
                     }
#ifdef DEBUG_KFREE
                     printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
                     kfree ( vg[v]->lv[l]);
                  }
               }
               for ( p = 0; p < vg[v]->pv_max; p++)
                  if ( vg[v]->pv[p] != NULL) {
#ifdef DEBUG_KFREE
                     printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
#ifdef LVM_GET_INODE
                     clear_inode ( vg[v]->pv[p]->inode);
#endif
                     kfree ( vg[v]->pv[p]);
                  }
#ifdef DEBUG_KFREE
               printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
               kfree ( vg[v]);
               vg[v] = NULL;
            }
         }
#ifdef MODULE
#if ( OS_Major > 1) && ( OS_Minor > 0)
         (&__this_module)->usecount = lvm_chr_open_count;
#else
         mod_use_count_ = lvm_chr_open_count;
#endif
#endif
         lock = 0; /* release lock */
         wake_up ( &lvm_wait);
         return 0;
#endif /* LVM_TOTAL_RESET */


      /* set global read ahead setting */
      case BLKRASET:
#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_chr_ioctl -- BLKRASET: %d sectors for major %d\n",
                  lvm_name, ( int) arg, MAJOR( inode->i_rdev));
#endif
         if ( ( int) arg < LVM_MIN_READ_AHEAD ||
              ( int) arg > LVM_MAX_READ_AHEAD) return -EINVAL;
/* add code for read ahead array for all LVs here */
         read_ahead[MAJOR( inode->i_rdev)] = ( int) arg;
         return 0;


      /* get current global read ahead setting */
      case BLKRAGET:
#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_chr_ioctl -- BLKRAGET\n", lvm_name);
#endif
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( long *) arg,
                sizeof(long))) != 0) return ret;
#endif
         /* add code for read ahead array here */
#if ( OS_Major > 1) && ( OS_Minor > 0)
         copy_to_user ( ( long*) arg, &read_ahead[MAJOR( inode->i_rdev)],
                        sizeof ( read_ahead[MAJOR( inode->i_rdev)]));
#else
         put_user ( read_ahead[MAJOR( inode->i_rdev)], (long *) arg);
#endif
         return 0;


      /* lock/unlock i/o to a physical extend to move it to another
         physical volume (move's done in user space's pvmove) */
      case PE_LOCK_UNLOCK:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pe_lock_req_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &pe_lock_req, arg, sizeof ( pe_lock_req_t));

         switch ( pe_lock_req.lock) {
            case LOCK_PE:
               for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
                  if ( vg[VG(minor)]->pv[p] != NULL &&
                       pe_lock_req.data.pv_dev == vg[VG(minor)]->pv[p]->pv_dev)
                     break;
               }
      
               if ( p == vg[VG(minor)]->pv_max) return -ENXIO;

               pe_lock_req.lock = UNLOCK_PE;
               fsync_dev ( pe_lock_req.data.lv_dev);
               pe_lock_req.lock = LOCK_PE;
               break;

            case UNLOCK_PE:
               pe_lock_req.lock = UNLOCK_PE;
               pe_lock_req.data.lv_dev = \
               pe_lock_req.data.pv_dev = \
               pe_lock_req.data.pv_offset = 0;
               wake_up ( &lvm_map_wait);
               break;

            default:
               return -EINVAL;
         }

         return 0;


      /* remap a logical extend (after moving the physical extend) */
      case LE_REMAP:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( le_remap_req_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &le_remap_req, arg, sizeof ( le_remap_req_t));
         for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
            if ( vg[VG(minor)]->lv[l] != NULL &&
                 lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name,
                              le_remap_req.lv_name) == 0) {
               for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
                  if ( vg[VG(minor)]->lv[l]->lv_current_pe[le].dev ==
                       le_remap_req.old_dev &&
                       vg[VG(minor)]->lv[l]->lv_current_pe[le].pe ==
                       le_remap_req.old_pe) {
                     vg[VG(minor)]->lv[l]->lv_current_pe[le].dev =
                        le_remap_req.new_dev;
                     vg[VG(minor)]->lv[l]->lv_current_pe[le].pe =
                        le_remap_req.new_pe;
                     return 0;
                  }
               }
               return -EINVAL;
            }
         }

         return -ENXIO;


      /* create a VGDA */
      case VG_CREATE:
         if ( vg[VG(minor)] == NULL)
            vg[VG(minor)] = kmalloc ( sizeof ( vg_t), GFP_KERNEL);
         else return -EPERM;

         if ( vg[VG(minor)] == NULL) {
            printk ( "%s -- VG_CREATE: kmalloc error VG\n", lvm_name);
            return -ENOMEM;
         }

         /* VG */
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( vg_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( vg[VG(minor)], arg, sizeof ( vg_t));
         vg[VG(minor)]->vg_status &= ~VG_ACTIVE;

         error = FALSE;
         if ( vg[VG(minor)]->pv_max > ABS_MAX_PV) {
            printk ( "%s -- Can't activate VG: ABS_MAX_PV to small\n",
                     lvm_name);
            error = TRUE;
         }
         if ( vg[VG(minor)]->lv_max > ABS_MAX_LV) {
            printk ( "%s -- Can't activate VG: ABS_MAX_LV to small for %lu\n",
                     lvm_name, vg[VG(minor)]->lv_max);
            error = TRUE;
         }
         if ( error == TRUE) {
#ifdef DEBUG_KFREE
            printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
            kfree ( vg[VG(minor)]);
            vg[VG(minor)] = NULL;
            return -ENOMEM;
         }

         /* PVs */
         vg[VG(minor)]->pv_act = 0;
         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            /* user space address */
            if ( ( pvp = vg[VG(minor)]->pv[p]) != NULL) {
               vg[VG(minor)]->pv[p] = kmalloc ( sizeof ( pv_t), GFP_KERNEL);
               if ( vg[VG(minor)]->pv[p] == NULL) {
                  printk ( "%s -- VG_CREATE: kmalloc error PV\n", lvm_name);
                  for ( ; p < vg[VG(minor)]->pv_max; p++)
                     vg[VG(minor)]->pv[p] = NULL;
                  do_vg_remove ( minor);
                  return -ENOMEM;
               }
#ifdef VERIFY_READ
               if ( ( ret = verify_area ( VERIFY_READ, pvp, sizeof ( pv_t)))
                    != 0) return ret;
#endif
               memcpy_fromfs ( vg[VG(minor)]->pv[p], pvp, sizeof ( pv_t));
               /* We don't need the PE free list
                  in kernel space as with LVs pe_t list (see below) */
               vg[VG(minor)]->pv[p]->pv_status = PV_ACTIVE;
               vg[VG(minor)]->pv_act++;

#ifdef LVM_GET_INODE
               /* insert a dummy inode for fs_may_mount */
               vg[VG(minor)]->pv[p]->inode =
                  lvm_get_inode ( vg[VG(minor)]->pv[p]->pv_dev);
#endif
            }
         }

         /* LVs */
         for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
            /* user space address */
            if ( ( lvp = vg[VG(minor)]->lv[l]) != NULL) {
               vg[VG(minor)]->lv[l] = kmalloc ( sizeof ( lv_t), GFP_KERNEL);
               if ( vg[VG(minor)]->lv[l] == NULL) {
                  printk ( "%s -- VG_CREATE: kmalloc error LV\n", lvm_name);
                  do_vg_remove ( minor);
                  return -ENOMEM;
               }
#ifdef VERIFY_READ
               if ( ( ret = verify_area ( VERIFY_READ, lvp, sizeof ( lv_t)))
                    != 0) return ret;
#endif
               memcpy_fromfs ( vg[VG(minor)]->lv[l], lvp, sizeof ( lv_t));

               lvm_gendisk.part[minor+l+1].start_sect = 0;
               lvm_gendisk.part[minor+l+1].nr_sects =
                  vg[VG(minor)]->lv[l]->lv_size;
               lvm_size[minor+l+1] = vg[VG(minor)]->lv[l]->lv_size >> 1;
#ifdef DEBUG_VG_CREATE
               printk ( "%s -- VG_CREATE: lvm_size[%d]: %d\n",
                        lvm_name,
                        minor+l+1,
                        lvm_size[minor+l+1]);
#endif

               /* Get the lv_current_pe list for device mapping */
               pep = vg[VG(minor)]->lv[l]->lv_current_pe; /* user address */
               if ( pep != NULL) {
                  size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
                  if ( ( vg[VG(minor)]->lv[l]->lv_current_pe =
                            kmalloc ( size, GFP_KERNEL)) == NULL) {
                     printk ( "%s -- VG_CREATE: kmalloc error LV_CURRENT_PE\n",
                              lvm_name);
                     for ( ; l < ABS_MAX_LV + 1; l++)
                        vg[VG(minor)]->lv[l] = NULL;
                     do_vg_remove ( minor);
                     return -ENOMEM;
                  }
#ifdef VERIFY_READ
                  if ( ( ret = verify_area ( VERIFY_READ, pep, size))
                       != 0) return ret;
#endif
                  memcpy_fromfs ( vg[VG(minor)]->lv[l]->lv_current_pe,
                                  pep, size);
               }
            }
         }

         vg[VG(minor)]->vg_status |= VG_ACTIVE;
         vg_count++;

         MOD_INC_USE_COUNT;
         return 0;


      /* remove an inactive VGDA */
      case VG_REMOVE:
         return do_vg_remove ( minor);


      /* extend a volume group */
      case VG_EXTEND:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
         if ( vg[VG(minor)]->pv_cur < vg[VG(minor)]->pv_max) {
            for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
               if ( vg[VG(minor)]->pv[p] == NULL) {
                  vg[VG(minor)]->pv[p] = kmalloc ( sizeof ( pv_t), GFP_KERNEL);
                  if ( vg[VG(minor)]->pv[p] == NULL) {
                     printk ( "%s -- VG_EXTEND: kmalloc error PV\n", lvm_name);
                     return -ENOMEM;
                  }
#ifdef VERIFY_READ
                  if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_t)))
                       != 0) return ret;
#endif
                  memcpy_fromfs ( vg[VG(minor)]->pv[p], arg, sizeof ( pv_t));
                  vg[VG(minor)]->pv[p]->pv_status = PV_ACTIVE;
                  vg[VG(minor)]->pv_cur++;
                  vg[VG(minor)]->pv_act++;
                  vg[VG(minor)]->pe_total += vg[VG(minor)]->pv[p]->pe_total;
#ifdef LVM_GET_INODE
                  /* insert a dummy inode for fs_may_mount */
                  vg[VG(minor)]->pv[p]->inode =
                     lvm_get_inode ( vg[VG(minor)]->pv[p]->pv_dev);
#endif
                  return 0;
               }
            }
         }
         return -EPERM;


      /* reduce a volume group */
      case VG_REDUCE:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_name)))
               != 0) return ret;
#endif
         memcpy_fromfs ( pv_name, arg, sizeof ( pv_name));

         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            if ( vg[VG(minor)]->pv[p] != NULL && 
                 lvm_strcmp ( vg[VG(minor)]->pv[p]->pv_name,pv_name) == 0) {
               if ( vg[VG(minor)]->pv[p]->lv_cur > 0) return -EPERM;
               vg[VG(minor)]->pe_total -= vg[VG(minor)]->pv[p]->pe_total;
               vg[VG(minor)]->pv_cur--;
               vg[VG(minor)]->pv_act--;
#ifdef DEBUG_KFREE
               printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
#ifdef LVM_GET_INODE
               clear_inode ( vg[VG(minor)]->pv[p]->inode);
#endif
               kfree ( vg[VG(minor)]->pv[p]);
               vg[VG(minor)]->pv[p] = NULL;
               return 0;
            }
         }
         return -ENXIO;


      /* set/clear extendability flag of volume group */
      case VG_SET_EXTENDABLE:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_READ, arg,
                                    sizeof ( extendable))) != 0)
            return ret;
#endif
         memcpy_fromfs ( &extendable, arg, sizeof ( extendable));
         if ( extendable == VG_EXTENDABLE ||
              extendable == ~VG_EXTENDABLE) {
            if ( extendable == VG_EXTENDABLE) {
               vg[VG(minor)]->vg_status |= VG_EXTENDABLE;
            } else {
               vg[VG(minor)]->vg_status &= ~VG_EXTENDABLE;
            }
         } else return -EINVAL;
         return 0;


      /* get volume group data (only the vg_t struct) */
      case VG_STATUS:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, arg, sizeof ( vg_t))) != 0)
            return ret;
#endif
         memcpy_tofs ( arg, vg[VG(minor)], sizeof ( vg_t));
         return 0;


      /* get volume group count */
      case VG_STATUS_GET_COUNT:
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, arg,
                                    sizeof ( vg_count))) != 0)
            return ret;
#endif
         memcpy_tofs ( arg, &vg_count, sizeof ( vg_count));
         return 0;


      /* get volume group count */
      case VG_STATUS_GET_NAMELIST:
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, arg,
                                    vg_count * NAME_LEN)) != 0)
            return ret;
#endif
         l = 0;
         for ( v = 0; v < ABS_MAX_VG; v++) 
            if ( vg[v] != NULL)
               memcpy_tofs ( arg + l++ * NAME_LEN, vg[v]->vg_name, NAME_LEN);
         return 0;


      /* create, remove, extend or reduce a logical volume */
      case LV_CREATE:
      case LV_REMOVE:
      case LV_EXTEND:
      case LV_REDUCE:
      	 if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( lv_name)))
              != 0) return ret;
#endif
         memcpy_fromfs ( lv_name, arg, sizeof ( lv_name));
         arg += sizeof ( lv_name);

         if ( command != LV_REMOVE) {
#ifdef VERIFY_READ
            if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( lv_t*)))
                 != 0) return ret;
#endif
            memcpy_fromfs ( &lvp, arg, sizeof ( lv_t*));
#ifdef VERIFY_READ
            if ( ( ret = verify_area ( VERIFY_READ, lvp, sizeof ( lv_t)))
                 != 0) return ret;
#endif
            memcpy_fromfs ( &lv, lvp, sizeof ( lv_t));
         }

         switch ( command) {
            case LV_CREATE:
               return do_lv_create ( minor, lv_name, &lv, inode);

            case LV_REMOVE:
               return do_lv_remove ( minor, lv_name);

            case LV_EXTEND:
            case LV_REDUCE:
               return do_lv_extend_reduce ( minor, lv_name, &lv, inode);
         }


      /* get status of a logical volume by name */
      case LV_STATUS_BYNAME:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg,
                                    sizeof ( lv_status_byname_req))) != 0)
            return ret;
#endif
         memcpy_fromfs ( &lv_status_byname_req, arg,
                         sizeof ( lv_status_byname_req_t));

         if ( lv_status_byname_req.lv == NULL) return -EINVAL;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, lv_status_byname_req.lv,
                                    sizeof ( lv_t))) != 0) return ret;
#endif
         memcpy_fromfs ( &lv, lv_status_byname_req.lv, sizeof ( lv_t));
         l = 0;
         while ( l < vg[VG(minor)]->lv_max) {
            if ( vg[VG(minor)]->lv[l] != NULL &&
                 lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name,
                              lv_status_byname_req.lv_name) == 0) {
#ifdef VERIFY_WRITE
               if ( ( ret = verify_area ( VERIFY_WRITE, lv_status_byname_req.lv,
                                          sizeof ( lv_t))) != 0) return ret;
#endif
               memcpy_tofs ( lv_status_byname_req.lv,
                             vg[VG(minor)]->lv[l], sizeof ( lv_t));
               if ( lv.lv_current_pe != NULL) {
                  size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
#ifdef VERIFY_WRITE
                  if ( ( ret = verify_area ( VERIFY_WRITE,
                                             lv.lv_current_pe,
                                             size)) != 0) return ret;
#endif
                  memcpy_tofs ( lv.lv_current_pe,
                                vg[VG(minor)]->lv[l]->lv_current_pe,
                                size);
               }
               return 0;
            }
            l++;
         }
         return -ENXIO;


      /* get status of a logical volume by index */
      case LV_STATUS_BYINDEX:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg,
                                    sizeof ( lv_status_byindex_req_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &lv_status_byindex_req, arg,
                         sizeof ( lv_status_byindex_req));

         l = lv_status_byindex_req.lv_index;
         lvp = lv_status_byindex_req.lv;
         if ( lvp == NULL) return -EINVAL;

#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, lvp, sizeof ( lv_t)))
              != 0) return ret;
#endif
         if ( vg[VG(minor)]->lv[l] == NULL) return -ENXIO;
         memcpy_tofs ( lvp, vg[VG(minor)]->lv[l], sizeof ( lv_t));
         return 0;


      /* change a physical volume */
      case PV_CHANGE:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_change_req)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &pv_change_req, arg, sizeof ( pv_change_req));

         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            if ( vg[VG(minor)]->pv[p] != NULL &&
                 lvm_strcmp ( vg[VG(minor)]->pv[p]->pv_name,
                              pv_change_req.pv_name) == 0) {
#ifdef VERIFY_READ
               if ( ( ret = verify_area ( VERIFY_READ, pv_change_req.pv,
                                          sizeof ( pv_t))) != 0) return ret;
#endif
#ifdef LVM_GET_INODE
               inode_sav = vg[VG(minor)]->pv[p]->inode;
#endif
               memcpy_fromfs ( vg[VG(minor)]->pv[p], pv_change_req.pv,
                               sizeof ( pv_t));
#ifdef LVM_GET_INODE
               vg[VG(minor)]->pv[p]->inode = inode_sav;
#endif
               return 0;
            }
         }
         return -ENXIO;


      /* get physical volume data (only the pv_t struct) */
      case PV_STATUS:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_name)))
              != 0) return ret;
#endif
         memcpy_fromfs ( pv_name, arg, sizeof ( pv_name));
         arg += sizeof ( pv_name);
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_t*)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &pvp, arg, sizeof ( pv_t *));
         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            if ( vg[VG(minor)]->pv[p] != NULL) {
               if ( lvm_strcmp ( vg[VG(minor)]->pv[p]->pv_name, pv_name) == 0) {
#ifdef VERIFY_WRITE
                  if ( ( ret = verify_area ( VERIFY_WRITE, pvp,
                         sizeof ( pv_t))) != 0) return ret;
#endif
                  memcpy_tofs ( pvp, vg[VG(minor)]->pv[p], sizeof ( pv_t));
                  return 0;
               }
            }
         }
         return -ENXIO;


      /* search for a physical volume to flush;
         don't care about invalid volumes --> return 0 */
      case PV_FLUSH:
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_name)))
              != 0) return ret;
#endif
         memcpy_fromfs ( pv_name, arg, sizeof ( pv_name));

         for ( v = 0; v < ABS_MAX_VG; v++) {
            if ( vg[v] == NULL) continue;
            for ( p = 0; p < vg[v]->pv_max; p++) {
               if ( vg[v]->pv[p] != NULL &&
                    lvm_strcmp ( vg[v]->pv[p]->pv_name, pv_name) == 0) {
                  fsync_dev ( vg[v]->pv[p]->pv_dev);
                  invalidate_buffers ( vg[v]->pv[p]->pv_dev);
                  return 0;
               }
            }
         }
         return 0;


      default:
         printk ( "%s -- lvm_chr_ioctl: unknown command %d\n",
                  lvm_name, command);
         return -EINVAL;
   }
}


/*
 * character device close routine
 */
#if ( OS_Major > 1) && ( OS_Minor > 0)
static int lvm_chr_close ( struct inode *inode, struct file *file)
#else
static void lvm_chr_close ( struct inode *inode, struct file *file)
#endif
{
#ifdef DEBUG
   int minor = MINOR ( inode->i_rdev);

   printk ( "%s -- lvm_chr_close MINOR: %d  VG#: %d  LV#: %d\n",
            lvm_name, minor, VG(minor), LV(minor));
#endif

   MOD_DEC_USE_COUNT;

   if ( lock == current->pid) lvm_chr_open_count--;
   if ( lvm_chr_open_count == 0) {
      lock = 0; /* release lock */
      wake_up ( &lvm_wait);
   }

#if ( OS_Major > 1) && ( OS_Minor > 0)
   return 0;
#else
   return;
#endif
}


/********************************************************************
 *
 * Block device functions
 *
 ********************************************************************/

/*
 * block device open routine
 */
static int lvm_blk_open ( struct inode *inode, struct file *file) {
   int minor = MINOR ( inode->i_rdev);

#ifdef DEBUG_LVM_BLK_OPEN
   printk ( "%s -- lvm_blk_open MINOR: %d  VG#: %d  LV#: %d  mode: 0x%X\n",
            lvm_name, minor, VG(minor), LV(minor), file->f_mode);
#endif

   if ( vg[VG(minor)] != NULL &&
        ( vg[VG(minor)]->vg_status & VG_ACTIVE) &&
        vg[VG(minor)]->lv[LV(minor)-1] != NULL &&
        LV(minor) > 0 &&
        LV(minor) <= vg[VG(minor)]->lv_max) {

      /* Check parallel LV spindown (LV remove) */
      if ( vg[VG(minor)]->lv[LV(minor)-1]->lv_status & LV_SPINDOWN)
         return -EPERM;

      /* Check inactive LV and open for read/write */
      if ( file->f_mode & O_RDWR) {
         if ( ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_status & LV_ACTIVE))
            return -EPERM;
         if ( ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_access & LV_WRITE))
            return -EACCES;
      }

      vg[VG(minor)]->lv_open++;
      vg[VG(minor)]->lv[LV(minor)-1]->lv_open++;

      MOD_INC_USE_COUNT;

#ifdef DEBUG_LVM_BLK_OPEN
      printk ( "%s -- lvm_blk_open MINOR: %d  VG#: %d  LV#: %d  size: %d\n",
               lvm_name, minor, VG(minor), LV(minor),
               vg[VG(minor)]->lv[LV(minor)-1]->lv_size);
#endif

      return 0;
   }

   return -ENXIO;
}


/*
 * block device i/o-control routine
 */
static int lvm_blk_ioctl (struct inode *inode, struct file *file,
                          uint command, ulong a) {
   int minor = MINOR ( inode->i_rdev);
#ifdef VERIFY_WRITE
   int ret;
#endif
   void *arg = ( void*) a;
   struct hd_geometry *hd = ( struct hd_geometry *) a;

#ifdef DEBUG_IOCTL
   printk ( "%s -- lvm_blk_ioctl MINOR: %d  command: 0x%X  arg: %X  "
            "VG#: %d  LV#: %d\n",
            lvm_name, minor, command, ( uint) arg, VG(minor), LV(minor));
#endif

   switch ( command) {
      /* return device size */
      case BLKGETSIZE:
#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_blk_ioctl -- BLKGETSIZE: %u\n",
                  lvm_name, vg[VG(minor)]->lv[LV(minor)-1]->lv_size);
#endif
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( long *) arg,
                sizeof ( long))) != 0) return ret;
#endif
#if ( OS_Major > 1) && ( OS_Minor > 0)
         copy_to_user ( ( long*) arg, &vg[VG(minor)]->lv[LV(minor)-1]->lv_size,
                        sizeof ( vg[VG(minor)]->lv[LV(minor)-1]->lv_size));
#else
         put_user ( vg[VG(minor)]->lv[LV(minor)-1]->lv_size, ( long *) arg);
#endif
         break;


      /* flush buffer cache */
      case BLKFLSBUF:
#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_blk_ioctl -- BLKFLSBUF\n", lvm_name);
#endif
         fsync_dev ( inode->i_rdev);
         invalidate_buffers ( inode->i_rdev);
         break;


      /* set read ahead for block device */
      case BLKRASET:
         /* super user validation */
         if ( ! suser ()) return -EACCES;

#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_chr_ioctl -- BLKRASET: %d sectors for major %d\n",
                  lvm_name, ( int) arg, MAJOR( inode->i_rdev));
#endif
         if ( ( int) arg < LVM_MIN_READ_AHEAD ||
              ( int) arg > LVM_MAX_READ_AHEAD) return -EINVAL;
         /* add code for single LV read ahead sector handling here */
         read_ahead[MAJOR( inode->i_rdev)] = ( int) arg;
         break;


      /* get current read ahead setting */
      case BLKRAGET:
#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_blk_ioctl -- BLKRAGET\n", lvm_name);
#endif
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( long *) arg,
                sizeof(long))) != 0) return ret;
#endif
#if ( OS_Major > 1) && ( OS_Minor > 0)
         copy_to_user ( ( long*) arg, &read_ahead[MAJOR( inode->i_rdev)],
                        sizeof ( read_ahead[MAJOR( inode->i_rdev)]));
#else
         put_user ( read_ahead[MAJOR( inode->i_rdev)], (long *) arg);
#endif
         break;


      /* get disk geometry */
      case HDIO_GETGEO:
         /* I have the same problem as MD here :-)
            I currently pretend to have a 2 heads, 4 sectors disk. */
#ifdef DEBUG_IOCTL
         printk ( "%s -- lvm_blk_ioctl -- HDIO_GETGEO\n", lvm_name);
#endif
         if ( hd == NULL) return -EINVAL;
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, hd,
                                    sizeof ( struct hd_geometry))) != 0)
            return ret;
#endif
#if ( OS_Major > 1) && ( OS_Minor > 0)
	{
           int heads = 4; short sectors = 2; long start = 0;
           short cylinders = vg[VG(minor)]->lv[LV(minor)]->lv_size / 8;

           copy_to_user ( ( char*) &hd->heads, &heads, sizeof ( heads));
           copy_to_user ( ( char*) &hd->sectors, &sectors, sizeof ( sectors));
           copy_to_user ( ( short*) &hd->cylinders,
                          &cylinders, sizeof ( cylinders));
           copy_to_user ( ( long*) &hd->start, &start, sizeof ( start));
        }
#else
         put_user ( 4, ( char*) &hd->heads);
         put_user ( 2, ( char*) &hd->sectors);
         put_user ( vg[VG(minor)]->lv[LV(minor)]->lv_size / 8,
                    ( short*) &hd->cylinders);
         put_user ( 0L, ( long *) &hd->start);
#endif

#ifdef DEBUG_IOCTL
   printk ( "%s -- lvm_blk_ioctl -- cylinders: %d\n",
            lvm_name, vg[VG(minor)]->lv[LV(minor)]->lv_size / 8);
#endif
         break;


      /* set access flags of a logical volume */
      case LV_SET_ACCESS:
         /* super user validation */
         if ( ! suser ()) return -EACCES;
         vg[VG(minor)]->lv[LV(minor)-1]->lv_access = ( uint) arg;
         break;


      /* set status flags of a logical volume */
      case LV_SET_STATUS:
         /* super user validation */
         if ( ! suser ()) return -EACCES;
         if ( ! ( ( uint) arg & LV_ACTIVE) &&
              vg[VG(minor)]->lv[LV(minor)-1]->lv_open > 1) return -EPERM;
         vg[VG(minor)]->lv[LV(minor)-1]->lv_status = ( uint) arg;
         break;


      /* set allocation flags of a logical volume */
      case LV_SET_ALLOCATION:
         /* super user validation */
         if ( ! suser ()) return -EACCES;
         vg[VG(minor)]->lv[LV(minor)-1]->lv_allocation = ( uint) arg;
         break;


      default:
         printk ( "%s -- lvm_blk_ioctl: unknown command %d\n",
                  lvm_name, command);
         return -EINVAL;
   }

   return 0;
}


/*
 * block device close routine
 */
#if ( OS_Major > 1) && ( OS_Minor > 0)
static int lvm_blk_close ( struct inode *inode, struct file *file)
#else
static void lvm_blk_close ( struct inode *inode, struct file *file)
#endif
{
   int minor = MINOR ( inode->i_rdev);

#ifdef DEBUG
   printk ( "%s -- lvm_blk_close MINOR: %d  VG#: %d  LV#: %d\n",
            lvm_name, minor, VG(minor), LV(minor));
#endif

   sync_dev ( inode->i_rdev);
   vg[VG(minor)]->lv_open--;
   vg[VG(minor)]->lv[LV(minor)-1]->lv_open--;

   MOD_DEC_USE_COUNT;

#if ( OS_Major > 1) && ( OS_Minor > 0)
   return 0;
#else
   return;
#endif
} /* static int/void lvm_blk_close () */


#if defined LVM_CONFIG_PROC_FS && defined CONFIG_PROC_FS
/*
 * Support function /proc-Filesystem
 */
#define	LVM_PROC_BUF	( i == 0 ? dummy_buf : &buf[sz])

static int lvm_proc_get_info ( char *page, char **start, off_t pos,
                               int count, int whence) {
   int c, i, l, p, v, vg_counter, pv_counter, lv_counter,
       lv_counter_tmp, pe_t_bytes, seconds;
   static off_t sz;
   off_t sz_last;
   char allocation_flag, inactive_flag, rw_flag, stripes_flag;
   char *lv_name = NULL;
   static char *buf = NULL;

#ifdef DEBUG_LVM_PROC_GET_INFO
   printk ( "%s - lvm_proc_get_info CALLED  pos: %lu  count: %d  whence: %d\n",
            lvm_name, pos, count, whence);
#endif

   if ( pos == 0 || buf == NULL) {
      sz_last = vg_counter = pv_counter = lv_counter = pe_t_bytes = 0;
   
      /* search for activity */
      for ( v = 0; v < ABS_MAX_VG; v++) {
         if ( vg[v] != NULL) {
            vg_counter++;
            pv_counter += vg[v]->pv_cur;
            lv_counter += vg[v]->lv_cur;
            for ( l = 0; l < vg[v]->lv_max; l++)
               if ( vg[v]->lv[l] != NULL)
                  pe_t_bytes += vg[v]->lv[l]->lv_allocated_le * sizeof ( pe_t);
         }
      }
   
      if ( buf != NULL) {
#ifdef DEBUG_KFREE
         printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
         kfree ( buf);
         buf = NULL;
      }

      /* 2 times: first to get size, 2nd to fill the kmalloced buffer */
      for ( i = 0; i < 2; i++) {
         sz = 0;
         sz += sprintf ( LVM_PROC_BUF, "%d VG", vg_counter);
         if ( vg_counter > 1) sz += sprintf ( LVM_PROC_BUF, "s");
         sz += sprintf ( LVM_PROC_BUF, " %d PV", pv_counter);
         if ( pv_counter > 1) sz += sprintf ( LVM_PROC_BUF, "s");
         sz += sprintf ( LVM_PROC_BUF, " and %d LV", lv_counter);
         if ( lv_counter > 1) sz += sprintf ( LVM_PROC_BUF, "s");
         sz += sprintf ( LVM_PROC_BUF, "  %d read_ahead  %d malloced  ",
                         read_ahead[LVM_MAJOR],
                         vg_counter * sizeof ( vg_t) +
                         pv_counter * sizeof ( pv_t) +
                         lv_counter * sizeof ( lv_t) +
                         pe_t_bytes + sz_last);
         seconds = CURRENT_TIME - loadtime;
         sz += sprintf ( LVM_PROC_BUF, "%2d:%02d:%02d active\n",
                                       seconds / 3600,
                                       ( seconds % 3600)/ 60,
                                       seconds % 60);

         if ( vg_counter > 0) {
            for ( v = 0; v < ABS_MAX_VG; v++) {
               /* volume group */
               if ( vg[v] != NULL) {
                  inactive_flag = ' ';
                  if ( ! ( vg[v]->vg_status & VG_ACTIVE))
                     inactive_flag = 'I';
                  sz += sprintf ( LVM_PROC_BUF,
                                  "\nVG: %c%s  [%lu PV, %lu LV]  PE Size: "
                                  "%lu KB\n"
                                  "  Usage [KB/PE]: %lu /%lu total  "
                                  "%lu /%lu used  %lu /%lu free",
                                  inactive_flag,
                                  vg[v]->vg_name,
                                  vg[v]->pv_cur,
                                  vg[v]->lv_cur,
                                  vg[v]->pe_size >> 1,
                                  vg[v]->pe_size * vg[v]->pe_total >> 1,
                                  vg[v]->pe_total,
                                  vg[v]->pe_allocated * vg[v]->pe_size >> 1,
                                  vg[v]->pe_allocated,
                                  ( vg[v]->pe_total - vg[v]->pe_allocated) *
                                  vg[v]->pe_size >> 1,
                                  vg[v]->pe_total - vg[v]->pe_allocated);

                  /* physical volumes */
                  sz += sprintf ( LVM_PROC_BUF, "\n  PVs: ");
                  c = 0;
                  for ( p = 0; p < vg[v]->pv_max; p++) {
                     if ( vg[v]->pv[p] != NULL) {
                        inactive_flag = 'A';
                        if ( ! ( vg[v]->pv[p]->pv_status & PV_ACTIVE))
                           inactive_flag = 'I';
                        allocation_flag = 'A';
                        if ( ! ( vg[v]->pv[p]->pv_allocatable & PV_ALLOCATABLE))
                           allocation_flag = 'N';
                        sz += sprintf ( LVM_PROC_BUF,
                                        "[%c%c] %-11s %8lu /%-6lu  "
                                        "%8lu /%-6lu  %8lu /%-6lu",
                                        inactive_flag,
                                        allocation_flag,
                                        vg[v]->pv[p]->pv_name,
                                        vg[v]->pv[p]->pe_total *
                                        vg[v]->pv[p]->pe_size >> 1,
                                        vg[v]->pv[p]->pe_total,
                                        vg[v]->pv[p]->pe_allocated *
                                        vg[v]->pv[p]->pe_size >> 1,
                                        vg[v]->pv[p]->pe_allocated,
                                        ( vg[v]->pv[p]->pe_total -
                                          vg[v]->pv[p]->pe_allocated) *
                                        vg[v]->pv[p]->pe_size >> 1,
                                        vg[v]->pv[p]->pe_total -
                                        vg[v]->pv[p]->pe_allocated);
                        c++;
                        if ( c < vg[v]->pv_cur)
                           sz += sprintf ( LVM_PROC_BUF,
                                           "\n       ");
                     }
                  }

                  /* logical volumes */
                  sz += sprintf ( LVM_PROC_BUF,
                                  "\n    LVs: ");
                  c = lv_counter_tmp = 0;
                  for ( l = 0; l < vg[v]->lv_max; l++) {
                     if ( vg[v]->lv[l] != NULL) {
                        lv_counter_tmp++;
                        inactive_flag = 'A';
                        if ( ! ( vg[v]->lv[l]->lv_status & LV_ACTIVE))
                           inactive_flag = 'I';
                        rw_flag = 'R';
                        if ( vg[v]->lv[l]->lv_access & LV_WRITE) rw_flag = 'W';
                        allocation_flag = 'D';
                        if ( vg[v]->lv[l]->lv_allocation & LV_CONTIGUOUS)
                           allocation_flag = 'C';
                        stripes_flag = 'L';
                        if ( vg[v]->lv[l]->lv_stripes > 1) stripes_flag = 'S';
                        sz += sprintf ( LVM_PROC_BUF,
                                        "[%c%c%c%c",
                                        inactive_flag,
                                        rw_flag,
                                        allocation_flag,
                                        stripes_flag);
                        if ( vg[v]->lv[l]->lv_stripes > 1)
                           sz += sprintf ( LVM_PROC_BUF, "%-2lu",
                                           vg[v]->lv[l]->lv_stripes);
                        else
                           sz += sprintf ( LVM_PROC_BUF, "  ");
                        lv_name = lvm_strrchr ( vg[v]->lv[l]->lv_name, '/');
                        if ( lv_name != NULL) lv_name++;
                        else lv_name = vg[v]->lv[l]->lv_name;
                        sz += sprintf ( LVM_PROC_BUF, "] %-25s %9lu /%-6lu   ",
                                        lv_name,
                                        vg[v]->lv[l]->lv_size >> 1,
                                        vg[v]->lv[l]->lv_size / vg[v]->pe_size);
                        if ( vg[v]->lv[l]->lv_open == 0)
                           sz += sprintf ( LVM_PROC_BUF, "close");
                        else
                           sz += sprintf ( LVM_PROC_BUF, "%lux open",
                                           vg[v]->lv[l]->lv_open);
                        c++;
                        if ( c < vg[v]->lv_cur)
                           sz += sprintf ( LVM_PROC_BUF,
                                           "\n         ");
                     }
                  }
                  if ( lv_counter_tmp == 0)
                     sz += sprintf ( LVM_PROC_BUF, "none");
                  sz += sprintf ( LVM_PROC_BUF, "\n");
               }
            }
         }

         if ( buf == NULL) {
            if ( ( buf = kmalloc ( sz + 64, GFP_KERNEL)) == NULL) {
               sz = 0;
               return sprintf ( page, "%s - kmalloc error at line %d\n",
                                      lvm_name, __LINE__);
            }
         }
         sz_last = sz;
      }
   }

   if ( pos > sz - 1) {
      kfree ( buf);
      buf = NULL;
      return 0;
   }

   *start = &buf[pos];
   if ( sz - pos < count) return sz - pos;
   else                   return count;
}
#endif /* #if defined LVM_CONFIG_PROC_FS && defined CONFIG_PROC_FS*/ 


/*
 * Support function for /usr/src/linux/drivers/block/ll_rw_blk.c
 * (see init_module/lvm_init)
 */
static int lvm_map ( int minor, kdev_t *rdev,
                     ulong *rsector, ulong size, int rw) {
   ulong index = 0;
   ulong index_offset = 0;
   ulong rsector_sav = 0;
   ulong stripe_index = 0;
   ulong stripe_length = 0;
   kdev_t rdev_sav = 0;

   if ( ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_status & LV_ACTIVE)) {
      printk ( "%s - lvm_map: ll_rw_blk for inactive LV %s\n",
               lvm_name, vg[VG(minor)]->lv[LV(minor)-1]->lv_name);
      return -1;
   }

   if ( rw == WRITE &&
        ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_access & LV_WRITE)) {
      printk ( "%s - lvm_map: ll_rw_blk write for readonly LV %s\n",
               lvm_name, vg[VG(minor)]->lv[LV(minor)-1]->lv_name);
      return -1;
   }

#ifdef DEBUG_MAP_SIZE
   if ( size != 2)
      printk ( "%s - lvm_map minor:%d size:%lu ", lvm_name, minor, size);
#endif

#ifdef DEBUG_MAP
   printk ( "%s - lvm_map minor:%d  *rdev: %02d:%02d  *rsector: %lu  "
            "size:%lu\n",
            lvm_name, minor,
            MAJOR ( *rdev),
            MINOR ( *rdev),
            *rsector, size);
#endif

   rsector_sav = *rsector;
   rdev_sav    = *rdev;

   if ( *rsector + size > vg[VG(minor)]->lv[LV(minor)-1]->lv_size) {
      printk ( "%s - lvm_map *rsector: %lu + size: %lu too large for"
               " %02d:%02d\n", lvm_name, *rsector, size, VG(minor), LV(minor));
      return -1;
   }


lvm_second_remap:

   /* linear mapping */
   if ( vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes < 2) {
      index = *rsector / vg[VG(minor)]->pe_size; /* get the index */
      *rsector = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe +
                 ( *rsector % vg[VG(minor)]->pe_size);
      *rdev = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].dev;

#ifdef DEBUG_MAP
      printk ( "lv_current_pe[%ld].pe: %ld  rdev: %02d:%02d  rsector:%ld\n",
               index,
               vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe,
               MAJOR ( *rdev),
               MINOR ( *rdev),
               *rsector);
#endif

   /* striped mapping */
   } else {
      stripe_length = vg[VG(minor)]->pe_size *
                      vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes;
      index_offset = vg[VG(minor)]->lv[LV(minor)-1]->lv_allocated_le /
                     vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes;
      stripe_index = *rsector % stripe_length /
                     vg[VG(minor)]->lv[LV(minor)-1]->lv_stripesize;
      index = *rsector / stripe_length +
              ( stripe_index % vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes) *
              index_offset;
      *rsector = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe +
                 ( *rsector % stripe_length) -
                 stripe_index % vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes *
                 vg[VG(minor)]->lv[LV(minor)-1]->lv_stripesize -
                 stripe_index / vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes *
                 ( vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes - 1) *
                 vg[VG(minor)]->lv[LV(minor)-1]->lv_stripesize;
      *rdev = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].dev;
   }

   if ( MAJOR ( *rdev) != SCSI_DISK_MAJOR &&
        MAJOR ( *rdev) != IDE0_MAJOR &&
        MAJOR ( *rdev) != IDE1_MAJOR &&
        MAJOR ( *rdev) != IDE2_MAJOR &&
        MAJOR ( *rdev) != IDE3_MAJOR &&
#if ( OS_Major > 1) && ( OS_Minor > 0) && ( OS_SubMinor > 97)
        MAJOR ( *rdev) != IDE4_MAJOR &&
        MAJOR ( *rdev) != IDE5_MAJOR &&
#endif
        MAJOR ( *rdev) != MD_MAJOR) {
      printk ( "lv_current_pe[%ld].pe ERROR: %ld  rdev: %02d:%02d  "
               "rsector: %ld\nstripe_length: %ld  stripe_index: %ld  "
               "rsector_sav: %ld  index_offset: %ld\n",
               index,
               vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe,
               MAJOR ( *rdev),
               MINOR ( *rdev),
               *rsector,
               stripe_length,
               stripe_index,
               rsector_sav,
               index_offset);
      return -1;
   }

#ifdef DEBUG_MAP
   printk ( "lv_current_pe[%ld].pe: %ld  rdev: %02d:%02d  rsector:%ld\n"
            "stripe_length: %ld  stripe_index: %ld  index_offset: %ld\n",
            index,
            vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe,
            MAJOR ( *rdev),
            MINOR ( *rdev),
            *rsector,
            stripe_length,
            stripe_index,
            index_offset);
#endif

   /* handle physical extends on the move */
   if ( pe_lock_req.lock == LOCK_PE) {
      if ( *rdev == pe_lock_req.data.pv_dev &&
           *rsector >= pe_lock_req.data.pv_offset &&
           *rsector < ( pe_lock_req.data.pv_offset + vg[VG(minor)]->pe_size)) {
         sleep_on ( &lvm_map_wait);
         *rsector = rsector_sav;
         *rdev    = rsector_sav;
         goto lvm_second_remap;
      }
   }

   return 0;
}


/* internal support functions */

/*
 * this one never should be called...
 */
static void do_request (void)
{
  printk ("%s -- oops, got lvm request?\n", lvm_name);
  return;
}


/*
 * support function VGDA remove
 */
static int do_vg_remove ( int minor) {
   int l, p;

   if ( vg[VG(minor)] == NULL)      return -ENXIO;
   if ( vg[VG(minor)]->lv_open > 0) return -EPERM;

   vg[VG(minor)]->vg_status &= ~VG_ACTIVE;

   /* Free PVs */
   for ( p = 0; p < vg[VG(minor)]->pv_max; p++)
      if ( vg[VG(minor)]->pv[p] != NULL) {
#ifdef DEBUG_KFREE
         printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
#ifdef LVM_GET_INODE
         clear_inode ( vg[VG(minor)]->pv[p]->inode);
#endif
         kfree ( vg[VG(minor)]->pv[p]);
      }

   /* Free LVs */
   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] != NULL) {
#ifdef DEBUG
         printk ( "%s -- fsync_dev and "
                  "invalidate_buffers for %s [%s]\n",
                  lvm_name,
                  vg[VG(minor)]->lv[l]->lv_name,
                  kdevname ( vg[VG(minor)]->lv[l]->lv_dev));
#endif
         vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
         fsync_dev ( vg[VG(minor)]->lv[l]->lv_dev);
         vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
         invalidate_buffers ( vg[VG(minor)]->lv[l]->lv_dev);
         lvm_gendisk.part[minor].start_sect = 0;
         lvm_gendisk.part[minor].nr_sects = 0;
         lvm_size[minor] = 0;

         /* Free PEs */
         if ( vg[VG(minor)]->lv[l]->lv_current_pe != NULL) {
#ifdef DEBUG_KFREE
            printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
            kfree ( vg[VG(minor)]->lv[l]->lv_current_pe);
         }
         /* Free LVs */
#ifdef DEBUG_KFREE
         printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
         kfree ( vg[VG(minor)]->lv[l]);
      }
   }

#ifdef DEBUG_KFREE
   printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
   kfree ( vg[VG(minor)]);
   vg[VG(minor)] = NULL;
   vg_count--;

   MOD_DEC_USE_COUNT;
   return 0;
}


/*
 * support function logical volume create
 */
static int do_lv_create ( int minor, char *lv_name, lv_t *lv,
                          struct inode *inode) {
   int l, le, l_new, p, size;
#ifdef VERIFY_READ
   int ret;
#endif
   ulong lv_status_save;

   if ( ( pep = lv->lv_current_pe) == NULL) return -EINVAL;

   l_new = -1;
   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] == NULL) {
         if ( l_new == -1) l_new = l;
      } else if ( lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name, lv_name) == 0)
                return -EPERM;
   }
   if ( l_new == -1) return -EPERM;
   l = l_new;

   vg[VG(minor)]->lv[l] = kmalloc ( sizeof ( lv_t), GFP_KERNEL);
   if ( vg[VG(minor)]->lv[l] == NULL) {
      printk ( "%s -- LV_CREATE: kmalloc error LV\n", lvm_name);
      return -ENOMEM;
   }

   /* copy preloaded LV */
   lvm_memcpy ( ( char*) vg[VG(minor)]->lv[l], ( char *) lv, sizeof ( lv_t));
   lv_status_save = vg[VG(minor)]->lv[l]->lv_status;
   vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;

   /* get the PE structures from user space */
   size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
   if ( ( vg[VG(minor)]->lv[l]->lv_current_pe =
          kmalloc ( size, GFP_KERNEL)) == NULL) {
      printk ( "%s -- LV_CREATE: kmalloc error LV_CURRENT_PE of %d Byte\n",
               lvm_name, size);
#ifdef DEBUG_KFREE
      printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
      kfree ( vg[VG(minor)]->lv[l]);
      vg[VG(minor)]->lv[l] = NULL;
      return -ENOMEM;
   }

#ifdef VERIFY_READ
   if ( ( ret = verify_area ( VERIFY_READ, pep, size)) != 0) {
      kfree ( vg[VG(minor)]->lv[l]);
      vg[VG(minor)]->lv[l] = NULL;
      return ret;
   }
#endif
   memcpy_fromfs ( vg[VG(minor)]->lv[l]->lv_current_pe, pep, size);

   vg[VG(minor)]->lv_cur++;

   /* correct the PE count in PVs */
   for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
      vg[VG(minor)]->pe_allocated++;
      for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
         if ( vg[VG(minor)]->pv[p]->pv_dev ==
              vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
            vg[VG(minor)]->pv[p]->pe_allocated++;
      }
   }

   lvm_gendisk.part[minor+l+1].start_sect = 0;
   lvm_gendisk.part[minor+l+1].nr_sects = vg[VG(minor)]->lv[l]->lv_size;
   lvm_size[minor+l+1] = vg[VG(minor)]->lv[l]->lv_size >> 1;

   vg[VG(minor)]->lv[l]->lv_status = lv_status_save;

   return 0;
}


/*
 * support function logical volume remove
 */
static int do_lv_remove ( int minor, char *lv_name) {
   uint l, le, p;

   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] != NULL &&
           lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name, lv_name) == 0) {
         if ( vg[VG(minor)]->lv[l]->lv_open > 0) return -EBUSY;
#ifdef DEBUG
         printk ( "%s -- fsync_dev and "
                  "invalidate_buffers for %s [%s] in %s\n",
                  lvm_name, vg[VG(minor)]->lv[l]->lv_name,
                  kdevname ( vg[VG(minor)]->lv[l]->lv_dev),
                  vg[VG(minor)]->vg_name);
#endif
         vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
         fsync_dev ( vg[VG(minor)]->lv[l]->lv_dev);
         vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
         invalidate_buffers ( vg[VG(minor)]->lv[l]->lv_dev);
         lvm_gendisk.part[minor].start_sect = 0;
         lvm_gendisk.part[minor].nr_sects = 0;
         lvm_size[minor] = 0;

         for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
            vg[VG(minor)]->pe_allocated--;
            for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
               if (  vg[VG(minor)]->pv[p]->pv_dev ==
                     vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
                  vg[VG(minor)]->pv[p]->pe_allocated--;
            }
         }

#ifdef DEBUG_KFREE
         printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
         kfree ( vg[VG(minor)]->lv[l]->lv_current_pe);

#ifdef DEBUG_KFREE
         printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
         kfree ( vg[VG(minor)]->lv[l]);

         vg[VG(minor)]->lv[l] = NULL;
         vg[VG(minor)]->lv_cur--;
         return 0;
      }
   }
   return -ENXIO;
}


/*
 * support function logical volume extend or reduce
 */
static int do_lv_extend_reduce ( int minor, char *lv_name, lv_t *lv,
                                 struct inode *inode) {
   int l, le, p, size;
#ifdef VERIFY_READ
   int ret;
#endif
   uint lv_status_save;
   pe_t *pe;

   if ( ( pep = lv->lv_current_pe) == NULL) return -EINVAL;

   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] != NULL &&
           lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name, lv_name) == 0)
         break;
   }
   if ( l == vg[VG(minor)]->lv_max) return -ENXIO;

   size = lv->lv_current_le * sizeof ( pe_t);
   if ( ( pe = kmalloc ( size, GFP_KERNEL)) == NULL) {
      printk ( "%s -- do_lv_extend_reduce: kmalloc error LV_CURRENT_PE "
               "of %d Byte\n", lvm_name, size);
      return -ENOMEM;
   }

#ifdef VERIFY_READ
   /* verify size of PE structures in user space */
   if ( ( ret = verify_area ( VERIFY_READ, pep, size)) != 0) {
      kfree ( pe);
      return ret;
   }
#endif

#ifdef DEBUG
   printk ( "%s -- fsync_dev and "
            "invalidate_buffers for %s [%s] in %s\n",
            lvm_name, vg[VG(minor)]->lv[l]->lv_name,
            kdevname ( vg[VG(minor)]->lv[l]->lv_dev),
            vg[VG(minor)]->vg_name);
#endif
   vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
   fsync_dev ( vg[VG(minor)]->lv[l]->lv_dev);
   vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
   invalidate_buffers ( vg[VG(minor)]->lv[l]->lv_dev);

   /* reduce allocation counters */
   for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
      vg[VG(minor)]->pe_allocated--;
      for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
         if (  vg[VG(minor)]->pv[p]->pv_dev ==
               vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
            vg[VG(minor)]->pv[p]->pe_allocated--;
      }
   }

#ifdef DEBUG_KFREE
   printk ( "%s -- kfree %d\n", lvm_name, __LINE__);
#endif
   kfree ( vg[VG(minor)]->lv[l]->lv_current_pe);

   lv_open = vg[VG(minor)]->lv[l]->lv_open;

   /* copy preloaded LV */
   lvm_memcpy ( ( char*) vg[VG(minor)]->lv[l], ( char*) lv, sizeof ( lv_t));
   lv_status_save = vg[VG(minor)]->lv[l]->lv_status;
   vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
   vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
   vg[VG(minor)]->lv[l]->lv_current_pe = pe;

   /* get the PE structures from user space */
   memcpy_fromfs ( vg[VG(minor)]->lv[l]->lv_current_pe, pep, size);

   vg[VG(minor)]->lv[l]->lv_open = lv_open;

   /* extend the PE count in PVs */
   for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
      vg[VG(minor)]->pe_allocated++;
      for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
         if ( vg[VG(minor)]->pv[p]->pv_dev ==
              vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
            vg[VG(minor)]->pv[p]->pe_allocated++;
      }
   }

   lvm_gendisk.part[minor+l+1].start_sect = 0;
   lvm_gendisk.part[minor+l+1].nr_sects = vg[VG(minor)]->lv[l]->lv_size;
   lvm_size[minor+l+1] = vg[VG(minor)]->lv[l]->lv_size >> 1;

   vg[VG(minor)]->lv[l]->lv_status = lv_status_save;

   return 0;
}


/*
 * support function initialize gendisk variables
 */
static void lvm_geninit ( struct gendisk *lvm_gdisk) {
   int i = 0;

#ifdef DEBUG_GENDISK
   printk ( "%s -- lvm_gendisk\n", lvm_name);
#endif

   for ( i = 0; i < MAX_VG * MAX_LV; i++) {
      lvm_size[i] = lvm_gendisk.part[i].nr_sects = 0;
      lvm_blocksizes[i] = BLOCK_SIZE;
      lvm_gendisk.part[i].start_sect = -1; /* avoid partition check */
   }

   blksize_size[LVM_MAJOR] = lvm_blocksizes;
   blk_size[LVM_MAJOR] = lvm_size;

   return;
}


#ifdef LVM_GET_INODE
/*
 * support function to get an empty inode
 *
 * Gets an empty inode to be inserted into the inode hash,
 * so that a physical volume can't be mounted.
 * This is analog to drivers/block/md.c
 *
 * Is this the real thing?
 *
 */
struct inode *lvm_get_inode ( int dev) {
   struct inode *inode_this = NULL;

   /* Lock the device analog MD by inserting a dummy inode. */
   inode_this = get_empty_inode ();
   inode_this->i_dev = dev;
   insert_inode_hash ( inode_this);
   return ( inode_this);
}
#endif


static int lvm_strcmp ( char *s1, char *s2) {
   while ( *s1 != 0 && *s2 != 0) {
      if ( *s1 != *s2) return -1;
      s1++; s2++;
   }
   if ( *s1 == 0 && *s2 == 0) return 0;
   return -1;
}


static char *lvm_strrchr ( char *s1, char c) {
   char *s2 = NULL;

   while ( *s1 != 0) {
      if ( *s1 == c) s2 = s1;
      s1++;
   }
   return s2;
}


static void lvm_memcpy ( char *dest, char *source, int size) {
   for ( ;size > 0; size--) {
      *dest = *source;
      dest++;
      source++;
   }
}
/* END internal support functions */
