/*
 * Author: Heinz Mauelshagen, Germany
 *
 * February-November 1997
 *
 * lvm is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * lvm is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *     10/10/1997 - beginning of new structure creation
 *     05/12/1998 - incorporated structures from lvm_v1.h and deleted lvm_v1.h
 *
 */

#ifndef _LVM_H_INCLUDE
#define _LVM_H_INCLUDE

#include <linux/kdev_t.h>

#define LVM_STRUCT_VERSION	1  /* structure version */
#define MIN_LVM_STRUCT_VERSION	1
#define MAX_LVM_STRUCT_VERSION	2


/* set the default structure version */
#if ( LVM_STRUCT_VERSION == 1)
#  define pv_t pv_v1_t
#  define lv_t lv_v1_t
#  define vg_t vg_v1_t
#else
#  if ( LVM_STRUCT_VERSION == 2)
#    define pv_t pv_v2_t
#    define lv_t lv_v2_t
#    define vg_t vg_v2_t
#  endif
#endif

#if ( LVM_STRUCT_VERSION != 1 &&  LVM_STRUCT_VERSION != 2)
   #pragme Please set LVM_STRUCT_VERSION above
#endif


#define	LVM_MAJOR	42         /* my testing major number */
#define	LVM_IOP_VERSION	        2  /* i/o protocoll version */
#define LVM_NAME        "lvm"

#define ABS_MAX_VG	99	/* absolute maximum for VGs */
#define ABS_MAX_PV	256	/* absolute maximum for PVs */
#define ABS_MAX_LV	256	/* absolute maximum for LVs */

/*
 * if we will have more than 8 bits for minor numbers in the future,
 * I will limit myself to 12 for memory usage reasons.
 *
 * this has to be enhanced!!!
 */
#if ( MINORBITS >= 12)
#	define LVM_MINORBITS	12
#   define MAX_VG_BITS 5 /* otherwise too much memory for hd_structs ... */
#   define BITS 0xFFF
#else
#	define LVM_MINORBITS	8
#   define MAX_VG_BITS 3
#   define BITS 0xFF
#endif

#define VG_BITSHIFT ( LVM_MINORBITS - MAX_VG_BITS)
#define VG_BITMASK	( ~( ( BITS >> MAX_VG_BITS) << MAX_VG_BITS) << VG_BITSHIFT)
#define VG(a) ( ( a & VG_BITMASK) >> VG_BITSHIFT)
#define LV(a) ( a & ~VG_BITMASK)
#define MAX_VG  ( 1 << MAX_VG_BITS)
#define MAX_LV	( ( 1 << ( LVM_MINORBITS - MAX_VG_BITS)) - 1)
#define	MAX_PV	ABS_MAX_PV

#if ( MAX_VG > ABS_MAX_VG)
#   undef MAX_VG
#   define MAX_VG ABS_MAX_VG
#endif

#if ( MAX_LV > ABS_MAX_LV)
#   undef MAX_LV
#   define MAX_LV ABS_MAX_LV
#endif

/* default disk spaces and offsets for PVs */
#define	LVM_PV_DISK_BASE  	0L    /* base of PV structure in disk partition */
#define	LVM_PV_DISK_SIZE  	1024L /* size reserved for PV structure on disk */
#define	LVM_VG_DISK_BASE  	1024L /* base of VG structure in disk partition */
#define	LVM_VG_DISK_SIZE  	( 5 * 1024L) /* size reserved for VG structure */
#define	LVM_PV_NAMELIST_DISK_BASE	( LVM_VG_DISK_BASE + LVM_VG_DISK_SIZE)
#define	LVM_LV_DISK_OFFSET(a, b)	( (a)->lv_on_disk.base + sizeof ( lv_t) * b)
#define	LVM_DISK_SIZE(pv)		( (pv)->pe_on_disk.base + \
                                          (pv)->pe_on_disk.size)


/*
 * LVM_KMALLOC_MAX corresponds to:
 *
 * 8KB PE size can map a 3GB logical volume at the cost of 3MB memory,
 * but the kmalloc restrictions in Linux 2.0 (LVM_KMALLOC_MAX) only allow
 * a size of 127,9xMB.
 *
 * 128MB PE size can map a 48TB logical volume at the cost of 3 MB memory, but
 * LVM_KMALLOC_MAX only ;-) allows a size of 1.999xTB.
 *
 * Default PE size of 1MB gives a maximum logical volume size of 16GB.
 *
 * AFAIK, the kernel is able to handle 1 TB for one disk (array).
 *
 * Should be a sufficient spectrum ;-)
 */
#define	LVM_KMALLOC_MAX		131048	/* max allocation kmalloc */
#define	LVM_PE_T_MAX		( LVM_KMALLOC_MAX / sizeof ( pe_t))
                            /* max # of pe_t structures possible in kmalloc */
#define	LVM_LV_SIZE_MAX(a)	( LVM_PE_T_MAX * (a)->pe_size)
#define	LVM_MIN_PE_SIZE		( 8L * 2)           /* 8 KB in sectors */
#define	LVM_MAX_PE_SIZE		( 512L * 1024L * 2) /* 512 MB in sectors */
#define	LVM_DEFAULT_PE_SIZE	( 4096L * 2)        /* 4 MB in sectors */
#define	LVM_DEFAULT_STRIPE_SIZE	16L             /* 16 KB  */
#define	LVM_MIN_STRIPE_SIZE	2L                  /* 1 KB in sectors */
#define	LVM_MAX_STRIPE_SIZE	( 2 * 128L)         /* 128 KB in sectors */
#define	LVM_MAX_STRIPES		128					/* max # of stripes */
#define	LVM_MAX_SIZE	    ( 1024L * 1024L * 1024L) /* 1TB */
#define	LVM_MAX_MIRRORS    	2		/* future use */
#define	LVM_MIN_READ_AHEAD	2		/* min read ahead sectors (future use) */
#define	LVM_MAX_READ_AHEAD	120		/* max read ahead sectors (future use) */
#define	LVM_PARTITION           0xfe	/* LVM partition id */

#define	UNDEF	-1
#define FALSE	0
#define TRUE	1


/*
 * ioctls
 */

/* volume group */
#define	VG_CREATE               _IO ( 0xfe, 0x00)
#define	VG_REMOVE               _IO ( 0xfe, 0x01)

#define	VG_EXTEND               _IO ( 0xfe, 0x03)
#define	VG_REDUCE               _IO ( 0xfe, 0x04)

#define	VG_STATUS               _IO ( 0xfe, 0x05)
#define	VG_STATUS_GET_COUNT     _IO ( 0xfe, 0x06)
#define	VG_STATUS_GET_NAMELIST  _IO ( 0xfe, 0x07)

#define	VG_SET_EXTENDABLE       _IO ( 0xfe, 0x08)


/* logical volume */
#define	LV_CREATE               _IO ( 0xfe, 0x20)
#define	LV_REMOVE               _IO ( 0xfe, 0x21)

#define	LV_ACTIVATE             _IO ( 0xfe, 0x22)
#define	LV_DEACTIVATE           _IO ( 0xfe, 0x23)

#define	LV_EXTEND               _IO ( 0xfe, 0x24)
#define	LV_REDUCE               _IO ( 0xfe, 0x25)

#define	LV_STATUS_BYNAME        _IO ( 0xfe, 0x26)
#define	LV_STATUS_BYINDEX       _IO ( 0xfe, 0x27)

#define LV_SET_ACCESS           _IO ( 0xfe, 0x28)
#define LV_SET_ALLOCATION       _IO ( 0xfe, 0x29)
#define LV_SET_STATUS           _IO ( 0xfe, 0x2a)

#define LE_REMAP                _IO ( 0xfe, 0x2b)


/* physical volume */
#define	PV_STATUS               _IO ( 0xfe, 0x40)
#define	PV_CHANGE               _IO ( 0xfe, 0x41)
#define	PV_FLUSH                _IO ( 0xfe, 0x42)

/* physical extend */
#define	PE_LOCK_UNLOCK          _IO ( 0xfe, 0x50)

/* i/o protocol version */
#define	LVM_GET_IOP_VERSION     _IO ( 0xfe, 0x98)

/* special reset function for testing purposes */
#define	LVM_RESET               _IO ( 0xfe, 0x99)


/*
 * Status flags
 */

/* volume group */
#define	VG_ACTIVE            0x01 /* vg_status */
#define	VG_EXPORTED          0x02 /*     "     */
#define	VG_EXTENDABLE        0x04 /*     "     */

#define	VG_READ              0x01 /* vg_access */
#define	VG_WRITE             0x02 /*     "     */

/* logical volume */
#define	LV_ACTIVE            0x01 /* lv_status */
#define	LV_SPINDOWN          0x02 /*     "     */
#define	LV_READ              0x01 /* lv_access */
#define	LV_WRITE             0x02 /*      "    */
#define	LV_BADBLOCK_ON       0x01 /* lv_badblock */
#define	LV_STRICT            0x01 /* lv_allocation */
#define	LV_CONTIGUOUS        0x02 /*       "       */

/* physical volume */
#define	PV_ACTIVE            0x01 /* pv_status */
#define	PV_ALLOCATABLE       0x02 /* pv_allocatable */


/*
 * Structure definitions core/disk
 */

#define	LVM_ID		"HM"      /* Identifier PV (id in pv_t) */
#define	EXPORTED	"PV_EXP"  /* Identifier exported PV (system_id in pv_t) */
#define	IMPORTED	"PV_IMP"  /* Identifier imported PV (        "        ) */
#define	LVMTAB                  "/etc/lvmtab"	/* LVM table of VGs */
#define	LVMTAB_DIR              "/etc/lvmtab.d"	/* storage dir VG data */
#define	LVMTAB_MINSIZE   ( sizeof ( vg_t) + sizeof ( lv_t) + sizeof ( pv_t))
#define	LVM_DEV                 "/dev/lvm"
#define	VG_BACKUP_DIR           "/etc/lvmconf"
#define	NAME_LEN		128
#define	DISK_NAME_LEN		8
#define	LV_MIN_NAME_LEN		5
#define	LV_MAX_NAME_LEN		7
#define	MIN_PART		1
#define	MAX_PART	 	15


/* disk stored pe information */
typedef struct {
   unsigned short lv_num;
   unsigned short le_num;
} disk_pe_t;

/* disk stored PV, VG, LV and PE size and offset information */
typedef struct {
   unsigned long base;
   unsigned long size;
} lvm_disk_data_t;


/*
 * Structure Physical Volume (PV) Version 1
 */

typedef struct {
   char             id[2];                 /* Identifier */
   unsigned short   version;               /* HM lvm version */
   lvm_disk_data_t  pv_on_disk;
   lvm_disk_data_t  vg_on_disk;
   lvm_disk_data_t  pv_namelist_on_disk;
   lvm_disk_data_t  lv_on_disk;
   lvm_disk_data_t  pe_on_disk;
   char             pv_name[NAME_LEN];
   char             vg_name[NAME_LEN];
   char             system_id[NAME_LEN]; /* for vgexport/vgimport */
   kdev_t           pv_dev;
   unsigned long    pv_number;
   unsigned long    pv_status;
   unsigned long    pv_allocatable;
   unsigned long    pv_size;		/* HM */
   unsigned long    lv_cur;
   unsigned long    pe_size;
   unsigned long    pe_total;
   unsigned long    pe_allocated;
   unsigned long    pe_stale;     /* for future use */
   disk_pe_t        *pe;          /* HM */
   struct inode     *inode;       /* HM */
} pv_v1_t;


/*
 * Structure Physical Volume (PV) Version 2
 */

typedef struct {
   char             id[2];                 /* Identifier */
   unsigned short   version;               /* HM lvm version */
   lvm_disk_data_t  pv_on_disk;
   lvm_disk_data_t  vg_on_disk;
   lvm_disk_data_t  pv_namelist_on_disk;
   lvm_disk_data_t  lv_on_disk;
   lvm_disk_data_t  pe_on_disk;
   char             pv_name[NAME_LEN];
   char             vg_name[NAME_LEN];
   char             system_id[NAME_LEN]; /* for vgexport/vgimport */
   kdev_t           pv_dev;
   unsigned long    pv_number;
   unsigned long    pv_status;
   unsigned long    pv_allocatable;
   unsigned long    pv_size;		/* HM */
   unsigned long    lv_cur;
   unsigned long    pe_size;
   unsigned long    pe_total;
   unsigned long    pe_allocated;
   unsigned long    pe_stale;     /* for future use */
   disk_pe_t        *pe;          /* HM */
   struct inode     *inode;       /* HM */
   /* delta to version 1 starts here */
   time_t           pv_ctime;     /* PV creation time */
   time_t           pv_mtime;     /* PV modification time */
   char             pv_uuid[NAME_LEN];
} pv_v2_t;


/*
 * Structures for Logical Volume (LV)
 */

/* core PE information */
typedef struct {
   kdev_t dev;
   unsigned long pe;
} pe_t;

typedef struct {
   char   lv_name[NAME_LEN];
   kdev_t old_dev;
   kdev_t new_dev;
   ulong  old_pe;
   ulong  new_pe;
} le_remap_req_t;


/*
 * Structure Logical Volume (LV) Version 1
 */

typedef struct {
   char           lv_name[NAME_LEN];
   char           vg_name[NAME_LEN];
   unsigned long  lv_access;
   unsigned long  lv_status;
   unsigned long  lv_open;			/* HM */
   kdev_t         lv_dev;			/* HM */
   unsigned long  lv_number;		/* HM */
   unsigned long  lv_mirror_copies;	/* for future use */
   unsigned long  lv_recovery;      /*       "        */
   unsigned long  lv_schedule;		/*       "        */
   unsigned long  lv_size;
   pe_t           *lv_current_pe;	/* HM */
   unsigned long  lv_current_le;	/* for future use */
   unsigned long  lv_allocated_le;
   unsigned long  lv_stripes;
   unsigned long  lv_stripesize;
   unsigned long  lv_badblock;		/* for future use */
   unsigned long  lv_allocation;
   unsigned long  lv_io_timeout;	/* for future use */
   unsigned long  lv_read_ahead;	/* HM, for future use */
} lv_v1_t;


/*
 * Structure Logical Volume (LV) Version 2
 */

typedef struct {
   char           lv_name[NAME_LEN];
   char           vg_name[NAME_LEN];
   unsigned long  lv_access;
   unsigned long  lv_status;
   unsigned long  lv_open;			/* HM */
   kdev_t         lv_dev;			/* HM */
   unsigned long  lv_number;		/* HM */
   unsigned long  lv_mirror_copies;	/* for future use */
   unsigned long  lv_recovery;      /*       "        */
   unsigned long  lv_schedule;		/*       "        */
   unsigned long  lv_size;
   pe_t           *lv_current_pe;	/* HM */
   unsigned long  lv_current_le;	/* for future use */
   unsigned long  lv_allocated_le;
   unsigned long  lv_stripes;
   unsigned long  lv_stripesize;
   unsigned long  lv_badblock;		/* for future use */
   unsigned long  lv_allocation;
   unsigned long  lv_io_timeout;	/* for future use */
   unsigned long  lv_read_ahead;	/* HM, for future use */
   /* delta to version 1 starts here */
   time_t         lv_ctime;             /* LV creation time */
   time_t         lv_mtime;             /* LV modification time */
} lv_v2_t;


/*
 * Structure Volume Group (VG) Version 1
 */

typedef struct {
   char           vg_name[NAME_LEN];     /* volume group name */
   unsigned long  vg_number;             /* volume group number */
   unsigned long  vg_access;             /* read/write */
   unsigned long  vg_status;             /* active or not */
   unsigned long  lv_max;                /* maximum logical volumes */
   unsigned long  lv_cur;                /* current logical volumes */
   unsigned long  lv_open;               /* open    logical volumes */
   unsigned long  pv_max;        /* maximum physical volumes */
   unsigned long  pv_cur;		 /* current physical volumes FU */
   unsigned long  pv_act;        /* active physical volumes */
   unsigned long  max_pe_per_pv; /* maximum physical extends per physical vol */
   unsigned long  vgda;			 /* volume group descriptor arrays FU */
   unsigned long  pe_size;       /* physical extend size in sectors */
   unsigned long  pe_total;      /* total of physical extends */
   unsigned long  pe_allocated;  /* allocated physical extends */
   unsigned long  pvg_total;		 /* physical volume groups FU */
   struct proc_dir_entry *proc;
   pv_t           *pv[ABS_MAX_PV+1]; /* physical volume struct pointers */
   lv_t           *lv[ABS_MAX_LV+1]; /* logical  volume struct pointers */
} vg_v1_t;


/*
 * Structure Volume Group (VG) Version 2
 */

typedef struct {
   char           vg_name[NAME_LEN];     /* volume group name */
   unsigned long  vg_number;             /* volume group number */
   unsigned long  vg_access;             /* read/write */
   unsigned long  vg_status;             /* active or not */
   unsigned long  lv_max;                /* maximum logical volumes */
   unsigned long  lv_cur;                /* current logical volumes */
   unsigned long  lv_open;               /* open    logical volumes */
   unsigned long  pv_max;                /* maximum physical volumes */
   unsigned long  pv_cur;		 /* current physical volumes FU */
   unsigned long  pv_act;                /* future: active physical volumes */
   unsigned long  max_pe_per_pv;         /* OBSOLETE maximum PE/PV */
   unsigned long  vgda;			 /* volume group descriptor arrays FU */
   unsigned long  pe_size;               /* physical extend size in sectors */
   unsigned long  pe_total;              /* total of physical extends */
   unsigned long  pe_allocated;          /* allocated physical extends */
   unsigned long  pvg_total;		 /* physical volume groups FU */
   struct proc_dir_entry *proc;
   pv_t           *pv[ABS_MAX_PV+1]; /* physical volume struct pointers */
   lv_t           *lv[ABS_MAX_LV+1]; /* logical  volume struct pointers */
   /* delta to version 1 starts here */
   time_t         vg_ctime;              /* VG creation time */
   time_t         vg_mtime;              /* VG modification time */
} vg_v2_t;


/*
 * Request structure PV_STATUS
 */
typedef struct {
   char pv_name[NAME_LEN];
   pv_t *pv;
} pv_status_req_t, pv_change_req_t;

/*
 * Request structure PV_FLUSH
 */
typedef struct {
   char pv_name[NAME_LEN];
} pv_flush_req_t;


/*
 * Request structure PE_MOVE
 */
typedef struct {
   enum { LOCK_PE, UNLOCK_PE} lock;
   struct {
      kdev_t lv_dev;
      kdev_t pv_dev;
      unsigned long pv_offset;
   } data;
} pe_lock_req_t;


/*
 * Request structure LV_STATUS
 */
typedef struct {
   char lv_name[NAME_LEN];
   lv_t *lv;
} lv_status_byname_req_t, lv_req_t;

typedef struct {
   ulong lv_index;
   lv_t *lv;
} lv_status_byindex_req_t;

#endif /* #ifndef _LVM_H_INCLUDE */
