/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March 1997
 * May   1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    01/15/1998 - extended compare in lv_read() for LV names with
 *                 and without path
 *    05/01/1998 - rewritten lv_read() and lv_read_all_lv_of_vg()
 *                 to avoid LV number lookup from LV name
 *               - implemented lv_read_byindex()
 *
 */

#include <liblvm.h>

int lv_read_byindex ( char *vg_name, ulong lv_index, lv_t **lv) {
   int ret = 0;
   ulong l = 0;
   static lv_t lv_this;
   lv_t **lv_this_ptr;
   vg_t *vg_this = NULL;
   vg_t vg_this_sav;


#ifdef DEBUG
   debug ( "lv_read_byindex-- CALLED with: \"%s\" %d %X\n",
            vg_name, lv_index, ( uint) lv);
#endif

   if ( vg_name == NULL || lv == NULL ||
        vg_check_name ( vg_name) < 0) return -LVM_EPARA;

   *lv = NULL;

   if ( ( ret = vg_read ( vg_name, &vg_this)) < 0 &&
        ret != -LVM_EVG_READ_VG_EXPORTED) {
      return -LVM_ELV_READ_BYINDEX_VGREAD;
   }
   if ( lv_index > vg_this->lv_max - 1) return -LVM_EPARA;

   memcpy ( &vg_this_sav, vg_this, sizeof ( vg_t));
   vg_this = &vg_this_sav;

#ifdef DEBUG
   debug ( "lv_read_byindex-- BEFORE lf_read_all_lv_of_vg\n");
#endif
   if ( ( ret = lv_read_all_lv_of_vg ( vg_name, &lv_this_ptr, FALSE)) < 0)
      return -LVM_ELV_READ_BYINDEX_LV_READ_ALL_LV_OF_VG;

   ret = -LV_READ_LV;
   for ( l = 0; l < vg_this->lv_max; l++) {
      if ( lv_this_ptr[l] != NULL) {
         if ( lv_this_ptr[l]->lv_number == lv_index) {
            ret = 0;
            break;
         }
      }
   }
   if ( ret == 0) {
      if ( strcmp ( lv_this_ptr[lv_index]->vg_name, vg_name) != 0)
         ret = -LVM_ELV_READ_BYINDEX_VG_NAME;
      else {
         memcpy ( &lv_this, lv_this_ptr[l], sizeof ( lv_t));
         *lv = &lv_this;
         ret = 0;
      }
   }

#ifdef DEBUG
   debug ( "lv_read_byindex-- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lv_read ( char *vg_name, char *lv_name, lv_t **lv) {
   int l = 0;
   int ret = 0;
   static lv_t lv_this;
   lv_t **lv_this_ptr;
   vg_t *vg_this = NULL;
   vg_t vg_this_sav;


#ifdef DEBUG
   debug ( "lv_read -- CALLED with: \"%s\" \"%s\" %X\n",
            vg_name, lv_name, ( uint) lv);
   if ( vg_name == NULL ||
        lv_name == NULL ||
        lv == NULL ||
        vg_check_name ( vg_name) < 0 ||
        lv_check_name ( lv_name) < 0) return -LVM_EPARA;
#endif

   *lv = NULL;

   if ( ( ret = vg_read ( vg_name, &vg_this)) < 0 &&
        ret != -LVM_EVG_READ_VG_EXPORTED) {
      return -LVM_ELV_READ_VGREAD;
   }
   memcpy ( &vg_this_sav, vg_this, sizeof ( vg_t));
   vg_this = &vg_this_sav;

#ifdef DEBUG
   debug ( "lv_read -- BEFORE lf_read_all_lv_of_vg\n");
#endif
   if ( ( ret = lv_read_all_lv_of_vg ( vg_name, &lv_this_ptr, TRUE)) < 0)
      return -LVM_ELV_READ_LV_READ_ALL_LV_OF_VG;

   ret = -LV_READ_LV;
   for ( l = 0; l < vg_this->lv_max; l++) {
      if ( strcmp ( lv_this_ptr[l]->lv_name, lv_name) == 0) {
         memcpy ( &lv_this, lv_this_ptr[l], sizeof ( lv_t));
         *lv = &lv_this;
         if ( strcmp ( lv_this.vg_name, vg_name) != 0)
            ret = -LVM_ELV_READ_VG_NAME;
         else ret = 0;
      }
   }

#ifdef DEBUG
   debug ( "lv_read -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lv_read_with_pe ( char *vg_name, char *lv_name, lv_t **lv) {
   int l = 0;
   int ret = 0;
   vg_t *vg = NULL;

#ifdef DEBUG
   debug ( "lv_read_with_pe -- CALLED\n");
   if ( vg_name == NULL || lv_name == NULL || lv == NULL ||
        lv_check_name ( lv_name) < 0) return -LVM_EPARA;
#endif

   ret = vg_read_with_pv_and_lv ( vg_name, &vg);

   *lv = NULL;
   if ( ret == 0 || ret == -LVM_EVG_READ_VG_EXPORTED) {
      for ( l = 0; l < vg->lv_max; l++) {
         if ( vg->lv[l] != NULL &&
              strcmp ( vg->lv[l]->lv_name, lv_name) == 0) {
            *lv = vg->lv[l];
            break;
         }
      }
   }

#ifdef DEBUG
   debug ( "lv_read_with_pe -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lv_read_all_lv_of_vg ( char *vg_name, lv_t ***lv, int reread) {
   int l  = 0;
   int nl = 0;
   int pv_handle = -1;
   int ret = 0;
   ulong offset = 0;
   static int first = 0;
   size_t size = 0;
   vg_t *vg_this = NULL;
   vg_t vg_this_sav;
   static char vg_name_sav[NAME_LEN] = { 0, };
   static lv_t **lv_this_ptr = NULL;
   static lv_t *lv_this = NULL;
   pv_t **pv_tmp = NULL;

#ifdef DEBUG
   debug ( "lv_read_all_lv_of_vg -- CALLED\n");
   if ( lv == NULL || vg_name == NULL ||
        ( reread != TRUE && reread != FALSE) ||
        vg_check_name ( vg_name) != 0) return -LVM_EPARA;
#endif

   *lv = NULL;

   if ( strcmp ( vg_name, vg_name_sav) != 0) {
      strcpy ( vg_name_sav, vg_name);
      reread = TRUE;
   }

   if ( reread == TRUE) {
      *vg_name_sav = 0;
      if ( lv_this_ptr != NULL) {
         free ( lv_this_ptr);
         lv_this_ptr = NULL;
      }
      if ( lv_this != NULL) {
         free ( lv_this);
         lv_this = NULL;
      }
      first = 0;
   }

   /* first time ... */
   if ( first == 0) {
      if ( ( ret = vg_read ( vg_name, &vg_this)) < 0 &&
           ret != -LVM_EVG_READ_VG_EXPORTED) {
         return -LVM_ELV_READ_ALL_LV_OF_VG_VGREAD;
      }
      memcpy ( &vg_this_sav, vg_this, sizeof ( vg_t));
      vg_this = &vg_this_sav;

#ifdef DEBUG
      debug ( "lv_read_all_lv_of_vg -- lv_max: %lu\n", vg_this->lv_max);
#endif

      size = vg_this->lv_max * sizeof ( lv_t*);
      if ( ( lv_this_ptr = malloc ( size)) == NULL) {
         fprintf ( stderr, "malloc error in %s [line %d]\n",
                           __FILE__, __LINE__);
         return -LVM_ELV_READ_ALL_LV_OF_VG_MALLOC;
      }
      memset ( lv_this_ptr, 0, size);

      size = vg_this->lv_max * sizeof ( lv_t);
      if ( ( lv_this = malloc ( size)) == NULL) {
         free ( lv_this_ptr);
         lv_this_ptr = NULL;
         return -LVM_ELV_READ_ALL_LV_OF_VG_MALLOC;
      }

#ifdef DEBUG
      debug ( "lv_read_all_lv_of_vg -- BEFORE pv_read_all_pv_of_vg\n");
#endif
      if ( ( ret = pv_read_all_pv_of_vg ( vg_name, &pv_tmp, FALSE)) < 0 &&
           ret != -LVM_EPV_READ_PV_EXPORTED)
         return ret;

      offset = LVM_LV_DISK_OFFSET ( pv_tmp[0], 0);

      if ( ( pv_handle = open ( pv_tmp[0]->pv_name, O_RDONLY)) == -1)
         ret = -LVM_ELV_READ_ALL_LV_OF_VG_OPEN;
      else if ( lseek ( pv_handle, offset, SEEK_SET) != offset)
         ret = -LVM_ELV_READ_ALL_LV_OF_VG_LSEEK;
      else if ( read ( pv_handle, lv_this, size) != size)
         ret = -LVM_ELV_READ_ALL_LV_OF_VG_READ;
   
      if ( ret >= 0) {
         nl = 0;
         for ( l = 0; l < vg_this->lv_max; l++) {
            lv_this_ptr[l] = NULL;
            if ( vg_this->lv[l] != NULL) {
               lv_this_ptr[l] = &lv_this[l];
               nl++;
            }
         }
   
#ifdef DEBUG
         debug ( "lv_read_all_lv_of_vg -- l: %d  nl: %d  "
                 "vg_this->lv_cur: %lu\n",
                 l, nl, vg_this->lv_cur);
#endif
         if ( nl != vg_this->lv_cur) ret = -LVM_ELV_READ_ALL_LV_OF_VG_NL;
         else                        ret = 0;
         strcpy ( vg_name_sav, vg_name);
         first = 1;
      }
   }

   *lv = lv_this_ptr;
   if ( pv_handle != -1) close ( pv_handle);

#ifdef DEBUG
   debug ( "lv_read_all_lv_of_vg -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}
