/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March,October,November 1997
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/08/1997 - removed setting PV-pointer in vg_read
 *    11/13/1997 - implemented vg_read_from_pv ()
 *
 */


#include <liblvm.h>

int vg_read ( char *vg_name, vg_t **vg) {
   int p = 0;
   int pv_handle = -1;
   int ret = 0;
   static vg_t vg_this;
   pv_t **pv = NULL;

#ifdef DEBUG
   debug ( "vg_read -- CALLED\n");
#endif

   if ( vg_name == NULL ||
        vg_check_name ( vg_name) < 0 ||
        vg == NULL) return -LVM_EPARA;

   *vg = NULL;

   if ( ( ret = pv_read_all_pv_of_vg ( vg_name, &pv, FALSE)) < 0) {
#ifdef DEBUG
      debug ( "vg_read -- pv_read_all_pv_of_vg returned: %d\n", ret);
#endif
      return ret;
   }

   if ( pv != NULL && pv[0] != NULL) {
#ifdef DEBUG
      if ( opt_d > 0) {
         for ( p = 0; pv[p] != NULL; p++)
            printf ( "vg_read -- pv[%d]->pv_name: \"%s\"\n", p, pv[p]->pv_name);
      }
#endif
   
      /* read the VG info */
      if ( ( pv_handle = open ( pv[0]->pv_name, O_RDONLY)) == -1)
         ret = -LVM_EVG_READ_OPEN;
      else if ( lseek ( pv_handle, pv[0]->vg_on_disk.base, SEEK_SET) != 
                pv[0]->vg_on_disk.base) ret = -LVM_EVG_READ_LSEEK;
      else if ( read ( pv_handle, &vg_this, sizeof ( vg_t)) != \
                sizeof ( vg_t)) ret = -LVM_EVG_READ_READ;
      for ( p = 0; p < vg_this.pv_max; p++) vg_this.pv[p] = NULL;
   
      if ( pv_handle != -1) close ( pv_handle);
      if ( vg_this.vg_status & VG_EXPORTED) ret = -LVM_EVG_READ_VG_EXPORTED;
   
      *vg = &vg_this;
      if ( ret == 0) {
         for ( p = 0; pv != NULL && pv[p] != NULL; p++) {
            if ( pv[p]->version != LVM_STRUCT_VERSION) {
               ret = -LVM_EVG_READ_LVM_STRUCT_VERSION;
               break;
            }
         }
      }
   } else ret = -LVM_EVG_READ_PV;

#ifdef DEBUG
   debug ( "vg_read -- LEAVING %d\n", ret);
#endif
   return ret;
}


int vg_read_from_pv ( char *pv_name, vg_t **vg) {
   int p = 0;
   int pv_handle = -1;
   int ret = 0;
   static vg_t vg_this;
   pv_t *pv = NULL;

#ifdef DEBUG
   debug ( "vg_read_from_pv -- CALLED\n");
#endif

   if ( pv_name == NULL ||
        pv_check_name ( pv_name) < 0 ||
        vg == NULL) return -LVM_EPARA;

   *vg = NULL;

   if ( pv_read ( pv_name, &pv, NULL) < 0)
      ret = -LVM_EVG_READ_FROM_PV_PV_READ;
   else if ( pv_check_consistency ( pv) < 0)
      ret = -LVM_EVG_READ_FROM_PV_CHECK_CONSISTENCY;
   else if ( ( pv_handle = open ( pv_name, O_RDONLY)) == -1)
      ret = -LVM_EVG_READ_FROM_PV_OPEN;
   else if ( lseek ( pv_handle, pv->vg_on_disk.base, SEEK_SET) != 
             pv->vg_on_disk.base) ret = -LVM_EVG_READ_FROM_PV_LSEEK;
   else if ( read ( pv_handle, &vg_this, sizeof ( vg_t)) != \
             sizeof ( vg_t)) ret = -LVM_EVG_READ_FROM_PV_READ;
   else {
      vg_this.pv[0] = pv;
      for ( p = 1; p < vg_this.pv_max; p++) vg_this.pv[p] = NULL;
   
      if ( vg_this.vg_status & VG_EXPORTED) ret = -LVM_EVG_READ_VG_EXPORTED;
   
      *vg = &vg_this;

      if ( ret == 0 && pv->version != LVM_STRUCT_VERSION)
         ret = -LVM_EVG_READ_FROM_PV_LVM_STRUCT_VERSION;
   }

   if ( pv_handle != -1) close ( pv_handle);
#ifdef DEBUG
   debug ( "vg_read_from_pv -- LEAVING %d\n", ret);
#endif
   return ret;
}


int vg_read_with_pv_and_lv ( char *vg_name, vg_t **vg) {
   int l = 0;
   int lv_num = 0;
   int p = 0;
   int ope = 0;
   int npe = 0;
   int ret = 0;
   uint pe_index = 0;
   disk_pe_t **pe = NULL;
   lv_t **lv = NULL;
   vg_t *vg_this = NULL;
   pv_t **pv = NULL;

#ifdef DEBUG
   debug ( "vg_read_with_pv_and_lv -- CALLED\n");
#endif

   if ( vg_name == NULL || vg == NULL ||
        vg_check_name ( vg_name) < 0) return -LVM_EPARA;

   *vg = NULL;
   if ( ( ret = vg_read ( vg_name, &vg_this)) < 0 &&
        ret != -LVM_EVG_READ_VG_EXPORTED) return ret;

#ifdef DEBUG
   debug ( "vg_read_with_pv_and_lv -- AFTER vg_read; "
            " vg_this->lv_cur: %lu\n", vg_this->lv_cur);
   debug ( "vg_read_with_pv_and_lv -- vg_show AFTER first read\n");
   if ( opt_d > 0) vg_show ( *vg);
#endif

   if ( vg_this->pv_cur == 0) return -LVM_EVG_READ_WITH_PV_AND_LV_PV_CUR;

   if ( ( ret = pv_read_all_pv_of_vg ( vg_name, &pv, FALSE)) < 0)
      return ret;
#ifdef DEBUG
   debug ( "vg_read_with_pv_and_lv -- AFTER pv_read_all_pv_of_vg\n");
#endif

   if ( ( ret = pv_read_all_pe_of_vg ( vg_name, &pe, FALSE)) < 0)
      return ret;
#ifdef DEBUG
   debug ( "vg_read_with_pv_and_lv -- AFTER pv_read_all_pe_of_vg\n");
#endif

   if ( ( ret = lv_read_all_lv_of_vg ( vg_name, &lv, FALSE)) < 0)
      return ret;
#ifdef DEBUG
   debug ( "vg_read_with_pv_and_lv -- AFTER lv_read_all_lv_of_vg;"
            " vg_this->pv_cur: %lu  vg_this->pv_max: %lu  ret: %d\n",
            vg_this->pv_cur, vg_this->pv_max, ret);
#endif

   if ( ret == 0) {
#ifdef DEBUG
      debug (  "vg_read_with_pv_and_lv -- BEFORE for PE\n");
#endif
      /* set up PE pointers in PVs */
      for ( p = 0; p < vg_this->pv_cur; p++) {
         vg_this->pv[p] = pv[p];
         vg_this->pv[p]->pe = pe[p];
      }
#ifdef DEBUG
      debug (  "vg_read_with_pv_and_lv -- AFTER for PE\n");
#endif

      /* build the lv_current_pe structure array */
#ifdef DEBUG
      debug (  "vg_read_with_pv_and_lv -- BEFORE for LV\n");
#endif
      for ( l = 0; l < vg_this->lv_max; l++) vg_this->lv[l] = NULL;
      if ( vg_this->lv_cur > 0) {
         for ( l = 0; l < vg_this->lv_max; l++) {
            lv_num = l + 1;
            if ( lv[l] != NULL) {
               vg_this->lv[l] = lv[l];
#ifdef DEBUG
               debug ( "vg_read_with_pv_and_lv -- vg_this->lv[%d]->"
                        "lv_allocated_le: %lu\n", l,
                        vg_this->lv[l]->lv_allocated_le);
#endif
               if ( ( vg_this->lv[l]->lv_current_pe = malloc (
                         vg_this->lv[l]->lv_allocated_le *
                         sizeof ( pe_t))) == NULL) {
                  fprintf ( stderr, "malloc error in %s [line %d]\n",
                                    __FILE__, __LINE__);
                  return -LVM_EVG_READ_WITH_PV_AND_LV_MALLOC;
               }
               p = npe = 0;
               while ( p < vg_this->pv_cur &&
                       npe < vg_this->lv[l]->lv_allocated_le) {
                  for ( ope = 0; ope < vg_this->pv[p]->pe_total; ope++) {
                     if ( vg_this->pv[p]->pe[ope].lv_num == lv_num) {
                        pe_index = vg_this->pv[p]->pe[ope].le_num;
                        vg_this->lv[l]->lv_current_pe[pe_index].dev =
                           vg_this->pv[p]->pv_dev;
                        vg_this->lv[l]->lv_current_pe[pe_index].pe =
                           ( ope * vg_this->pv[p]->pe_size) +
                           ( LVM_DISK_SIZE ( vg_this->pv[p]) / SECTOR_SIZE);
                        npe++;
                     }
                  }
                  p++;
               }
               if ( npe != vg_this->lv[l]->lv_allocated_le)
                  return -LVM_EVG_READ_WITH_PV_AND_LV_LV_ALLOCATED_LE;
            } else vg_this->lv[l] = NULL;
         }
      }
   } /* if ( ret == 0) */

   if ( ret == 0) *vg = vg_this;

#ifdef DEBUG
   debug ( "vg_read_with_pv_and_lv -- LEAVING %d\n", ret);
#endif
   return ret;
}
