/*
 * Author: Heinz Mauelshagen, Germany
 *
 * June,October 1997
 * April,May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/09/1997 - added lvmtab handling
 *    05/07/1998 - implemented -n for LV selection by name
 *    05/16/1998 - added lvmtab checking
 *
 */

/*
 * TODO
 *
 *    05/14/1998 - handle cleanup in case of interrupt
 *
 */


#include <lvm_user.h>


#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int c1 = 0;
   int i = 0;
   int p = 0;
   int lv_index = -1;
   int lv_num = 0;
   int np = 0;
   int opt_A = 1;
   int opt_f = 0;
   int opt_n = 0;
   int opt_t = 0;
   int opt_v = 0;
   int ret = 0;
   int src_pv_index = 0;
   int vg_active = FALSE;
   ulong pe_moved = 0;
   char lv_name_buf[NAME_LEN] = { 0, };
   char *buffer = NULL;
   char *cmd = NULL;
   char *lv_name = NULL;
#ifdef DEBUG
   char *options = "A:dfh?n:tv";
#else
   char *options = "A:fh?n:tv";
#endif
   char **pv_allowed = NULL;
   char *src_pv_name = NULL;
   char *vg_name = NULL;
   pv_t pv_this;
   pv_t *pv = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'f':
            if ( opt_f > 0) {
               fprintf ( stderr, "%s -- f option yet given\n\n", cmd);
               return 1;
            }
            opt_f++;
            break;

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Physical volume move\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-f]\n"
                     "\t[-h/?]\n"
                     "\t[-n LogicalVolumePath]\n"
                     "\t[-t]\n"
                     "\t[-v[v]]\n"
                     "\tSourcePhysicalVolumePath "
                     "[DestinationPhysicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'n':
            if ( opt_n > 0) {
               fprintf ( stderr, "%s -- n option yet given\n\n", cmd);
               return 1;
            }
            lv_name = optarg;
            opt_n++;
            break;

         case 't':
            if ( opt_t > 0) {
               fprintf ( stderr, "%s -- t option yet given\n\n", cmd);
               return 1;
            }
            opt_t++;
            break;

         case 'v':
            if ( opt_v > 1) {
               fprintf ( stderr, "%s -- v option yet given two times\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }


   if ( optind < argc && *argv[optind] == '-') {
      fprintf ( stderr, "%s -- invalid command line\n\n", cmd);
      return 1;
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a source physical volume path\n\n",
                        cmd);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking source physical volume\n", cmd);
   if ( pv_check_name ( src_pv_name = argv[optind]) < 0) {
      fprintf ( stderr, "%s -- %s is an invalid physical volume name\n\n",
                        cmd, src_pv_name);
      return 1;
   }
   optind++;

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;
   lvm_dont_interrupt ( 0);

   if ( opt_v > 0) printf ( "%s -- reading source physical volume data "
                            "from %s\n", cmd, src_pv_name);
   if ( ( ret = pv_read ( src_pv_name, &pv, NULL)) < 0) {
      if ( ret == -LVM_EPV_READ_OPEN) {
         fprintf ( stderr, "%s -- source physical volume %s doesn't exist\n\n",
                           cmd, src_pv_name);
      } else {
         fprintf ( stderr, "%s -- ERROR %d reading source physical "
                           "volume %s\n\n", cmd, ret, src_pv_name);
      }
      return 1;
   }
   memcpy ( &pv_this, pv, sizeof ( pv_this));
   vg_name = pv_this.vg_name;

   if ( opt_v > 0) printf ( "%s -- checking volume group name in "
                            "source physical volume\n", cmd);
   if ( vg_check_name ( vg_name) < 0) {
      fprintf ( stderr, "%s -- no valid volume group name %s\n\n",
                        cmd, vg_name);
      return 1;
   }

   /* does VG exist? */
   if ( opt_v > 0) printf ( "%s -- checking volume group existence\n", cmd);
   if ( lvm_tab_vg_check_exist ( vg_name, NULL) != TRUE) {
      fprintf ( stderr, "%s -- can't move physical extents: "
                        "%s doesn't exist\n\n", cmd, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- reading volume group data of %s\n",
                            cmd, vg_name);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d: couldn't get volume "
                        "group data of %s\n\n",
                        cmd, ret, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking volume group consistency of %s\n",
                            cmd, vg_name);
   if ( ( ret = vg_check_consistency_with_pv_and_lv ( vg)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d: volume group %s is inconsistent\n\n",
                        cmd, ret, vg_name);
      return 1;
   }

   /* get index of source physical volume */
   if ( ( src_pv_index = pv_get_index_by_name ( vg, src_pv_name)) < 0) {
      fprintf ( stderr, "%s -- couldn't find %s in %s\n\n",
                        cmd, src_pv_name, vg_name);
      return 1;
   }

   if ( vg->pv[src_pv_index]->pe_allocated == 0) {
      fprintf ( stderr, "%s -- no logical volumes on empty %s\n\n",
                        cmd, src_pv_name);
      return 1;
   }

   if ( vg->pv_cur == 1) {
      fprintf ( stderr, "%s -- %s only has 1 physical volume\n\n",
                        cmd, vg_name);
      return 1;
   }

   if ( opt_n > 0) {
      if ( opt_v > 0) printf ( "%s -- checking logical volume %s\n",
                               cmd, lv_name);
      if ( strchr ( lv_name, '/') == NULL) {
         sprintf ( lv_name_buf, "/dev/%s/%s%c", vg_name, lv_name, 0);
         lv_name = lv_name_buf;
      }
      if ( lv_check_name ( lv_name) < 0) {
         fprintf ( stderr, "%s -- %s is an invalid logical volume name\n\n",
                           cmd, lv_name);
         return 1;
      }
      if ( ( lv_num = lv_number_from_name_in_vg ( lv_name, vg)) < 0) {
         fprintf ( stderr, "%s -- logical volume %s is not in volume "
                           "group %s\n\n",
                           cmd, lv_name, vg_name);
         return 1;
      }
      lv_num++;
      if ( lv_check_on_pv ( vg->pv[src_pv_index], lv_num) == FALSE) {
         fprintf ( stderr, "%s -- logical volume %s isn't on %s\n\n",
                           cmd, lv_name, src_pv_name);
         return 1;
      }
   }

   if ( opt_v > 0) printf ( "%s -- building list of possible destination "
                            "physical volumes\n", cmd);
   if ( ( pv_allowed = malloc ( vg->pv_cur * sizeof ( char*))) == NULL) {
      fprintf ( stderr, "%s -- malloc error in %s [line: %d]\n\n",
                        cmd, __FILE__, __LINE__);
      return 1;
   }

   /* destination physical volumes in command line ? */
   np = 0;
   if ( optind < argc) {
      if ( opt_v > 0) printf ( "%s -- checking destination physical "
                               "volume names in command line\n", cmd);
      for ( i = optind; i < argc; i++) {
         if ( pv_check_name ( argv[i]) < 0) {
            fprintf ( stderr, "%s -- invalid physical volume path %s\n\n",
                              cmd, argv[i]);
            return 1;
         }
         if ( strcmp ( src_pv_name, argv[i]) == 0) {
            fprintf ( stderr, "%s -- source physical volume %s can't be "
                              "a destination\n\n",
                              cmd, argv[i]);
            return 1;
         }
         if ( ( ret = pv_read ( argv[i], &pv, NULL)) < 0) {
            if ( ret == -LVM_EPV_READ_OPEN) {
               fprintf ( stderr, "%s -- destination physical volume "
                                 "%s doesn't exist\n\n",
                                 cmd, argv[i]);
            } else {
               fprintf ( stderr, "%s -- ERROR %d reading destination physical "
                                 "volume %s\n\n", cmd, ret, argv[i]);
            }
            return 1;
         }
         if ( ( ret = pv_check_consistency ( pv)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d:  physical volume %s "
                              "is inconsistent\n\n",
                              cmd, ret, argv[i]);
            return 1;
         }
         if ( pv_check_in_vg ( vg, pv->pv_name) == FALSE) {
            fprintf ( stderr, "%s -- %s doesn't belong to volume "
                              "group of %s\n\n",
                              cmd, argv[i], vg_name);
            return 1;
         }
         pv_allowed[np++] = argv[i];
      }
   } else {
      for ( p = 0; p < vg->pv_max && vg->pv[p] != NULL; p++) {
         if ( p != src_pv_index) pv_allowed[np++] = vg->pv[p]->pv_name;
      }
   }
   pv_allowed[np] = NULL;


   if ( opt_n > 0) {
      if ( ( lv_index = lv_get_index_by_name ( vg, lv_name)) < 0) {
         fprintf ( stderr, "%s -- couldn't find %s in %s\n\n",
                           cmd, lv_name, vg_name);
         return 1;
      }
   }

   /* get transfer buffer memory */
   if ( ( buffer = malloc ( vg->pe_size * SECTOR_SIZE)) == NULL) {
      fprintf ( stderr, "%s -- malloc error in %s [line: %d]\n\n",
                        cmd, __FILE__, __LINE__);
      return 1;
   }
   
   if ( opt_v > 0) printf ( "%s -- checking volume group activity\n", cmd);
   printf ( "%s -- moving physical extents in ", cmd);
   if ( vg_check_active ( vg_name) == TRUE) {
      vg_active = TRUE;
   } else {
      vg_active = FALSE;
      printf ( "in");
   }
   printf ( "active volume group %s\n", vg_name);

   if ( opt_f == 0) {
      printf ( "%s -- WARNING: moving of active logical volumes "
               "may cause data loss!\n", cmd);
      c = 0;
      while ( c != 'y' && c != 'n') {
         if ( c == '\n' || c == 0)
            printf ( "%s -- do you want to continue? [y/n] ", cmd);
         c = tolower ( getchar ());
      }
      c1 = c;
      while ( c != '\n') c = tolower ( getchar ());
      if ( c1 == 'n') goto pvmove_end;
   }

   /* let's rock and role */
   if ( opt_v > 0) {
      printf ( "%s -- starting to move ", cmd);
      if ( lv_index > -1) {
         if ( vg->lv[lv_index]->lv_stripes > 1) printf ( "striped");
         else                                   printf ( "linear");
         printf ( " logical volume %s\n", vg->lv[lv_index]->lv_name);
      } else {
         printf ( "extends away from %s\n", src_pv_name);
      }
   }
  
   pe_moved = 0;
   ret = 1;
   while ( ret > 0) {
      if ( ( ret = pv_move_pes ( vg, buffer, pv_allowed,
                                 src_pv_index, lv_index,
                                 cmd, opt_v, opt_t)) < 0) {
         if ( ret == -LVM_EPV_MOVE_PES_NO_SPACE) {
            fprintf ( stderr, "%s -- not enough free/allocatable "
                              "physical extends\n\n", cmd);
         } else if ( ret == -LVM_EPV_MOVE_PES_NO_PES) {
            ret = 0;
         } else {
            fprintf ( stderr, "%s -- ERROR %d moving physical extends\n\n",
                              cmd, ret);
         }
         return 1;
      }
      pe_moved += ret;
      if ( lv_index > -1 || vg->pv[src_pv_index]->pe_allocated == 0) break;
   }

   free ( buffer);

   if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
   if ( opt_t == 0) {
      if ( vg_cfgbackup ( vg_name, LVMTAB_DIR, cmd, opt_v, vg) == 0 &&
           opt_A > 0) {
         printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
         vg_cfgbackup ( vg_name, VG_BACKUP_DIR, cmd, opt_v, vg);
      }
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   printf ( "%s -- %lu extend", cmd, pe_moved);
   if ( pe_moved > 1) printf ( "s");
   printf ( " of physical volume %s successfully moved\n", src_pv_name);
   if ( opt_t > 0) printf ( "%s -- this has been a test run WITHOUT "
                            "any real change\n", cmd);

pvmove_end:
   printf ( "\n");

   return 0;
}
