/*
 * Author: Heinz Mauelshagen, Germany
 *
 * Oktober 1997
 * January 1998
 * May     1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    01/11/1998 - added creation of volume group directory and special files
 *    05/03/1998 - enhanced error checking with lvm_tab_vg_insert()
 *                 and lvm_tab_vg_remove()
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif


int main ( int argc, char **argv)
{
   int c = 0;
   int i = 0;
   int opt_v = 0;
   int p = 0;
   int ret = 0;
#ifdef DEBUG
   char *options = "dh?v";
#else
   char *options = "h?v";
#endif
   char *cmd = NULL;
   char command[NAME_LEN] = { 0, };
   char **vg_name_ptr = NULL;
   pv_t **pv = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group scan\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v]]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   LVM_LOCK ( 1);
   LVM_CHECK_IOP;

   printf ( "%s -- reading all physical volumes (this may take a while...)\n",
            cmd);
   if ( ( ret = pv_read_all_pv ( &pv, TRUE)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d reading all physical volumes\n",
                        cmd, ret);
      return 1;
   }

   lvm_dont_interrupt ( 0);

   if ( opt_v > 0) printf ( "%s -- removing %s and %s\n",
                            cmd, LVMTAB, LVMTAB_DIR);
   sprintf ( command, "rm -fr %s %s%c", LVMTAB, LVMTAB_DIR, 0);
   system ( command);

   if ( opt_v > 0) printf ( "%s -- inserting found volume group "
                            "names into %s\n", cmd, LVMTAB);
   for ( p = 0; pv != NULL && pv[p] != NULL; p++) {
      if ( system_id_check_exported ( pv[p]->system_id) == FALSE &&
           pv_check_consistency ( pv[p]) == 0) {
         if ( pv_check_new ( pv[p]) == FALSE) {
            if ( opt_v > 0) printf ( "%s -- inserting %s into %s\n",
                                     cmd, pv[p]->vg_name, LVMTAB);
            if ( ( ret = lvm_tab_vg_insert ( pv[p]->vg_name)) < 0 &&
                 ret != -LVM_ELVM_TAB_VG_INSERT_VG_EXISTS) {
               fprintf ( stderr, "%s -- ERROR %d inserting %s into %s\n",
                                 cmd, ret, pv[p]->vg_name, LVMTAB);
               return 1;
            }
         }
      } else {
         fprintf ( stderr, "%s -- invalid device %s found\n",
                           cmd, pv[p]->pv_name);
      }
   }

   if ( ret == -LVM_ELVM_TAB_VG_INSERT_VG_EXISTS) ret = 0;

   if ( ( vg_name_ptr = lvm_tab_vg_check_exist_all_vg ()) != NULL) {
      for ( i = 0; vg_name_ptr[i] != NULL; i++) {
         printf ( "%s -- found volume group %s\n", cmd, vg_name_ptr[i]);
         if ( opt_v > 0) printf ( "%s -- backing up %s\n", cmd, vg_name_ptr[i]);
         if ( ( ret = vg_cfgbackup ( vg_name_ptr[i], LVMTAB_DIR, cmd,
                                     opt_v, NULL)) < 0) {
            fprintf ( stderr, "%s -- ERROR: unable to do a backup of %s\n",
                              cmd, vg_name_ptr[i]);
            if ( ( ret = lvm_tab_vg_remove ( vg_name_ptr[i])) < 0) {
               fprintf ( stderr, "%s -- ERROR %d removing %s from %s\n",
                                 cmd, ret, vg_name_ptr[i], LVMTAB);
            }
            continue;
         }
         if ( opt_v > 0) printf ( "%s -- reading volume group data "
                                  "from lvmtab\n", cmd);
         if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name_ptr[i],
                                                       &vg)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d reading volume group data from "
                              "lvmtab\n", cmd, ret);
            continue;
         }
         if ( opt_v > 0) printf ( "%s -- creating volume group directory "
                                  "and special files for %s\n",
                                  cmd, vg_name_ptr[i]);
         if ( ( ret = vg_create_dir_and_group_and_nodes ( vg, cmd,
                                                          opt_v)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d creating volume group directory "
                              "and special files\n", cmd, ret);
            continue;
         }
      }
   } else ret = 1;

   lvm_interrupt ();
   LVM_UNLOCK ( 1);

   if ( ret == 0) {
      printf ( "%s -- %s and %s successfully created\n",
               cmd, LVMTAB, LVMTAB_DIR);
   } else {
      fprintf ( stderr, "%s -- ERROR %d creating %s and %s\n",
                        cmd, ret, LVMTAB, LVMTAB_DIR);
      ret = 1;
   }

   printf ( "%s -- You may not have an actual backup of your "
            "volume group(s)\n\n", cmd);

   return ret;
}
