/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March-May,October 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/11/1997 - used new pv_read_namelist ()
 *    05/09/1998 - extended V option output by pv_dev
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int i = 0;
   int l = 0;
   int opt = 0;
   int opt_a = 0;
   int opt_E = 0;
   int opt_L = 0;
   int opt_N = 0;
   int opt_P = 0;
   int opt_v = 0;
   int opt_V = 0;
   int pv_handle = -1;
   int ret = 0;
#ifdef DEBUG
   char *options = "adEh?LNPvV";
#else
   char *options = "aEh?LNPvV";
#endif
   char *pv_name = NULL;
   char *pv_namelist = NULL;
   pv_t pv;
   vg_t vg;
   lv_t lv;
   disk_pe_t *pe = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'a':
            if ( opt_a > 0) {
               fprintf ( stderr, "%s -- a option yet given\n\n", cmd);
               return 1;
            }
            opt_a++;
            opt++;
            opt_E++; opt_L++; opt_N++; opt_P++; opt_V++;
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            opt++;
            break;
#endif

         case 'E':
            if ( opt_E > 0) {
               fprintf ( stderr, "%s -- E option yet given\n\n", cmd);
               return 1;
            }
            opt_E++;
            opt++;
            break;

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Physical Volume Debug Data Display\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-a] # show all structures\n"
#ifdef DEBUG
                     "\t[-d] # debug\n"
#endif
                     "\t[-E] # physical extend structs\n"
                     "\t[-L] # logical volume structs\n"
                     "\t[-N] # physical volume name list\n"
                     "\t[-P[P]] # physical volume struct\n"
                     "\t[-V] # volume group struct\n"
                     "\t[-h/?]\n"
                     "\t[-v[v]]\n"
                     "\tPhysicalVolumePath [PhysicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            opt++;
            return 0;
            break;

         case 'L':
            if ( opt_L > 0) {
               fprintf ( stderr, "%s -- L option yet given\n\n", cmd);
               return 1;
            }
            opt_L++;
            opt++;
            break;

         case 'N':
            if ( opt_N > 0) {
               fprintf ( stderr, "%s -- N option yet given\n\n", cmd);
               return 1;
            }
            opt_N++;
            opt++;
            break;

         case 'P':
            if ( opt_P > 1) {
               fprintf ( stderr, "%s -- P option yet given two times\n\n", cmd);
               return 1;
            }
            opt_P++;
            opt++;
            break;

         case 'v':
            if ( opt_v > 1) {
               fprintf ( stderr, "%s -- v option yet given two times\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         case 'V':
            if ( opt_V > 0) {
               fprintf ( stderr, "%s -- V option yet given\n\n", cmd);
               return 1;
            }
            opt_V++;
            opt++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a physical volume path\n\n", cmd);
      return 1;
   }

   if ( opt == 0) {
      fprintf ( stderr, "%s -- please enter an option", cmd);
      if ( optind == argc) printf ( " and a physical volume path");
      printf ( "\n\n");
      return 1;
   }

   opt_P++;
   if ( opt_V > 0 || opt_L > 0 || opt_E > 0 || opt_N > 0) opt_V++;

   LVM_LOCK ( 1);

   for ( ; optind < argc; optind++) {
      pv_name = argv[optind];

      if ( pe != NULL) {
         free ( pe);
         pe = NULL;
      }

      if ( pv_handle != -1) close ( pv_handle);

      if ( ( pv_handle = open ( pv_name, O_RDONLY)) == -1) {
         fprintf ( stderr, "%s -- can't open %s readonly\n", cmd, pv_name);
         continue;
      }

      /* physical volume */
      if ( opt_P > 0) {
         if ( opt_v > 1) printf ( "%s -- reading physical volume data "
                                  "from disk\n", cmd);
         if ( read ( pv_handle, &pv, sizeof ( pv)) != sizeof ( pv)) {
            fprintf ( stderr,
                      "%s -- can't read physical volume data from %s\n",
                      cmd, pv_name);
            continue;
         }
         if ( opt_P > 1) {
            pv_show ( &pv);
            if ( opt_P > 2) {
               printf ( "pv_dev                   %d:%d\n",
                        MAJOR ( pv.pv_dev), MINOR ( pv.pv_dev));
               printf ( "system_id                %s\n", pv.system_id);
               printf ( "pv_on_disk.base          %u\n", pv.pv_on_disk.base);
               printf ( "pv_on_disk.size          %u\n", pv.pv_on_disk.size);
               printf ( "vg_on_disk.base          %u\n", pv.vg_on_disk.base);
               printf ( "vg_on_disk.size          %u\n", pv.vg_on_disk.size);
               printf ( "pv_namelist_on_disk.base %u\n",
                        pv.pv_namelist_on_disk.base);
               printf ( "pv_namelist_on_disk.size %u\n",
                        pv.pv_namelist_on_disk.size);
               printf ( "lv_on_disk.base          %u\n", pv.lv_on_disk.base);
               printf ( "lv_on_disk.size          %u\n", pv.lv_on_disk.size);
               printf ( "pe_on_disk.base          %u\n", pv.pe_on_disk.base);
               printf ( "pe_on_disk.size          %u\n\n", pv.pe_on_disk.size);
            }
            putchar ( '\n');
         }
      }

      /* volume group */
      if ( opt_V > 0) {
         if ( lseek ( pv_handle, pv.vg_on_disk.base, SEEK_SET) !=
              pv.vg_on_disk.base) {
            fprintf ( stderr, "%s -- can't seek to volume group struct on %s\n",
                      cmd, pv_name);
            continue;
         }

         if ( opt_v > 1) printf ( "%s -- reading volume group data "
                                  "from disk\n", cmd);
         if ( read ( pv_handle, &vg, sizeof ( vg)) != sizeof ( vg)) {
            fprintf ( stderr,
                      "%s -- can't read volume group data from %s\n",
                      cmd, pv_name);
            continue;
         }
         vg_show ( &vg);
      }

      /* logical volumes */
      if ( opt_L > 0) {
         printf ( "\n--- List of logical volumes ---\n\n");
         for ( l = 0; l < vg.lv_max; l++) {
            if ( opt_v > 1) printf ( "%s -- seeking to logical volume "
                                     "struct #%d on disk\n", cmd, l);
            if ( lseek ( pv_handle, LVM_LV_DISK_OFFSET ( &pv, l), SEEK_SET)
                 != LVM_LV_DISK_OFFSET ( &pv, l)) {
               fprintf ( stderr,
                         "%s -- can't seek to logical volume struct "
                         "#%d on %s\n", cmd, l, pv_name);
               continue;
            }

            if ( opt_v > 1) printf ( "%s -- reading logical volume "
                                     "struct #%d from disk\n", cmd, l);
            if ( read ( pv_handle, &lv, sizeof ( lv)) != sizeof ( lv)) {
               fprintf ( stderr,
                         "%s -- can't read logical volume struct #%d from %s\n",
                         cmd, l, pv_name);
               continue;
            }
            if ( opt_v > 1) printf ( "%s -- checking consistency of "
                                     "logical volume #%d\n", cmd, l);
            if ( strlen ( lv.vg_name) == 0 && strlen ( lv.lv_name) == 0) {
               printf ( "%s -- logical volume struct at offset %3d "
                        "is empty\n", cmd, l);
               continue;
            }
            else if ( ( ret = lv_check_consistency ( &lv)) < 0) {
               fprintf ( stderr,
                         "%s -- logical volume struct at offset %3d "
                         "is inconsistent\n", cmd, l);
               continue;
            }

            if ( opt_v > 1) printf ( "\n%s -- logical volume struct #%d\n",
                                     cmd, l);
            if ( opt_v > 0) {
               lv_show ( &lv);
               printf ( "\n");
            } else {
               printf ( "%s -- logical volume %s at offset %3d\n",
                        cmd, lv.lv_name, l);
            }
         }
      }

      /* physical extends */
      if ( opt_E > 0) {
         if ( opt_v > 1) printf ( "%s -- seeking to physical extend structs "
                                  "on disk\n", cmd);
         if ( lseek ( pv_handle, pv.pe_on_disk.base, SEEK_SET)
              != pv.pe_on_disk.base) {
            fprintf ( stderr,
                      "%s -- can't seek to physical extend structs on %s\n",
                      cmd, pv_name);
            continue;
         }
         if ( ( pe = malloc ( pv.pe_total * sizeof ( disk_pe_t))) == NULL) {
            fprintf ( stderr, "%s -- malloc error in file %s [line %d]\n\n",
                              cmd, __FILE__, __LINE__);
            continue;
         }

         printf ( "\n--- List of physical extends ---\n\n");
         if ( opt_v > 1) printf ( "%s -- reading physical extend structs "
                                  "from disk", cmd);
         if ( read ( pv_handle, pe, pv.pe_total * sizeof ( disk_pe_t)) !=
              pv.pe_total * sizeof ( disk_pe_t)) {
            fprintf ( stderr,
                      "%s -- can't read physical extend structs from %s\n",
                      cmd, pv_name);
            continue;
         }
         for ( i = 0; i < pv.pe_total; i++) {
            printf ( "PE: %05d  LV: ", i);
            if ( pe[i].lv_num == 0) printf ( "---");
            else                    printf ( "%03d", pe[i].lv_num);
            printf ( "  LE: ");
            if ( pe[i].lv_num == 0) {
               if ( pe[i].le_num == 0) printf ( "-----\n");
               else                    printf ( "error\n");
            } else printf ( "%05d\n", pe[i].le_num);
         }
      }

      /* physical volume name list */
      if ( opt_N > 0) {
         vg.pv[0] = &pv;
         if ( ( ret = pv_read_namelist ( pv_name, &vg, &pv_namelist)) < 0) {
            fprintf ( stderr,
                      "%s -- ERROR %d reading physical volume name list "
                      "from %s\n",
                      cmd, ret, pv_name);
            continue;
         }

         printf ( "--- List of physical volumes ---\n\n");
         for ( i = 0; i < vg.pv_cur; i++) {
            printf ( "%03d: ", i);
            if ( pv_check_name ( &pv_namelist[i*NAME_LEN]) == 0)
               printf ( "%s\n", &pv_namelist[i*NAME_LEN]);
            else
               printf ( "empty\n");
         }
      }
   }

   LVM_UNLOCK ( 1);

   printf ( "\n");
   return 0;
}
