/*
 * kernel/lvm.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * February-November 1997
 * April-May,July-August,November 1998
 * January 1999
 *
 * lvm is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * lvm is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/09/1997 - added chr ioctls VG_STATUS_GET_COUNT
 *                 and VG_STATUS_GET_NAMELIST
 *    01/18/1998 - change lvm_chr_open/close lock handling
 *    04/30/1998 - changed LV_STATUS ioctl to LV_STATUS_BYNAME and
 *               - added   LV_STATUS_BYINDEX ioctl
 *               - used lvm_status_byname_req_t and
 *                      lvm_status_byindex_req_t vars
 *    05/04/1998 - added multiple device support
 *    05/08/1998 - added support to set/clear extendable flag in volume group
 *    05/09/1998 - changed output of lvm_proc_get_info() because of
 *                 support for free (eg. longer) logical volume names
 *    05/12/1998 - added spin_locks (thanks to Pascal van Dam
 *                 <pascal@ramoth.xs4all.nl>)
 *    05/25/1998 - fixed handling of locked PEs in lvm_map() and lvm_chr_ioctl()
 *    05/26/1998 - reactivated verify_area by access_ok
 *    06/07/1998 - used vmalloc/vfree instead of kmalloc/kfree to go
 *                 beyond 128/256 KB max allocation limit per call
 *               - #ifdef blocked spin_lock calls to avoid compile errors
 *                 with 2.0.x
 *    06/11/1998 - another enhancement to spinlock code in lvm_chr_open()
 *                 and use of LVM_VERSION_CODE instead of my own macros
 *                 (thanks to  Michael Marxmeier <mike@msede.com>)
 *    07/07/1998 - added statistics in lvm_map()
 *    07/08/1998 - saved statistics in do_lv_extend_reduce()
 *    07/25/1998 - used __initfunc macro
 *    08/02/1998 - changes for official char/block major numbers
 *    08/07/1998 - avoided init_module() and cleanup_module() to be static
 *    08/30/1998 - changed VG lv_open counter from sum of LV lv_open counters
 *                 to sum of LVs open (no matter how often each is)
 *    09/01/1998 - fixed lvm_gendisk.part[] index error
 *    09/07/1998 - added copying of lv_current_pe-array
 *                 in LV_STATUS_BYINDEX ioctl
 *    11/17/1998 - added KERN_* levels to printk
 *    01/13/1999 - fixed LV index bug in do_lv_create() which hit lvrename
 *
 */

/*
 * TODO
 *
 *   - add code for LV specific read_ahead sectors?
 *   - implement special handling of unavailable physical volumes
 *   - implement RAID1/5,10/50
 *
 */

char *lvm_version = "LVM 0.5alpha by Heinz Mauelshagen   01/30/1999\n";
char *lvm_short_version = "0.5alpha";

#include <linux/config.h>

#include <linux/version.h>
/* for 2.0.x series */
#ifndef KERNEL_VERSION
#  define KERNEL_VERSION(a,b,c) (((a) << 16) + ((b) << 8) + (c))
#endif

#ifdef MODVERSIONS
#  undef MODULE
#  define MODULE
#  if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
#    include <linux/modversions.h>
#  endif
#endif

#ifdef MODULE
#  include <linux/module.h>
#endif

#include <linux/kernel.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION ( 2, 1 ,0)
#  include <linux/mm.h>
#else
#  include <linux/vmalloc.h>
#  include <linux/init.h>
#endif
#include <linux/hdreg.h>
#include <linux/stat.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/blkdev.h>
#include <linux/genhd.h>
#include <asm/ioctl.h>
#include <asm/segment.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
#  include <asm/uaccess.h>
#  define   memcpy_fromfs     copy_from_user
#  define   memcpy_tofs    copy_to_user
#  define   proc_register_dynamic   proc_register
#  define   verify_area    !access_ok
#endif

#ifdef CONFIG_KERNELD
#include <linux/kerneld.h>
#endif
#include <linux/errno.h>

#include <linux/lvm.h>

#ifdef MODULE
#   define MODULE_NAME  "lvm"
#endif


/* Internal function prototypes */
#ifdef MODULE
int init_module ( void);
void cleanup_module ( void);
#else
extern int lvm_init ( void);
#endif

#define	suser()	( current->uid == 0 && current->euid == 0)

static void lvm_dummy_device_request ( void);

static int  lvm_blk_ioctl ( struct inode *, struct file *, unsigned int,
                            unsigned long);
static int  lvm_blk_open  ( struct inode *, struct file *);

static int  lvm_chr_open  ( struct inode *, struct file *);

#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1, 0)
static int lvm_chr_close ( struct inode *, struct file *);
static int lvm_blk_close ( struct inode *, struct file *);
#else
static void lvm_chr_close ( struct inode *, struct file *);
static void lvm_blk_close ( struct inode *, struct file *);
#endif

static int  lvm_chr_ioctl ( struct inode *, struct file *, unsigned int,
                            unsigned long);

#if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS
static int lvm_proc_get_info ( char *, char **, off_t, int, int);
#endif

static void lvm_init_vars ( void);

extern int  (*lvm_map_ptr) ( int, kdev_t *, unsigned long *,
                             unsigned long, int);
static int  lvm_map ( int, kdev_t *, unsigned long *, unsigned long, int);

static int do_vg_remove ( int);

static int do_lv_create ( int, char *, lv_t *, struct inode *);
static int do_lv_remove ( int, char *);
static int do_lv_extend_reduce ( int, char *, lv_t *, struct inode *);

static void lvm_geninit ( struct gendisk *);
static struct inode *lvm_get_inode ( int);

inline int  lvm_strlen ( char *);
inline void lvm_memcpy ( char *, char *, int);
inline int  lvm_strcmp ( char *, char *);
inline char *lvm_strrchr ( char *, char c);
/* END Internal function prototypes */


/* Volume group structure pointers */
static vg_t *vg[ABS_MAX_VG];
static pv_t *pvp  = NULL;
static lv_t *lvp  = NULL;
static pe_t *pep  = NULL;
static pe_t *pep1 = NULL;

/* association array LV -> VG to optimize minor number usage */
/* static int lvs[MAX_LV]; */

/* Request structures (lvm_chr_ioctl) */
static pv_change_req_t pv_change_req;
static pe_lock_req_t   pe_lock_req;
static le_remap_req_t  le_remap_req;

static char pv_name[NAME_LEN];
static char lv_name[NAME_LEN];
#if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS
static char dummy_buf[160]; /* sized for 2 lines in lvm_proc_get_info */
#endif

static uint lv_open = 0;

static const char* const lvm_name = "lvm";
#ifdef MODULE
static int lvm_reset_spindown = 0;
#endif
static int lock = 0;
static int loadtime = 0;
static uint vg_count = 0;
static long lvm_chr_open_count = 0;
static ushort lvm_iop_version = LVM_IOP_VERSION;
struct wait_queue *lvm_wait = NULL;
struct wait_queue *lvm_map_wait = NULL;

#if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS
static struct proc_dir_entry lvm_proc_entry = {
   0, 3, "lvm", S_IFREG | S_IRUGO,
   1, 0, 0, 0,
   NULL,
   lvm_proc_get_info,
   NULL, NULL, NULL, NULL, NULL,
};
#endif

static struct file_operations lvm_chr_fops = {
   NULL,    /* No lseek              */
   NULL,    /* No read               */
   NULL,    /* No write              */
   NULL,    /* No readdir            */
   NULL,    /* No select             */
   lvm_chr_ioctl,
   NULL,    /* No mmap               */
   lvm_chr_open,
#if LINUX_VERSION_CODE < KERNEL_VERSION ( 2, 1, 119)
   lvm_chr_close,
#else
   NULL,    /* No flush              */
   lvm_chr_close,
#endif
   NULL,    /* No fsync              */
   NULL,    /* No fasync             */
   NULL,    /* No check_media_change */
   NULL,    /* No revalidate         */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1, 0)
   NULL,    /* No lock               */
#endif
};

static struct file_operations lvm_blk_fops = {
   NULL,           /* No lseek             */
   block_read,     /* generic read          */
   block_write,    /* genric write          */
   NULL,           /* No readdir            */
   NULL,           /* No select             */
   lvm_blk_ioctl,
   NULL,           /* No mmap               */
   lvm_blk_open,
#if LINUX_VERSION_CODE < KERNEL_VERSION ( 2, 1, 118)
   lvm_blk_close,
#else
   NULL,           /* No flush              */
   lvm_blk_close,
#endif
   NULL,           /* No fsync              */
   NULL,           /* No fasync             */
   NULL,           /* No check_media_change */
   NULL,           /* No revalidate         */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1, 0)
   NULL,           /* No lock               */
#endif
};


/* gendisk structures */
static struct hd_struct lvm_hd_struct[MAX_VG*(MAX_LV+1)];
static int lvm_blocksizes[MAX_VG*(MAX_LV+1)] = { 0, };
static int lvm_size[MAX_VG*(MAX_LV+1)] = { 0, };
static struct gendisk lvm_gendisk = {
   LVM_BLOCK_MAJOR,
   "lvm",
   0,
   1,
   MAX_VG * ( MAX_LV + 1),
   lvm_geninit,
   lvm_hd_struct,
   lvm_size,
   MAX_VG * ( MAX_LV + 1),
   NULL,
   NULL
};


#ifdef MODULE
/*
 * Module initialization...
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
__initfunc ( int init_module ( void))
#else
int init_module ( void)
#endif
#else
/*
 * Driver initialization...
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
__initfunc ( extern int lvm_init ( void))
#else
extern int lvm_init ( void)
#endif
#endif /* #ifdef MODULE */
{
   lvm_init_vars ();

   if ( register_chrdev ( LVM_CHAR_MAJOR, lvm_name, &lvm_chr_fops) < 0) {
      printk ( KERN_ERR "%s -- register_chrdev failed\n", lvm_name);
      return -EIO;
   }
   if ( register_blkdev ( LVM_BLOCK_MAJOR, lvm_name, &lvm_blk_fops) < 0) {
      printk ( "%s -- register_blkdev failed\n", lvm_name);
      if ( unregister_chrdev ( LVM_CHAR_MAJOR, lvm_name) < 0)
         printk ( KERN_ERR "%s -- unregister_chrdev failed\n", lvm_name);
      return -EIO;
   }

#if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS
    proc_register_dynamic ( &proc_root, &lvm_proc_entry);
#endif

#ifdef DEBUG
   printk ( KERN_INFO "%s -- registered\n", lvm_name);
#endif

   blk_dev[LVM_BLOCK_MAJOR].request_fn = lvm_dummy_device_request;
   blk_dev[LVM_BLOCK_MAJOR].current_request = NULL;
   /* read ahead per LV later? */
   read_ahead[LVM_BLOCK_MAJOR] = LVM_MAX_READ_AHEAD;

   /* insert our gendisk */
   lvm_geninit ( &lvm_gendisk);
   lvm_gendisk.next = gendisk_head;
   gendisk_head = &lvm_gendisk;

   /* reference from ll_rw_blk */
   lvm_map_ptr = lvm_map;

   {
#ifdef MODULE
      char *str = "Module";
#else
      char *str = "Driver";
#endif
      printk ( KERN_INFO
               "%s%s -- %s successfully initialized\n",
               lvm_version, lvm_name, str);
   }

   return 0;
}


#ifdef MODULE
/*
 * Module cleanup...
 */
void cleanup_module ( void) {
   /* DANGER: another one might be in chain beyond us */
   if ( gendisk_head != &lvm_gendisk) {
      printk ( KERN_ERR "%s -- ERROR: another one is in chain "
               "beyond me --> can't cleanup\n",
               lvm_name);
      MOD_INC_USE_COUNT;
      return;
   }

   /* delete our gendisk from chain */
   gendisk_head = lvm_gendisk.next;
   blk_size[LVM_BLOCK_MAJOR] = NULL;
   blksize_size[LVM_BLOCK_MAJOR] = NULL;

#if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS
   proc_unregister ( &proc_root, lvm_proc_entry.low_ino);
#endif

   if ( unregister_chrdev ( LVM_CHAR_MAJOR, lvm_name) < 0) {
      printk ( KERN_ERR "%s -- unregister_chrdev failed\n", lvm_name);
   }
   if ( unregister_blkdev ( LVM_BLOCK_MAJOR, lvm_name) < 0) {
      printk ( KERN_ERR "%s -- unregister_blkdev failed\n", lvm_name);
   }

   /* reference from ll_rw_blk */
   lvm_map_ptr = NULL;

   blk_dev[LVM_BLOCK_MAJOR].request_fn = NULL;
   blk_dev[LVM_BLOCK_MAJOR].current_request = NULL;

   printk ( KERN_INFO "%s -- Module succesfully deactivated\n", lvm_name);

   return;
} /* void cleanup_module () */
#endif /* #ifdef MODULE */


/*
 * support function to initialize lvm variables
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
__initfunc ( static void lvm_init_vars ( void))
#else
static void lvm_init_vars ( void)
#endif
{
   int v;

   loadtime = CURRENT_TIME;

   pe_lock_req.lock = UNLOCK_PE;
   pe_lock_req.data.lv_dev = \
   pe_lock_req.data.pv_dev = \
   pe_lock_req.data.pv_offset = 0;

   /* Initialize VG pointers */
   for ( v = 0; v < MAX_VG; v++) vg[v] = NULL;

   /* Initialize LV -> VG association */
   /* for ( v = 0; v < MAX_LV; v++) lvs[v] = 0; */

   return;
}


/********************************************************************
 *
 * Character device functions
 *
 ********************************************************************/

/*
 * character device open routine
 * Only one process can open /dev/lvm at one time, others will block.
 */
static int lvm_chr_open ( struct inode *inode,
                          struct file *file) {
   int minor = MINOR ( inode->i_rdev);
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
   ulong flags;
   spinlock_t lvm_lock = SPIN_LOCK_UNLOCKED;
   
   /* for UP gcc will complain about unused variables */
   ( void) flags;
   ( void) lvm_lock;
#endif

#ifdef DEBUG
   printk ( KERN_DEBUG
            "%s -- lvm_chr_open MINOR: %d  VG#: %d  LV#: %d  "
            "mode: 0x%X  lock: %d\n",
            lvm_name, minor, VG(minor), LV(minor), file->f_mode, lock);
#endif

   /* super user validation */
   if ( ! suser()) return -EACCES;

   /* Group special file open */
   if ( VG(minor) > MAX_VG || LV(minor) != 0) return -ENXIO;

#ifdef MODULE
   MOD_INC_USE_COUNT;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
try_again: ;
   spin_lock ( &lvm_lock);
   if( lock && lock != current->pid ) {
      printk ( KERN_INFO "lvm_chr_open: %s is locked by pid %d ...\n", lvm_name, lock);
      spin_unlock ( &lvm_lock);
      interruptible_sleep_on ( &lvm_wait);
      if ( current->sigpending != 0) {
#ifdef MODULE
         MOD_DEC_USE_COUNT;
#endif
         return -EINTR;
      }
#ifdef MODULE
      if ( lvm_reset_spindown > 0) {
         if ( GET_USE_COUNT ( &__this_module) < 2) lvm_reset_spindown = 0;
         MOD_DEC_USE_COUNT;
         return -EACCES;
      }
#endif
      goto try_again;
   }
   lock = current->pid;
   spin_unlock (&lvm_lock);
#else /* #if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0) */
   while( lock && lock != current->pid ) {
      printk (  KERN_INFO "lvm_chr_open: %s is locked by pid %d ...\n", lvm_name, lock);
      interruptible_sleep_on ( &lvm_wait);
      if ( current->signal &~ current->blocked) {
         MOD_DEC_USE_COUNT;
         return -EINTR;
      }
#ifdef MODULE
      if ( lvm_reset_spindown > 0) {
         if ( mod_use_count_ < 2) lvm_reset_spindown = 0;
         MOD_DEC_USE_COUNT;
         return -EACCES;
      }
#endif
   }
   lock = current->pid;
#endif /* #if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0) */

   lvm_chr_open_count++;
   return 0;
}


/*
 * character device i/o-control routine
 */
static int lvm_chr_ioctl ( struct inode *inode, struct file *file,
                           uint command, ulong a) {
   int minor = MINOR ( inode->i_rdev);
   int error;
   int extendable;
#if defined(VERIFY_READ) || defined(VERIFY_WRITE)
   int   ret;
#endif
   ulong  l, le, p, v;
   ulong size;
   void  *arg = ( void*) a;
   lv_t  lv;
#ifdef LVM_GET_INODE
   struct inode *inode_sav;
#endif
   lv_status_byname_req_t lv_status_byname_req;
   lv_status_byindex_req_t lv_status_byindex_req;


#ifdef DEBUG_IOCTL
   printk ( KERN_DEBUG
            "%s -- lvm_chr_ioctl: command: 0x%X  MINOR: %d  "
            "VG#: %d  LV#: %d  mode: 0x%X\n",
            lvm_name, command, minor, VG(minor), LV(minor), file->f_mode);
#endif

   switch ( command) {

      /* check lvm version to ensure driver/application interoperability */
      case LVM_GET_IOP_VERSION:
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( ushort*) arg,
                sizeof ( ushort))) != 0) return ret;
#endif
         memcpy_tofs ( arg, &lvm_iop_version, sizeof ( ushort));
         return 0;

#ifdef LVM_TOTAL_RESET
      /* test reset function */
      case LVM_RESET:
         for ( v = 0; v < ABS_MAX_VG; v++) {
            if ( vg[v] != NULL) {
               for ( l = 0; l < vg[v]->lv_max; l++) {
                  if ( vg[v]->lv[l] != NULL) {
#ifdef DEBUG
                     printk ( KERN_DEBUG
                              "%s -- fsync_dev and invalidate_buffers "
                              "for %s [%s]\n",
                              lvm_name,
                              vg[v]->lv[l]->lv_name,
                              kdevname ( vg[v]->lv[l]->lv_dev));
#endif
                     fsync_dev ( vg[v]->lv[l]->lv_dev);
                     vg[v]->vg_status &= ~VG_ACTIVE;
                     invalidate_buffers ( vg[v]->lv[l]->lv_dev);
                     if ( vg[v]->lv[l]->lv_current_pe != NULL) {
#ifdef DEBUG_VFREE
                        printk ( KERN_DEBUG
                                 "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
                        vfree ( vg[v]->lv[l]->lv_current_pe);
                     }
#ifdef DEBUG_VFREE
                     printk ( KERN_DEBUG
                              "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
                     vfree ( vg[v]->lv[l]);
                  }
               }
               for ( p = 0; p < vg[v]->pv_max; p++)
                  if ( vg[v]->pv[p] != NULL) {
#ifdef DEBUG_VFREE
                     printk ( KERN_DEBUG
                              "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
#ifdef LVM_GET_INODE
                     clear_inode ( vg[v]->pv[p]->inode);
#endif
                     vfree ( vg[v]->pv[p]);
                  }
#ifdef DEBUG_VFREE
               printk ( KERN_DEBUG
                        "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
               vfree ( vg[v]);
               vg[v] = NULL;
            }
         }
#ifdef MODULE
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
         while ( GET_USE_COUNT ( &__this_module) < lvm_chr_open_count)
            MOD_INC_USE_COUNT;
         while ( GET_USE_COUNT ( &__this_module) > lvm_chr_open_count)
            MOD_DEC_USE_COUNT;
#else
         mod_use_count_ = lvm_chr_open_count;
#endif
         lvm_reset_spindown = 1;
#endif /* MODULE */
         lock = 0; /* release lock */
         wake_up_interruptible ( &lvm_wait);
         return 0;
#endif /* LVM_TOTAL_RESET */


      /* set global read ahead setting */
      case BLKRASET:
#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_chr_ioctl -- BLKRASET: %d sectors for major %d\n",
                  lvm_name, ( int) arg, MAJOR( inode->i_rdev));
#endif
         if ( ( int) arg < LVM_MIN_READ_AHEAD ||
              ( int) arg > LVM_MAX_READ_AHEAD) return -EINVAL;
/* add code for read ahead array for all LVs here */
         read_ahead[MAJOR( inode->i_rdev)] = ( int) arg;
         return 0;


      /* get current global read ahead setting */
      case BLKRAGET:
#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_chr_ioctl -- BLKRAGET\n", lvm_name);
#endif
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( long *) arg,
                sizeof(long))) != 0) return ret;
#endif
         /* add code for read ahead array here */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
         copy_to_user ( ( long*) arg, &read_ahead[MAJOR( inode->i_rdev)],
                        sizeof ( read_ahead[MAJOR( inode->i_rdev)]));
#else
         put_user ( read_ahead[MAJOR( inode->i_rdev)], (long *) arg);
#endif
         return 0;


      /* lock/unlock i/o to a physical extent to move it to another
         physical volume (move's done in user space's pvmove) */
      case PE_LOCK_UNLOCK:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pe_lock_req_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &pe_lock_req, arg, sizeof ( pe_lock_req_t));

         switch ( pe_lock_req.lock) {
            case LOCK_PE:
               for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
                  if ( vg[VG(minor)]->pv[p] != NULL &&
                       pe_lock_req.data.pv_dev == vg[VG(minor)]->pv[p]->pv_dev)
                     break;
               }
      
               if ( p == vg[VG(minor)]->pv_max) return -ENXIO;

               pe_lock_req.lock = UNLOCK_PE;
               fsync_dev ( pe_lock_req.data.lv_dev);
               pe_lock_req.lock = LOCK_PE;
               break;

            case UNLOCK_PE:
               pe_lock_req.lock = UNLOCK_PE;
               pe_lock_req.data.lv_dev = \
               pe_lock_req.data.pv_dev = \
               pe_lock_req.data.pv_offset = 0;
               wake_up ( &lvm_map_wait);
               break;

            default:
               return -EINVAL;
         }

         return 0;


      /* remap a logical extent (after moving the physical extent) */
      case LE_REMAP:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( le_remap_req_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &le_remap_req, arg, sizeof ( le_remap_req_t));
         for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
            if ( vg[VG(minor)]->lv[l] != NULL &&
                 lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name,
                              le_remap_req.lv_name) == 0) {
               for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
                  if ( vg[VG(minor)]->lv[l]->lv_current_pe[le].dev ==
                       le_remap_req.old_dev &&
                       vg[VG(minor)]->lv[l]->lv_current_pe[le].pe ==
                       le_remap_req.old_pe) {
                     vg[VG(minor)]->lv[l]->lv_current_pe[le].dev =
                        le_remap_req.new_dev;
                     vg[VG(minor)]->lv[l]->lv_current_pe[le].pe =
                        le_remap_req.new_pe;
                     return 0;
                  }
               }
               return -EINVAL;
            }
         }

         return -ENXIO;


      /* create a VGDA */
      case VG_CREATE:
         if ( vg[VG(minor)] == NULL)
            vg[VG(minor)] = vmalloc ( sizeof ( vg_t));
         else return -EPERM;

         if ( vg[VG(minor)] == NULL) {
            printk ( KERN_CRIT
                     "%s -- VG_CREATE: vmalloc error VG\n", lvm_name);
            return -ENOMEM;
         }

         /* VG */
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( vg_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( vg[VG(minor)], arg, sizeof ( vg_t));
         vg[VG(minor)]->vg_status &= ~VG_ACTIVE;

         error = FALSE;
         if ( vg[VG(minor)]->pv_max > ABS_MAX_PV) {
            printk ( KERN_WARNING
                     "%s -- Can't activate VG: ABS_MAX_PV to small\n",
                     lvm_name);
            error = TRUE;
         }
         if ( vg[VG(minor)]->lv_max > ABS_MAX_LV) {
            printk ( KERN_WARNING
                     "%s -- Can't activate VG: ABS_MAX_LV to small for %u\n",
                     lvm_name, vg[VG(minor)]->lv_max);
            error = TRUE;
         }
         if ( error == TRUE) {
#ifdef DEBUG_VFREE
            printk ( KERN_DEBUG
                     "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
            vfree ( vg[VG(minor)]);
            vg[VG(minor)] = NULL;
            return -ENOMEM;
         }

         /* PVs */
         vg[VG(minor)]->pv_act = 0;
         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            /* user space address */
            if ( ( pvp = vg[VG(minor)]->pv[p]) != NULL) {
               vg[VG(minor)]->pv[p] = vmalloc ( sizeof ( pv_t));
               if ( vg[VG(minor)]->pv[p] == NULL) {
                  printk ( KERN_CRIT
                           "%s -- VG_CREATE: vmalloc error PV\n", lvm_name);
                  for ( ; p < vg[VG(minor)]->pv_max; p++)
                     vg[VG(minor)]->pv[p] = NULL;
                  do_vg_remove ( minor);
                  return -ENOMEM;
               }
#ifdef VERIFY_READ
               if ( ( ret = verify_area ( VERIFY_READ, pvp, sizeof ( pv_t)))
                    != 0) return ret;
#endif
               memcpy_fromfs ( vg[VG(minor)]->pv[p], pvp, sizeof ( pv_t));
               /* We don't need the PE free list
                  in kernel space as with LVs pe_t list (see below) */
               vg[VG(minor)]->pv[p]->pv_status = PV_ACTIVE;
               vg[VG(minor)]->pv_act++;

#ifdef LVM_GET_INODE
               /* insert a dummy inode for fs_may_mount */
               vg[VG(minor)]->pv[p]->inode =
                  lvm_get_inode ( vg[VG(minor)]->pv[p]->pv_dev);
#endif
            }
         }

         /* LVs */
         for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
            /* user space address */
            if ( ( lvp = vg[VG(minor)]->lv[l]) != NULL) {
               vg[VG(minor)]->lv[l] = vmalloc ( sizeof ( lv_t));
               if ( vg[VG(minor)]->lv[l] == NULL) {
                  printk ( KERN_CRIT
                           "%s -- VG_CREATE: vmalloc error LV\n", lvm_name);
                  do_vg_remove ( minor);
                  return -ENOMEM;
               }
#ifdef VERIFY_READ
               if ( ( ret = verify_area ( VERIFY_READ, lvp, sizeof ( lv_t)))
                    != 0) return ret;
#endif
               memcpy_fromfs ( vg[VG(minor)]->lv[l], lvp, sizeof ( lv_t));

               lvm_gendisk.part[minor+l+1].start_sect = 0;
               lvm_gendisk.part[minor+l+1].nr_sects =
                  vg[VG(minor)]->lv[l]->lv_size;
               lvm_size[minor+l+1] = vg[VG(minor)]->lv[l]->lv_size >> 1;
#ifdef DEBUG_VG_CREATE
               printk ( KERN_DEBUG
                        "%s -- VG_CREATE: lvm_size[%d]: %d\n",
                        lvm_name,
                        minor+l+1,
                        lvm_size[minor+l+1]);
#endif

               /* Get the lv_current_pe list for device mapping */
               pep = vg[VG(minor)]->lv[l]->lv_current_pe; /* user address */
               if ( pep != NULL) {
                  size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
                  if ( ( vg[VG(minor)]->lv[l]->lv_current_pe =
                            vmalloc ( size)) == NULL) {
                     printk ( KERN_CRIT
                              "%s -- VG_CREATE: vmalloc error LV_CURRENT_PE\n",
                              lvm_name);
                     for ( ; l < ABS_MAX_LV + 1; l++)
                        vg[VG(minor)]->lv[l] = NULL;
                     do_vg_remove ( minor);
                     return -ENOMEM;
                  }
#ifdef VERIFY_READ
                  if ( ( ret = verify_area ( VERIFY_READ, pep, size))
                       != 0) return ret;
#endif
                  memcpy_fromfs ( vg[VG(minor)]->lv[l]->lv_current_pe,
                                  pep, size);
               }
            }
         }

         vg[VG(minor)]->vg_status |= VG_ACTIVE;
         vg_count++;

#ifdef MODULE
         MOD_INC_USE_COUNT;
#endif
         return 0;


      /* remove an inactive VGDA */
      case VG_REMOVE:
         return do_vg_remove ( minor);


      /* extend a volume group */
      case VG_EXTEND:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
         if ( vg[VG(minor)]->pv_cur < vg[VG(minor)]->pv_max) {
            for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
               if ( vg[VG(minor)]->pv[p] == NULL) {
                  if ( ( vg[VG(minor)]->pv[p] =
                         vmalloc ( sizeof ( pv_t))) == NULL) {
                     printk ( KERN_CRIT
                              "%s -- VG_EXTEND: vmalloc error PV\n", lvm_name);
                     return -ENOMEM;
                  }
#ifdef VERIFY_READ
                  if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_t)))
                       != 0) return ret;
#endif
                  memcpy_fromfs ( vg[VG(minor)]->pv[p], arg, sizeof ( pv_t));
                  vg[VG(minor)]->pv[p]->pv_status = PV_ACTIVE;
                  vg[VG(minor)]->pv_cur++;
                  vg[VG(minor)]->pv_act++;
                  vg[VG(minor)]->pe_total += vg[VG(minor)]->pv[p]->pe_total;
#ifdef LVM_GET_INODE
                  /* insert a dummy inode for fs_may_mount */
                  vg[VG(minor)]->pv[p]->inode =
                     lvm_get_inode ( vg[VG(minor)]->pv[p]->pv_dev);
#endif
                  return 0;
               }
            }
         }
         return -EPERM;


      /* reduce a volume group */
      case VG_REDUCE:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_name)))
               != 0) return ret;
#endif
         memcpy_fromfs ( pv_name, arg, sizeof ( pv_name));

         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            if ( vg[VG(minor)]->pv[p] != NULL && 
                 lvm_strcmp ( vg[VG(minor)]->pv[p]->pv_name, pv_name) == 0) {
               if ( vg[VG(minor)]->pv[p]->lv_cur > 0) return -EPERM;
               vg[VG(minor)]->pe_total -= vg[VG(minor)]->pv[p]->pe_total;
               vg[VG(minor)]->pv_cur--;
               vg[VG(minor)]->pv_act--;
#ifdef DEBUG_VFREE
               printk ( KERN_DEBUG
                        "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
#ifdef LVM_GET_INODE
               clear_inode ( vg[VG(minor)]->pv[p]->inode);
#endif
               vfree ( vg[VG(minor)]->pv[p]);
               vg[VG(minor)]->pv[p] = NULL;
               return 0;
            }
         }
         return -ENXIO;


      /* set/clear extendability flag of volume group */
      case VG_SET_EXTENDABLE:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_READ, arg,
                                    sizeof ( extendable))) != 0)
            return ret;
#endif
         memcpy_fromfs ( &extendable, arg, sizeof ( extendable));
         if ( extendable == VG_EXTENDABLE ||
              extendable == ~VG_EXTENDABLE) {
            if ( extendable == VG_EXTENDABLE) {
               vg[VG(minor)]->vg_status |= VG_EXTENDABLE;
            } else {
               vg[VG(minor)]->vg_status &= ~VG_EXTENDABLE;
            }
         } else return -EINVAL;
         return 0;


      /* get volume group data (only the vg_t struct) */
      case VG_STATUS:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, arg, sizeof ( vg_t))) != 0)
            return ret;
#endif
         memcpy_tofs ( arg, vg[VG(minor)], sizeof ( vg_t));
         return 0;


      /* get volume group count */
      case VG_STATUS_GET_COUNT:
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, arg,
                                    sizeof ( vg_count))) != 0)
            return ret;
#endif
         memcpy_tofs ( arg, &vg_count, sizeof ( vg_count));
         return 0;


      /* get volume group count */
      case VG_STATUS_GET_NAMELIST:
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, arg,
                                    vg_count * NAME_LEN)) != 0)
            return ret;
#endif
         l = 0;
         for ( v = 0; v < ABS_MAX_VG; v++) 
            if ( vg[v] != NULL)
               memcpy_tofs ( arg + l++ * NAME_LEN, vg[v]->vg_name, NAME_LEN);
         return 0;


      /* create, remove, extend or reduce a logical volume */
      case LV_CREATE:
      case LV_REMOVE:
      case LV_EXTEND:
      case LV_REDUCE:
          if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( lv_name)))
              != 0) return ret;
#endif
         memcpy_fromfs ( lv_name, arg, sizeof ( lv_name));
         arg += sizeof ( lv_name);

         if ( command != LV_REMOVE) {
#ifdef VERIFY_READ
            if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( lv_t*)))
                 != 0) return ret;
#endif
            memcpy_fromfs ( &lvp, arg, sizeof ( lv_t*));
#ifdef VERIFY_READ
            if ( ( ret = verify_area ( VERIFY_READ, lvp, sizeof ( lv_t)))
                 != 0) return ret;
#endif
            memcpy_fromfs ( &lv, lvp, sizeof ( lv_t));
         }

         switch ( command) {
            case LV_CREATE:
               return do_lv_create ( minor, lv_name, &lv, inode);

            case LV_REMOVE:
               return do_lv_remove ( minor, lv_name);

            case LV_EXTEND:
            case LV_REDUCE:
               return do_lv_extend_reduce ( minor, lv_name, &lv, inode);
         }


      /* get status of a logical volume by name */
      case LV_STATUS_BYNAME:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg,
                                    sizeof ( lv_status_byname_req))) != 0)
            return ret;
#endif
         memcpy_fromfs ( &lv_status_byname_req, arg,
                         sizeof ( lv_status_byname_req_t));

         if ( lv_status_byname_req.lv == NULL) return -EINVAL;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, lv_status_byname_req.lv,
                                    sizeof ( lv_t))) != 0) return ret;
#endif
         memcpy_fromfs ( &lv, lv_status_byname_req.lv, sizeof ( lv_t));
         l = 0;
         while ( l < vg[VG(minor)]->lv_max) {
            if ( vg[VG(minor)]->lv[l] != NULL &&
                 lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name,
                              lv_status_byname_req.lv_name) == 0) {
#ifdef VERIFY_WRITE
               if ( ( ret = verify_area ( VERIFY_WRITE, lv_status_byname_req.lv,
                                          sizeof ( lv_t))) != 0) return ret;
#endif
               memcpy_tofs ( lv_status_byname_req.lv,
                             vg[VG(minor)]->lv[l], sizeof ( lv_t));
               if ( lv.lv_current_pe != NULL) {
                  size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
#ifdef VERIFY_WRITE
                  if ( ( ret = verify_area ( VERIFY_WRITE,
                                             lv.lv_current_pe,
                                             size)) != 0) return ret;
#endif
                  memcpy_tofs ( lv.lv_current_pe,
                                vg[VG(minor)]->lv[l]->lv_current_pe,
                                size);
               }
               return 0;
            }
            l++;
         }
         return -ENXIO;


      /* get status of a logical volume by index */
      case LV_STATUS_BYINDEX:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg,
                                    sizeof ( lv_status_byindex_req)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &lv_status_byindex_req, arg,
                         sizeof ( lv_status_byindex_req));

         lvp = lv_status_byindex_req.lv;
         if ( lvp == NULL) return -EINVAL;
         l = lv_status_byindex_req.lv_index;
         if ( vg[VG(minor)]->lv[l] == NULL) return -ENXIO;

#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, lvp, sizeof ( lv_t)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &lv, lvp, sizeof ( lv_t));
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, lvp, sizeof ( lv_t)))
              != 0) return ret;
#endif
         memcpy_tofs ( lvp, vg[VG(minor)]->lv[l], sizeof ( lv_t));

         if ( lv.lv_current_pe != NULL) {
            size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
#ifdef VERIFY_WRITE
            if ( ( ret = verify_area ( VERIFY_WRITE,
                                       lv.lv_current_pe,
                                       size)) != 0) return ret;
#endif
            memcpy_tofs ( lv.lv_current_pe,
                          vg[VG(minor)]->lv[l]->lv_current_pe,
                          size);
         }
         return 0;


      /* change a physical volume */
      case PV_CHANGE:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_change_req)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &pv_change_req, arg, sizeof ( pv_change_req));

         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            if ( vg[VG(minor)]->pv[p] != NULL &&
                 lvm_strcmp ( vg[VG(minor)]->pv[p]->pv_name,
                              pv_change_req.pv_name) == 0) {
#ifdef VERIFY_READ
               if ( ( ret = verify_area ( VERIFY_READ, pv_change_req.pv,
                                          sizeof ( pv_t))) != 0) return ret;
#endif
#ifdef LVM_GET_INODE
               inode_sav = vg[VG(minor)]->pv[p]->inode;
#endif
               memcpy_fromfs ( vg[VG(minor)]->pv[p], pv_change_req.pv,
                               sizeof ( pv_t));
#ifdef LVM_GET_INODE
               vg[VG(minor)]->pv[p]->inode = inode_sav;
#endif
               return 0;
            }
         }
         return -ENXIO;


      /* get physical volume data (only the pv_t struct) */
      case PV_STATUS:
         if ( vg[VG(minor)] == NULL) return -ENXIO;
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_name)))
              != 0) return ret;
#endif
         memcpy_fromfs ( pv_name, arg, sizeof ( pv_name));
         arg += sizeof ( pv_name);
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_t*)))
              != 0) return ret;
#endif
         memcpy_fromfs ( &pvp, arg, sizeof ( pv_t *));
         for ( p = 0; p < vg[VG(minor)]->pv_max; p++) {
            if ( vg[VG(minor)]->pv[p] != NULL) {
               if ( lvm_strcmp ( vg[VG(minor)]->pv[p]->pv_name, pv_name) == 0) {
#ifdef VERIFY_WRITE
                  if ( ( ret = verify_area ( VERIFY_WRITE, pvp,
                         sizeof ( pv_t))) != 0) return ret;
#endif
                  memcpy_tofs ( pvp, vg[VG(minor)]->pv[p], sizeof ( pv_t));
                  return 0;
               }
            }
         }
         return -ENXIO;


      /* search for a physical volume to flush;
         don't care about invalid volumes --> return 0 */
      case PV_FLUSH:
#ifdef VERIFY_READ
         if ( ( ret = verify_area ( VERIFY_READ, arg, sizeof ( pv_name)))
              != 0) return ret;
#endif
         memcpy_fromfs ( pv_name, arg, sizeof ( pv_name));

         for ( v = 0; v < ABS_MAX_VG; v++) {
            if ( vg[v] == NULL) continue;
            for ( p = 0; p < vg[v]->pv_max; p++) {
               if ( vg[v]->pv[p] != NULL &&
                    lvm_strcmp ( vg[v]->pv[p]->pv_name, pv_name) == 0) {
                  fsync_dev ( vg[v]->pv[p]->pv_dev);
                  invalidate_buffers ( vg[v]->pv[p]->pv_dev);
                  return 0;
               }
            }
         }
         return 0;


      default:
         printk ( KERN_WARNING
                  "%s -- lvm_chr_ioctl: unknown command %d\n",
                  lvm_name, command);
         return -EINVAL;
   }
}


/*
 * character device close routine
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
static int lvm_chr_close ( struct inode *inode, struct file *file)
#else
static void lvm_chr_close ( struct inode *inode, struct file *file)
#endif
{
#ifdef DEBUG
   int minor = MINOR ( inode->i_rdev);

   printk ( KERN_DEBUG
            "%s -- lvm_chr_close MINOR: %d  VG#: %d  LV#: %d\n",
            lvm_name, minor, VG(minor), LV(minor));
#endif

#ifdef MODULE
   MOD_DEC_USE_COUNT;
#endif

   lvm_chr_open_count--;
   if ( lvm_chr_open_count == 0) {
      lock = 0; /* release lock */
      wake_up_interruptible ( &lvm_wait);
   }

#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
   return 0;
#else
   return;
#endif
}


/********************************************************************
 *
 * Block device functions
 *
 ********************************************************************/

/*
 * block device open routine
 */
static int lvm_blk_open ( struct inode *inode, struct file *file) {
   int minor = MINOR ( inode->i_rdev);

#ifdef DEBUG_LVM_BLK_OPEN
   printk ( KERN_DEBUG
            "%s -- lvm_blk_open MINOR: %d  VG#: %d  LV#: %d  mode: 0x%X\n",
            lvm_name, minor, VG(minor), LV(minor), file->f_mode);
#endif

   if ( vg[VG(minor)] != NULL &&
        ( vg[VG(minor)]->vg_status & VG_ACTIVE) &&
        vg[VG(minor)]->lv[LV(minor)-1] != NULL &&
        LV(minor) > 0 &&
        LV(minor) <= vg[VG(minor)]->lv_max) {

      /* Check parallel LV spindown (LV remove) */
      if ( vg[VG(minor)]->lv[LV(minor)-1]->lv_status & LV_SPINDOWN)
         return -EPERM;

      /* Check inactive LV and open for read/write */
      if ( file->f_mode & O_RDWR) {
         if ( ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_status & LV_ACTIVE))
            return -EPERM;
         if ( ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_access & LV_WRITE))
            return -EACCES;
      }

      if ( vg[VG(minor)]->lv[LV(minor)-1]->lv_open == 0)
         vg[VG(minor)]->lv_open++;
      vg[VG(minor)]->lv[LV(minor)-1]->lv_open++;

#ifdef MODULE
      MOD_INC_USE_COUNT;
#endif

#ifdef DEBUG_LVM_BLK_OPEN
      printk ( KERN_DEBUG
               "%s -- lvm_blk_open MINOR: %d  VG#: %d  LV#: %d  size: %d\n",
               lvm_name, minor, VG(minor), LV(minor),
               vg[VG(minor)]->lv[LV(minor)-1]->lv_size);
#endif

      return 0;
   }

   return -ENXIO;
}


/*
 * block device i/o-control routine
 */
static int lvm_blk_ioctl (struct inode *inode, struct file *file,
                          uint command, ulong a) {
   int minor = MINOR ( inode->i_rdev);
#ifdef VERIFY_WRITE
   int ret;
#endif
   void *arg = ( void*) a;
   struct hd_geometry *hd = ( struct hd_geometry *) a;

#ifdef DEBUG_IOCTL
   printk ( KERN_DEBUG
            "%s -- lvm_blk_ioctl MINOR: %d  command: 0x%X  arg: %X  "
            "VG#: %d  LV#: %d\n",
            lvm_name, minor, command, ( uint) arg, VG(minor), LV(minor));
#endif

   switch ( command) {
      /* return device size */
      case BLKGETSIZE:
#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_blk_ioctl -- BLKGETSIZE: %u\n",
                  lvm_name, vg[VG(minor)]->lv[LV(minor)-1]->lv_size);
#endif
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( long *) arg,
                sizeof ( long))) != 0) return ret;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
         copy_to_user ( ( long*) arg, &vg[VG(minor)]->lv[LV(minor)-1]->lv_size,
                        sizeof ( vg[VG(minor)]->lv[LV(minor)-1]->lv_size));
#else
         put_user ( vg[VG(minor)]->lv[LV(minor)-1]->lv_size, ( long *) arg);
#endif
         break;


      /* flush buffer cache */
      case BLKFLSBUF:
#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_blk_ioctl -- BLKFLSBUF\n", lvm_name);
#endif
         fsync_dev ( inode->i_rdev);
         invalidate_buffers ( inode->i_rdev);
         break;


      /* set read ahead for block device */
      case BLKRASET:
         /* super user validation */
         if ( ! suser ()) return -EACCES;

#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_chr_ioctl -- BLKRASET: %d sectors for major %d\n",
                  lvm_name, ( int) arg, MAJOR( inode->i_rdev));
#endif
         if ( ( int) arg < LVM_MIN_READ_AHEAD ||
              ( int) arg > LVM_MAX_READ_AHEAD) return -EINVAL;
         /* add code for single LV read ahead sector handling here */
         read_ahead[MAJOR( inode->i_rdev)] = ( int) arg;
         break;


      /* get current read ahead setting */
      case BLKRAGET:
#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_blk_ioctl -- BLKRAGET\n", lvm_name);
#endif
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, ( long *) arg,
                sizeof(long))) != 0) return ret;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
         copy_to_user ( ( long*) arg, &read_ahead[MAJOR( inode->i_rdev)],
                        sizeof ( read_ahead[MAJOR( inode->i_rdev)]));
#else
         put_user ( read_ahead[MAJOR( inode->i_rdev)], (long *) arg);
#endif
         break;


      /* get disk geometry */
      case HDIO_GETGEO:
#ifdef DEBUG_IOCTL
         printk ( KERN_DEBUG
                  "%s -- lvm_blk_ioctl -- HDIO_GETGEO\n", lvm_name);
#endif
         if ( hd == NULL) return -EINVAL;
#ifdef VERIFY_WRITE
         if ( ( ret = verify_area ( VERIFY_WRITE, hd,
                                    sizeof ( struct hd_geometry))) != 0)
            return ret;
#endif
         {
            unsigned char heads = 64;
            unsigned char sectors = 32;
            long start = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
           short cylinders = vg[VG(minor)]->lv[LV(minor)]->lv_size /
                             heads / sectors;

           copy_to_user ( ( char*) &hd->heads, &heads, sizeof ( heads));
           copy_to_user ( ( char*) &hd->sectors, &sectors, sizeof ( sectors));
           copy_to_user ( ( short*) &hd->cylinders,
                          &cylinders, sizeof ( cylinders));
           copy_to_user ( ( long*) &hd->start, &start, sizeof ( start));
#else
            put_user ( heads, ( char*) &hd->heads);
            put_user ( sectors, ( char*) &hd->sectors);
            put_user ( vg[VG(minor)]->lv[LV(minor)]->lv_size / 8,
                       ( short*) &hd->cylinders);
            put_user ( start, ( long *) &hd->start);
#endif
         }

#ifdef DEBUG_IOCTL
   printk ( KERN_DEBUG
            "%s -- lvm_blk_ioctl -- cylinders: %d\n",
            lvm_name, vg[VG(minor)]->lv[LV(minor)]->lv_size / 8);
#endif
         break;


      /* set access flags of a logical volume */
      case LV_SET_ACCESS:
         /* super user validation */
         if ( ! suser ()) return -EACCES;
         vg[VG(minor)]->lv[LV(minor)-1]->lv_access = ( uint) arg;
         break;


      /* set status flags of a logical volume */
      case LV_SET_STATUS:
         /* super user validation */
         if ( ! suser ()) return -EACCES;
         if ( ! ( ( uint) arg & LV_ACTIVE) &&
              vg[VG(minor)]->lv[LV(minor)-1]->lv_open > 1) return -EPERM;
         vg[VG(minor)]->lv[LV(minor)-1]->lv_status = ( uint) arg;
         break;


      /* set allocation flags of a logical volume */
      case LV_SET_ALLOCATION:
         /* super user validation */
         if ( ! suser ()) return -EACCES;
         vg[VG(minor)]->lv[LV(minor)-1]->lv_allocation = ( uint) arg;
         break;


      default:
         printk ( KERN_WARNING
                  "%s -- lvm_blk_ioctl: unknown command %d\n",
                  lvm_name, command);
         return -EINVAL;
   }

   return 0;
}


/*
 * block device close routine
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
static int lvm_blk_close ( struct inode *inode, struct file *file)
#else
static void lvm_blk_close ( struct inode *inode, struct file *file)
#endif
{
   int minor = MINOR ( inode->i_rdev);

#ifdef DEBUG
   printk ( KERN_DEBUG
            "%s -- lvm_blk_close MINOR: %d  VG#: %d  LV#: %d\n",
            lvm_name, minor, VG(minor), LV(minor));
#endif

   sync_dev ( inode->i_rdev);
   if ( vg[VG(minor)]->lv[LV(minor)-1]->lv_open == 1) vg[VG(minor)]->lv_open--;
   vg[VG(minor)]->lv[LV(minor)-1]->lv_open--;

#ifdef MODULE
   MOD_DEC_USE_COUNT;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
   return 0;
#else
   return;
#endif
} /* static int/void lvm_blk_close () */


#if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS
/*
 * Support function /proc-Filesystem
 */
#define  LVM_PROC_BUF   ( i == 0 ? dummy_buf : &buf[sz])

static int lvm_proc_get_info ( char *page, char **start, off_t pos,
                               int count, int whence) {
   int c, i, l, p, v, vg_counter, pv_counter, lv_counter,
       lv_counter_tmp, pe_t_bytes, seconds;
   static off_t sz;
   off_t sz_last;
   char allocation_flag, inactive_flag, rw_flag, stripes_flag;
   char *lv_name = NULL;
   static char *buf = NULL;

#ifdef DEBUG_LVM_PROC_GET_INFO
   printk ( KERN_DEBUG
            "%s - lvm_proc_get_info CALLED  pos: %lu  count: %d  whence: %d\n",
            lvm_name, pos, count, whence);
#endif

   if ( pos == 0 || buf == NULL) {
      sz_last = vg_counter = pv_counter = lv_counter = pe_t_bytes = 0;
   
      /* search for activity */
      for ( v = 0; v < ABS_MAX_VG; v++) {
         if ( vg[v] != NULL) {
            vg_counter++;
            pv_counter += vg[v]->pv_cur;
            lv_counter += vg[v]->lv_cur;
            for ( l = 0; l < vg[v]->lv_max; l++)
               if ( vg[v]->lv[l] != NULL)
                  pe_t_bytes += vg[v]->lv[l]->lv_allocated_le * sizeof ( pe_t);
         }
      }
   
      if ( buf != NULL) {
#ifdef DEBUG_VFREE
         printk ( KERN_DEBUG
                  "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
         vfree ( buf);
         buf = NULL;
      }

      /* 2 times: first to get size, 2nd to fill the vmalloced buffer */
      for ( i = 0; i < 2; i++) {
         sz = 0;
         sz += sprintf ( LVM_PROC_BUF,
                         "LVM %s %s  %d VG%s  %d PV%s  %d LV%s\n",
#ifdef MODULE
                         "module",
#else
                         "driver",
#endif
                         lvm_short_version,
                         vg_counter,
                         vg_counter == 0 || vg_counter > 1 ? "s" : "",
                         pv_counter,
                         pv_counter == 0 || pv_counter > 1 ? "s" : "",
                         lv_counter,
                         lv_counter == 0 || lv_counter > 1 ? "s" : "");
         sprintf ( LVM_PROC_BUF, "                       ");
         sz += lvm_strlen ( lvm_short_version) + 4;
         sz += sprintf ( LVM_PROC_BUF,
                         "  %d read_ahead  %lu vmalloced",
                         read_ahead[LVM_BLOCK_MAJOR],
                         vg_counter * sizeof ( vg_t) +
                         pv_counter * sizeof ( pv_t) +
                         lv_counter * sizeof ( lv_t) +
                         pe_t_bytes + sz_last);
         seconds = CURRENT_TIME - loadtime;
         if ( seconds < 0) loadtime = CURRENT_TIME + seconds;
         sz += sprintf ( LVM_PROC_BUF, "  %d day%s %d:%02d:%02d active\n",
                                       seconds / 86400,
                                       seconds / 86400 == 0 ||
                                       seconds / 86400 > 1 ? "s": "",
                                       ( seconds % 86400) / 3600,
                                       ( seconds % 3600) / 60,
                                       seconds % 60);

         if ( vg_counter > 0) {
            for ( v = 0; v < ABS_MAX_VG; v++) {
               /* volume group */
               if ( vg[v] != NULL) {
                  inactive_flag = ' ';
                  if ( ! ( vg[v]->vg_status & VG_ACTIVE))
                     inactive_flag = 'I';
                  sz += sprintf ( LVM_PROC_BUF,
                                  "\nVG: %c%s  [%d PV, %d LV/%d open] "
                                  " PE Size: %d KB\n"
                                  "  Usage [KB/PE]: %d /%d total  "
                                  "%d /%d used  %d /%d free",
                                  inactive_flag,
                                  vg[v]->vg_name,
                                  vg[v]->pv_cur,
                                  vg[v]->lv_cur,
                                  vg[v]->lv_open,
                                  vg[v]->pe_size >> 1,
                                  vg[v]->pe_size * vg[v]->pe_total >> 1,
                                  vg[v]->pe_total,
                                  vg[v]->pe_allocated * vg[v]->pe_size >> 1,
                                  vg[v]->pe_allocated,
                                  ( vg[v]->pe_total - vg[v]->pe_allocated) *
                                  vg[v]->pe_size >> 1,
                                  vg[v]->pe_total - vg[v]->pe_allocated);

                  /* physical volumes */
                  sz += sprintf ( LVM_PROC_BUF, "\n  PVs: ");
                  c = 0;
                  for ( p = 0; p < vg[v]->pv_max; p++) {
                     if ( vg[v]->pv[p] != NULL) {
                        inactive_flag = 'A';
                        if ( ! ( vg[v]->pv[p]->pv_status & PV_ACTIVE))
                           inactive_flag = 'I';
                        allocation_flag = 'A';
                        if ( ! ( vg[v]->pv[p]->pv_allocatable & PV_ALLOCATABLE))
                           allocation_flag = 'N';
                        sz += sprintf ( LVM_PROC_BUF,
                                        "[%c%c] %-11s %8d /%-6d  "
                                        "%8d /%-6d  %8d /%-6d",
                                        inactive_flag,
                                        allocation_flag,
                                        vg[v]->pv[p]->pv_name,
                                        vg[v]->pv[p]->pe_total *
                                        vg[v]->pv[p]->pe_size >> 1,
                                        vg[v]->pv[p]->pe_total,
                                        vg[v]->pv[p]->pe_allocated *
                                        vg[v]->pv[p]->pe_size >> 1,
                                        vg[v]->pv[p]->pe_allocated,
                                        ( vg[v]->pv[p]->pe_total -
                                          vg[v]->pv[p]->pe_allocated) *
                                        vg[v]->pv[p]->pe_size >> 1,
                                        vg[v]->pv[p]->pe_total -
                                        vg[v]->pv[p]->pe_allocated);
                        c++;
                        if ( c < vg[v]->pv_cur)
                           sz += sprintf ( LVM_PROC_BUF,
                                           "\n       ");
                     }
                  }

                  /* logical volumes */
                  sz += sprintf ( LVM_PROC_BUF,
                                  "\n    LVs: ");
                  c = lv_counter_tmp = 0;
                  for ( l = 0; l < vg[v]->lv_max; l++) {
                     if ( vg[v]->lv[l] != NULL) {
                        lv_counter_tmp++;
                        inactive_flag = 'A';
                        if ( ! ( vg[v]->lv[l]->lv_status & LV_ACTIVE))
                           inactive_flag = 'I';
                        rw_flag = 'R';
                        if ( vg[v]->lv[l]->lv_access & LV_WRITE) rw_flag = 'W';
                        allocation_flag = 'D';
                        if ( vg[v]->lv[l]->lv_allocation & LV_CONTIGUOUS)
                           allocation_flag = 'C';
                        stripes_flag = 'L';
                        if ( vg[v]->lv[l]->lv_stripes > 1) stripes_flag = 'S';
                        sz += sprintf ( LVM_PROC_BUF,
                                        "[%c%c%c%c",
                                        inactive_flag,
                                        rw_flag,
                                        allocation_flag,
                                        stripes_flag);
                        if ( vg[v]->lv[l]->lv_stripes > 1)
                           sz += sprintf ( LVM_PROC_BUF, "%-2d",
                                           vg[v]->lv[l]->lv_stripes);
                        else
                           sz += sprintf ( LVM_PROC_BUF, "  ");
                        lv_name = lvm_strrchr ( vg[v]->lv[l]->lv_name, '/');
                        if ( lv_name != NULL) lv_name++;
                        else lv_name = vg[v]->lv[l]->lv_name;
                        sz += sprintf ( LVM_PROC_BUF, "] %-25s", lv_name);
                        if ( lvm_strlen ( lv_name) > 25)
                           sz += sprintf ( LVM_PROC_BUF,
                                           "\n                              ");
                        sz += sprintf ( LVM_PROC_BUF, "%9d /%-6d   ",
                                        vg[v]->lv[l]->lv_size >> 1,
                                        vg[v]->lv[l]->lv_size / vg[v]->pe_size);

                        if ( vg[v]->lv[l]->lv_open == 0)
                           sz += sprintf ( LVM_PROC_BUF, "closed");
                        else
                           sz += sprintf ( LVM_PROC_BUF, "%dx open",
                                           vg[v]->lv[l]->lv_open);
                        c++;
                        if ( c < vg[v]->lv_cur)
                           sz += sprintf ( LVM_PROC_BUF,
                                           "\n         ");
                     }
                  }
                  if ( lv_counter_tmp == 0)
                     sz += sprintf ( LVM_PROC_BUF, "none");
                  sz += sprintf ( LVM_PROC_BUF, "\n");
               }
            }
         }

         if ( buf == NULL) {
            if ( ( buf = vmalloc ( sz + 64)) == NULL) {
               sz = 0;
               return sprintf ( page, "%s - vmalloc error at line %d\n",
                                      lvm_name, __LINE__);
            }
         }
         sz_last = sz;
      }
   }

   if ( pos > sz - 1) {
      vfree ( buf);
      buf = NULL;
      return -1;
   }

   *start = &buf[pos];
   if ( sz - pos < count) return sz - pos;
   else                   return count;
}
#endif /* #if defined CONFIG_LVM_PROC_FS && defined CONFIG_PROC_FS */


/*
 * Support function for /usr/src/linux/drivers/block/ll_rw_blk.c
 * (see init_module/lvm_init)
 */
static int lvm_map ( int minor, kdev_t *rdev,
                     ulong *rsector, ulong size, int rw) {
   ulong index;
   ulong rsector_sav;
   kdev_t rdev_sav;

   if ( ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_status & LV_ACTIVE)) {
      printk ( KERN_ALERT
               "%s - lvm_map: ll_rw_blk for inactive LV %s\n",
               lvm_name, vg[VG(minor)]->lv[LV(minor)-1]->lv_name);
      return -1;
   }

   if ( rw == WRITE &&
        ! ( vg[VG(minor)]->lv[LV(minor)-1]->lv_access & LV_WRITE)) {
      printk ( KERN_CRIT
               "%s - lvm_map: ll_rw_blk write for readonly LV %s\n",
               lvm_name, vg[VG(minor)]->lv[LV(minor)-1]->lv_name);
      return -1;
   }

#ifdef DEBUG_MAP_SIZE
   if ( size != 2)
      printk ( KERN_DEBUG
               "%s - lvm_map minor:%d size:%lu ", lvm_name, minor, size);
#endif

#ifdef DEBUG_MAP
   printk ( KERN_DEBUG
            "%s - lvm_map minor:%d  *rdev: %02d:%02d  *rsector: %lu  "
            "size:%lu\n",
            lvm_name, minor,
            MAJOR ( *rdev),
            MINOR ( *rdev),
            *rsector, size);
#endif

   if ( *rsector + size > vg[VG(minor)]->lv[LV(minor)-1]->lv_size) {
      printk ( KERN_ALERT
               "%s - lvm_map *rsector: %lu + size: %lu too large for"
               " %02d:%02d\n", lvm_name, *rsector, size, VG(minor), LV(minor));
      return -1;
   }

   rsector_sav = *rsector;
   rdev_sav    = *rdev;

lvm_second_remap:

   /* linear mapping */
   if ( vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes < 2) {
      index = *rsector / vg[VG(minor)]->pe_size; /* get the index */
      *rsector = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe +
                 ( *rsector % vg[VG(minor)]->pe_size);
      *rdev = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].dev;

#ifdef DEBUG_MAP
      printk ( KERN_DEBUG
               "lv_current_pe[%ld].pe: %ld  rdev: %02d:%02d  rsector:%ld\n",
               index,
               vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe,
               MAJOR ( *rdev),
               MINOR ( *rdev),
               *rsector);
#endif

   /* striped mapping */
   } else {
      ulong stripe_index;
      ulong stripe_length;

      stripe_length = vg[VG(minor)]->pe_size *
                      vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes;
      stripe_index = *rsector % stripe_length /
                     vg[VG(minor)]->lv[LV(minor)-1]->lv_stripesize;
      index = *rsector / stripe_length +
              ( stripe_index % vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes) *
              ( vg[VG(minor)]->lv[LV(minor)-1]->lv_allocated_le /
                vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes);
      *rsector = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe +
                 ( *rsector % stripe_length) -
                 stripe_index % vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes *
                 vg[VG(minor)]->lv[LV(minor)-1]->lv_stripesize -
                 stripe_index / vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes *
                 ( vg[VG(minor)]->lv[LV(minor)-1]->lv_stripes - 1) *
                 vg[VG(minor)]->lv[LV(minor)-1]->lv_stripesize;
      *rdev = vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].dev;
   }

   /* statistic */
   if ( rw == WRITE || rw == WRITEA)
      vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].writes++;
   else
      vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].reads++;

#ifdef DEBUG_MAP
   printk ( KERN_DEBUG
            "lv_current_pe[%ld].pe: %ld  rdev: %02d:%02d  rsector:%ld\n"
            "stripe_length: %ld  stripe_index: %ld\n",
            index,
            vg[VG(minor)]->lv[LV(minor)-1]->lv_current_pe[index].pe,
            MAJOR ( *rdev),
            MINOR ( *rdev),
            *rsector,
            stripe_length,
            stripe_index);
#endif

   /* handle physical extents on the move */
   if ( pe_lock_req.lock == LOCK_PE) {
      if ( *rdev == pe_lock_req.data.pv_dev &&
           *rsector >= pe_lock_req.data.pv_offset &&
           *rsector < ( pe_lock_req.data.pv_offset + vg[VG(minor)]->pe_size)) {
         sleep_on ( &lvm_map_wait);
         *rsector = rsector_sav;
         *rdev    = rsector_sav;
         goto lvm_second_remap;
      }
   }

   return 0;
}


/* internal support functions */

/*
 * this one never should be called...
 */
static void lvm_dummy_device_request ( void)
{
  printk ( KERN_EMERG
           "%s -- oops, got lvm request?\n", lvm_name);
  return;
}


/*
 * support function VGDA remove
 */
static int do_vg_remove ( int minor) {
   int l, p;

   if ( vg[VG(minor)] == NULL)      return -ENXIO;
   if ( vg[VG(minor)]->lv_open > 0) return -EPERM;

   vg[VG(minor)]->vg_status &= ~VG_ACTIVE;

   /* Free PVs */
   for ( p = 0; p < vg[VG(minor)]->pv_max; p++)
      if ( vg[VG(minor)]->pv[p] != NULL) {
#ifdef DEBUG_VFREE
         printk ( KERN_DEBUG
                  "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
#ifdef LVM_GET_INODE
         clear_inode ( vg[VG(minor)]->pv[p]->inode);
#endif
         vfree ( vg[VG(minor)]->pv[p]);
      }

   /* Free LVs */
   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] != NULL) {
#ifdef DEBUG
         printk ( KERN_DEBUG
                  "%s -- fsync_dev and "
                  "invalidate_buffers for %s [%s]\n",
                  lvm_name,
                  vg[VG(minor)]->lv[l]->lv_name,
                  kdevname ( vg[VG(minor)]->lv[l]->lv_dev));
#endif
         vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
         fsync_dev ( vg[VG(minor)]->lv[l]->lv_dev);
         vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
         invalidate_buffers ( vg[VG(minor)]->lv[l]->lv_dev);
         lvm_gendisk.part[minor+l+1].start_sect = -1;
         lvm_gendisk.part[minor+l+1].nr_sects = 0;
         lvm_size[minor+l+1] = 0;

         /* Free PEs */
         if ( vg[VG(minor)]->lv[l]->lv_current_pe != NULL) {
#ifdef DEBUG_VFREE
            printk ( KERN_DEBUG
                     "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
            vfree ( vg[VG(minor)]->lv[l]->lv_current_pe);
         }
         /* Free LVs */
#ifdef DEBUG_VFREE
         printk ( KERN_DEBUG "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
         vfree ( vg[VG(minor)]->lv[l]);
      }
   }

#ifdef DEBUG_VFREE
   printk ( KERN_DEBUG "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
   vfree ( vg[VG(minor)]);
   vg[VG(minor)] = NULL;
   vg_count--;

#ifdef MODULE
   MOD_DEC_USE_COUNT;
#endif
   return 0;
}


/*
 * support function logical volume create
 */
static int do_lv_create ( int minor, char *lv_name, lv_t *lv,
                          struct inode *inode) {
   int l, le, l_new, p, size;
#ifdef VERIFY_READ
   int ret;
#endif
   ulong lv_status_save;

   if ( ( pep = lv->lv_current_pe) == NULL) return -EINVAL;

   l_new = -1;
   /* in case of lv_remove(), lv_create() pair; for egg. lvrename does this */
   if ( vg[VG(minor)]->lv[lv->lv_number] == NULL) l_new = lv->lv_number;
   else {
      for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
         if ( lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name, lv_name) == 0)
            return -EEXIST;
         if ( vg[VG(minor)]->lv[l] == NULL) if ( l_new == -1) l_new = l;
      }
   }
   if ( l_new == -1) return -EPERM;
   l = l_new;

   vg[VG(minor)]->lv[l] = vmalloc ( sizeof ( lv_t));
   if ( vg[VG(minor)]->lv[l] == NULL) {
      printk ( KERN_CRIT "%s -- LV_CREATE: vmalloc error LV\n", lvm_name);
      return -ENOMEM;
   }

   /* copy preloaded LV */
   lvm_memcpy ( ( char*) vg[VG(minor)]->lv[l], ( char *) lv, sizeof ( lv_t));
   lv_status_save = vg[VG(minor)]->lv[l]->lv_status;
   vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;

   /* get the PE structures from user space */
   size = vg[VG(minor)]->lv[l]->lv_allocated_le * sizeof ( pe_t);
   if ( ( vg[VG(minor)]->lv[l]->lv_current_pe =
          vmalloc ( size)) == NULL) {
      printk ( KERN_CRIT
               "%s -- LV_CREATE: vmalloc error LV_CURRENT_PE of %d Byte\n",
               lvm_name, size);
#ifdef DEBUG_VFREE
      printk ( KERN_DEBUG "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
      vfree ( vg[VG(minor)]->lv[l]);
      vg[VG(minor)]->lv[l] = NULL;
      return -ENOMEM;
   }

#ifdef VERIFY_READ
   if ( ( ret = verify_area ( VERIFY_READ, pep, size)) != 0) {
      vfree ( vg[VG(minor)]->lv[l]);
      vg[VG(minor)]->lv[l] = NULL;
      return ret;
   }
#endif
   memcpy_fromfs ( vg[VG(minor)]->lv[l]->lv_current_pe, pep, size);

   vg[VG(minor)]->lv_cur++;

   /* correct the PE count in PVs */
   for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
      vg[VG(minor)]->pe_allocated++;
      for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
         if ( vg[VG(minor)]->pv[p]->pv_dev ==
              vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
            vg[VG(minor)]->pv[p]->pe_allocated++;
      }
   }

   lvm_gendisk.part[minor+l+1].start_sect = 0;
   lvm_gendisk.part[minor+l+1].nr_sects = vg[VG(minor)]->lv[l]->lv_size;
   lvm_size[minor+l+1] = vg[VG(minor)]->lv[l]->lv_size >> 1;

   vg[VG(minor)]->lv[l]->lv_status = lv_status_save;

   return 0;
}


/*
 * support function logical volume remove
 */
static int do_lv_remove ( int minor, char *lv_name) {
   uint l, le, p;

   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] != NULL &&
           lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name, lv_name) == 0) {
         if ( vg[VG(minor)]->lv[l]->lv_open > 0) return -EBUSY;
#ifdef DEBUG
         printk ( KERN_DEBUG
                  "%s -- fsync_dev and "
                  "invalidate_buffers for %s [%s] in %s\n",
                  lvm_name, vg[VG(minor)]->lv[l]->lv_name,
                  kdevname ( vg[VG(minor)]->lv[l]->lv_dev),
                  vg[VG(minor)]->vg_name);
#endif
         vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
         fsync_dev ( vg[VG(minor)]->lv[l]->lv_dev);
         vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
         invalidate_buffers ( vg[VG(minor)]->lv[l]->lv_dev);
         lvm_gendisk.part[minor+l+1].start_sect = -1;
         lvm_gendisk.part[minor+l+1].nr_sects = 0;
         lvm_size[minor+l+1] = 0;

         for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
            vg[VG(minor)]->pe_allocated--;
            for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
               if (  vg[VG(minor)]->pv[p]->pv_dev ==
                     vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
                  vg[VG(minor)]->pv[p]->pe_allocated--;
            }
         }

#ifdef DEBUG_VFREE
         printk ( KERN_DEBUG "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
         vfree ( vg[VG(minor)]->lv[l]->lv_current_pe);

#ifdef DEBUG_VFREE
         printk ( KERN_DEBUG "%s -- vfree %d\n", lvm_name, __LINE__);
#endif
         vfree ( vg[VG(minor)]->lv[l]);

         vg[VG(minor)]->lv[l] = NULL;
         vg[VG(minor)]->lv_cur--;
         return 0;
      }
   }
   return -ENXIO;
}


/*
 * support function logical volume extend or reduce
 */
static int do_lv_extend_reduce ( int minor, char *lv_name, lv_t *lv,
                                 struct inode *inode) {
   int l, le, p, size;
#ifdef VERIFY_READ
   int ret;
#endif
   uint32_t end, lv_status_save;
   pe_t *pe;

   if ( ( pep = lv->lv_current_pe) == NULL) return -EINVAL;

   for ( l = 0; l < vg[VG(minor)]->lv_max; l++) {
      if ( vg[VG(minor)]->lv[l] != NULL &&
           lvm_strcmp ( vg[VG(minor)]->lv[l]->lv_name, lv_name) == 0)
         break;
   }
   if ( l == vg[VG(minor)]->lv_max) return -ENXIO;

   size = lv->lv_current_le * sizeof ( pe_t);
   if ( ( pe = vmalloc ( size)) == NULL) {
      printk ( KERN_CRIT
               "%s -- do_lv_extend_reduce: vmalloc error LV_CURRENT_PE "
               "of %d Byte\n", lvm_name, size);
      return -ENOMEM;
   }

#ifdef VERIFY_READ
   /* verify size of PE structures in user space */
   if ( ( ret = verify_area ( VERIFY_READ, pep, size)) != 0) {
      vfree ( pe);
      return ret;
   }
#endif

#ifdef DEBUG
   printk ( KERN_DEBUG
            "%s -- fsync_dev and "
            "invalidate_buffers for %s [%s] in %s\n",
            lvm_name, vg[VG(minor)]->lv[l]->lv_name,
            kdevname ( vg[VG(minor)]->lv[l]->lv_dev),
            vg[VG(minor)]->vg_name);
#endif
   vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
   fsync_dev ( vg[VG(minor)]->lv[l]->lv_dev);
   vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
   invalidate_buffers ( vg[VG(minor)]->lv[l]->lv_dev);

   /* reduce allocation counters */
   for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
      vg[VG(minor)]->pe_allocated--;
      for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
         if (  vg[VG(minor)]->pv[p]->pv_dev ==
               vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
            vg[VG(minor)]->pv[p]->pe_allocated--;
      }
   }

#ifdef DEBUG_VFREE
   printk ( KERN_DEBUG "%s -- vfree %d\n", lvm_name, __LINE__);
#endif

   /* save pointer to "old" lv/pe pointer array */
   pep1 = vg[VG(minor)]->lv[l]->lv_current_pe;
   end  = vg[VG(minor)]->lv[l]->lv_current_le;

   lv_open = vg[VG(minor)]->lv[l]->lv_open;

   /* copy preloaded LV */
   lvm_memcpy ( ( char*) vg[VG(minor)]->lv[l], ( char*) lv, sizeof ( lv_t));
   lv_status_save = vg[VG(minor)]->lv[l]->lv_status;
   vg[VG(minor)]->lv[l]->lv_status |= LV_SPINDOWN;
   vg[VG(minor)]->lv[l]->lv_status &= ~LV_ACTIVE;
   vg[VG(minor)]->lv[l]->lv_current_pe = pe;

   /* get the PE structures from user space */
   memcpy_fromfs ( vg[VG(minor)]->lv[l]->lv_current_pe, pep, size);

   vg[VG(minor)]->lv[l]->lv_open = lv_open;

   /* Check what last le shall be used */
   if ( end > vg[VG(minor)]->lv[l]->lv_current_le)
      end = vg[VG(minor)]->lv[l]->lv_current_le;

   /* save availiable i/o statistic data */
   for ( le = 0; le < end; le++) {
      vg[VG(minor)]->lv[l]->lv_current_pe[le].reads  = pep1[le].reads;
      vg[VG(minor)]->lv[l]->lv_current_pe[le].writes = pep1[le].writes;
   }
   vfree ( pep1); pep1 = NULL;


   /* extend the PE count in PVs */
   for ( le = 0; le < vg[VG(minor)]->lv[l]->lv_allocated_le; le++) {
      vg[VG(minor)]->pe_allocated++;
      for ( p = 0; p < vg[VG(minor)]->pv_cur; p++) {
         if ( vg[VG(minor)]->pv[p]->pv_dev ==
              vg[VG(minor)]->lv[l]->lv_current_pe[le].dev)
            vg[VG(minor)]->pv[p]->pe_allocated++;
      }
   }

   lvm_gendisk.part[minor+l+1].start_sect = 0;
   lvm_gendisk.part[minor+l+1].nr_sects = vg[VG(minor)]->lv[l]->lv_size;
   lvm_size[minor+l+1] = vg[VG(minor)]->lv[l]->lv_size >> 1;

   vg[VG(minor)]->lv[l]->lv_status = lv_status_save;

   return 0;
}


/*
 * support function initialize gendisk variables
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1 ,0)
__initfunc ( static void lvm_geninit ( struct gendisk *lvm_gdisk))
#else
static void lvm_geninit ( struct gendisk *lvm_gdisk)
#endif
{
   int i = 0;

#ifdef DEBUG_GENDISK
   printk ( KERN_DEBUG "%s -- lvm_gendisk\n", lvm_name);
#endif

   for ( i = 0; i < MAX_VG * ( MAX_LV + 1); i++) {
      lvm_gendisk.part[i].start_sect = -1; /* avoid partition check */
      lvm_size[i] = lvm_gendisk.part[i].nr_sects = 0;
      lvm_blocksizes[i] = BLOCK_SIZE;
   }

   blksize_size[LVM_BLOCK_MAJOR] = lvm_blocksizes;
   blk_size[LVM_BLOCK_MAJOR] = lvm_size;

   return;
}


#ifdef LVM_GET_INODE
/*
 * support function to get an empty inode
 *
 * Gets an empty inode to be inserted into the inode hash,
 * so that a physical volume can't be mounted.
 * This is analog to drivers/block/md.c
 *
 * Is this the real thing?
 *
 */
struct inode *lvm_get_inode ( int dev) {
   struct inode *inode_this = NULL;

   /* Lock the device analog MD by inserting a dummy inode. */
   inode_this = get_empty_inode ();
   inode_this->i_dev = dev;
   insert_inode_hash ( inode_this);
   return ( inode_this);
}
#endif


inline int lvm_strlen ( char *s1) {
   int len = 0;

   while ( s1[len] != 0) len++;
   return len;
}


inline int lvm_strcmp ( char *s1, char *s2) {
   while ( *s1 != 0 && *s2 != 0) {
      if ( *s1 != *s2) return -1;
      s1++; s2++;
   }
   if ( *s1 == 0 && *s2 == 0) return 0;
   return -1;
}


inline char *lvm_strrchr ( char *s1, char c) {
   char *s2 = NULL;

   while ( *s1 != 0) {
      if ( *s1 == c) s2 = s1;
      s1++;
   }
   return s2;
}


inline void lvm_memcpy ( char *dest, char *source, int size) {
   for ( ;size > 0; size--) {
      *dest = *source;
      dest++;
      source++;
   }
}
/* END internal support functions */
