/*
 * tools/lib/pv_write.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * March 1997
 * August-September 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with GNU CC; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    08/30/1998 - seperated disk and core pv structure in pv_write()
 *                 by using pv_copy_to_disk()
 *    09/06/1998 - implemented data layer in pv_write_pe() by using
 *                 new function pe_copy_to_disk()
 *
 */

#include <liblvm.h>

int pv_write ( char *pv_name, pv_t *pv) {
   int pv_handle = -1;
   int ret = 0;
   pv_disk_t *pv_disk = NULL;

#ifdef DEBUG
   debug ( "pv_write -- CALLED with %s %X\n", pv_name, (uint) pv);
#endif

   if ( pv_name == NULL ||
        pv_check_name ( pv_name) < 0 ||
        pv == NULL ||
        pv_check_consistency ( pv) < 0) return -LVM_EPARAM;

   pv_disk = pv_copy_to_disk ( pv);

   if ( ( pv_handle = open ( pv_name, O_WRONLY)) == -1)
      ret = -LVM_EPV_STORE_OPEN;
   else if ( lseek ( pv_handle, pv->pv_on_disk.base, SEEK_SET) !=
             pv->pv_on_disk.base) ret = -LVM_EPV_STORE_LSEEK;
   else if ( write ( pv_handle, pv_disk,
                     sizeof ( pv_disk_t)) != sizeof ( pv_disk_t))
      ret = -LVM_EPV_STORE_WRITE;

   free ( pv_disk);

   if ( pv_handle != -1) {
      fsync ( pv_handle);
      close ( pv_handle);
   }

#ifdef DEBUG
   debug ( "pv_write -- LEAVING\n");
#endif
   return ret;
}


int pv_write_namelist ( char *pv_name, vg_t *vg) {
   int p = 0;
   int pv_handle = -1;
   int ret = 0;
   int size = 0;
   char *pv_name_list = NULL;

#ifdef DEBUG
   debug ( "pv_write_namelist -- CALLED\n");
#endif

   if ( pv_name == NULL || pv_check_name ( pv_name) < 0 ||
        vg == NULL || vg_check_name ( vg->vg_name) < 0) return -LVM_EPARAM;

   if ( ( pv_handle = open ( pv_name, O_WRONLY)) == -1)
      ret = -LVM_EPV_STORE_NAMELIST_OPEN;
   else if ( lseek ( pv_handle, vg->pv[0]->pv_namelist_on_disk.base,
                     SEEK_SET) != vg->pv[0]->pv_namelist_on_disk.base)
      ret = -LVM_EPV_STORE_NAMELIST_LSEEK;
   else {
      size = vg->pv_cur * NAME_LEN;
      if ( ( pv_name_list = malloc ( size)) == NULL) {
         fprintf ( stderr, "malloc error in %s [line %d]\n",
                           __FILE__, __LINE__);
         ret = LVM_EPV_STORE_NAMELIST_MALLOC;
      } else {
         memset ( pv_name_list, 0, size);
         for ( p = 0; p < vg->pv_cur; p++) {
            if ( vg->pv[p] != NULL)
               strcpy ( &pv_name_list[p*NAME_LEN], vg->pv[p]->pv_name);
         }
         if ( write ( pv_handle, pv_name_list, size) != size)
            ret = -LVM_EPV_STORE_NAMELIST_WRITE;
         free ( pv_name_list);
      }
   }

   if ( pv_handle != -1) {
      fsync ( pv_handle);
      close ( pv_handle);
   }

#ifdef DEBUG
   debug ( "pv_write_namelist -- LEAVING\n");
#endif
   return ret;
}


int pv_write_all_pv_of_vg ( vg_t *vg) {
   int p = 0;
   int ret = 0;

#ifdef DEBUG
   debug ( "pv_write_all_pv_of_vg -- CALLED with vg->vg_name: %s "
            " vg->pv_cur: %lu\n", vg->vg_name, vg->pv_cur);
#endif

   if ( vg == NULL ||
        vg_check_name ( vg->vg_name) < 0) return -LVM_EPARAM;

   for ( p = 0; p < vg->pv_cur; p++)
      if ( ( ret = pv_write ( vg->pv[p]->pv_name, vg->pv[p])) < 0) break;

#ifdef DEBUG
   debug ( "pv_write_all_pv_of_vg -- LEAVING\n");
#endif
   return 0;
}


int pv_write_pe ( char *pv_name, pv_t *pv) {
   int pv_handle = -1;
   int ret  = 0;
   uint size = 0;
   disk_pe_t *pe_disk = NULL;

#ifdef DEBUG
   debug ( "pv_write_pe -- CALLED  pv->pe_total: %lu\n", pv->pe_total);
#endif

   if ( pv_name == NULL ||
        pv_check_name ( pv_name) < 0 ||
        pv == NULL) return -LVM_EPARAM;

   size = pv->pe_total * sizeof ( disk_pe_t);
   if ( size + pv->pe_on_disk.base > LVM_DISK_SIZE ( pv))
      return -LVM_EPV_STORE_PE_SIZE;;
   if ( ( pv_handle = open ( pv_name, O_WRONLY)) == -1)
      ret = -LVM_EPV_STORE_PE_OPEN;
   else if ( lseek ( pv_handle,  pv->pe_on_disk.base, SEEK_SET) !=
              pv->pe_on_disk.base)
      ret = -LVM_EPV_STORE_PE_LSEEK;
   else {
      pe_disk = pe_copy_to_disk ( pv->pe, pv->pe_total);
      if ( write ( pv_handle, pe_disk, size) != size)
         ret = -LVM_EPV_STORE_PE_WRITE;
      free ( pe_disk);
   }

   if ( pv_handle != -1) {
      fsync ( pv_handle);
      close ( pv_handle);
   }

#ifdef DEBUG
   debug ( "pv_write_pe -- LEAVING\n");
#endif
   return ret;
}


int pv_write_with_pe ( char *pv_name, pv_t *pv) {
   int ret = 0;

#ifdef DEBUG
   debug ( "pv_write_with_pe -- CALLED with pv->pe_total: %lu\n",
            pv->pe_total);
#endif

   if ( pv_name == NULL ||
        pv_check_name ( pv_name) < 0 ||
        pv == NULL) return -LVM_EPARAM;

   if ( ( ret = pv_write ( pv_name, pv)) == 0) ret = pv_write_pe ( pv_name, pv);

#ifdef DEBUG
   debug ( "pv_write_with_pe -- LEAVING\n");
#endif
   return ret;
}
