/* Copyright (c) 1987, 1988  Stanley T. Shebs. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* The "side" structure is the repository of information about players. */
/* Surprisingly, there is not much code to manipulate side directly.  */
/* Viewing code is somewhat tricky, since any hex may be viewed by any */
/* number of sides at once. */

#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "global.h"

char *reasonnames[] = REASONNAMES;	/* names of columns in unit record */

Side neutral_placeholder;	/* just has a pointer to next side and the neutral units. */
Side sides[MAXSIDES];		/* array containing all sides (not very many) */
Side *sidelist;			/* head of list of all sides */
Side *tmpside;			/* temporary used in many places */
int numsides;			/* number of sides in the game */

/* Reset any side structures that need it. */

void init_sides()
{
    int i;

    for (i = 0; i < MAXSIDES; ++i) {
	sides[i].name = NULL;
	sides[i].unithead = NULL;
    }
    sidelist = NULL;
    numsides = 0;
}

/* Given a side, get its relative position in array of sides (the "number"). */
/* Neutrals are -1, for lack of any better ideas. */

int side_number(Side *side) {

    return (side == NULL ? MAXSIDES : (side - sides));
}

/* A formal alliance requires the agreement of both sides. */

bool allied_side(Side *s1, Side *s2) {

    if (s1 == s2) return TRUE;
    return (s1 != NULL && s2 != NULL &&
            s1->attitude[side_number(s2)] >= ALLY &&
            s2->attitude[side_number(s1)] >= ALLY);
}

/* Being at war requires only ones of the sides to consider itself so. */

bool enemy_side(Side *s1, Side *s2) {

    if (s1 == s2) return FALSE;
    return (s1 != NULL && s2 != NULL &&
            (s1->attitude[side_number(s2)] <= ENEMY ||
             s2->attitude[side_number(s1)] <= ENEMY));
}

/* Neutralness is basically anything else. */

bool neutral_side(Side *s1, Side *s2) {

    return (!enemy_side(s1, s2) && !allied_side(s1, s2));
}

/* Internal non-noisy function. */

void make_war(Side *side1, Side *side2) {

    side1->attitude[side_number(side2)] = ENEMY;
    side2->attitude[side_number(side1)] = ENEMY;
}

/* Formal declarations of war need to do a transitive closure, as part of */
/* dragging allies in. */

void declare_war(Side *side1, Side *side2) {

    Side *side3;

    notify_all("The %s and the %s have declared war!!",
               copy_string(plural_form(side1->name)),
               copy_string(plural_form(side2->name)));
    make_war(side1, side2);
    for_all_sides(side3) {
        if (allied_side(side3, side1)) make_war(side3, side2);
        if (allied_side(side3, side2)) make_war(side3, side1);
    }
}

/* Internal non-noisy function. */

void make_neutrality(Side *side1, Side *side2) {

    side1->attitude[side_number(side2)] = NEUTRAL;
    side2->attitude[side_number(side1)] = NEUTRAL;
}

/* Establish neutrality for both sides. */

void declare_neutrality(Side *side1, Side *side2) {

    notify_all("The %s and the %s have agreed to neutrality.",
               copy_string(plural_form(side1->name)),
               copy_string(plural_form(side2->name)));
    make_neutrality(side1, side2);
}

/* Internal non-noisy function. */

void make_alliance(Side *side1, Side *side2) {

    if (side1 != side2) {
        side1->attitude[side_number(side2)] = ALLY;
        side2->attitude[side_number(side1)] = ALLY;
    }
}

/* Establish the alliance for both sides, then extend it to include */
/* every other ally (only need one pass over sides to ensure transitive */
/* closure, because alliances formed one at a time). */

void declare_alliance(Side *side1, Side *side2) {

    Side *side3;

    notify_all("The %s and the %s enter into an alliance.",
               copy_string(plural_form(side1->name)),
               copy_string(plural_form(side2->name)));
    make_alliance(side1, side2);
    for_all_sides(side3) {
        if (allied_side(side3, side1)) make_alliance(side3, side2);
        if (allied_side(side3, side2)) make_alliance(side3, side1);
    }
}

/* Initialize basic viewing structures for a side, in preparation for the */
/* placement of units. */

void init_view(Side *side) {

    int x, y, cov;
    viewdata seen;

    cov = (period.allseen ? 100 : 0);
    seen = (FALSE /*(period.allseen || world.known)*/ ? EMPTY : UNSEEN);
    for (x = 0; x < world.width; ++x) {
        for (y = 0; y < world.height; ++y) {
            set_cover(side, x, y, cov);
            side_view(side, x, y) = seen;
#ifdef PREVVIEW
            side_prevview(side, x, y) = UNSEEN;
#endif
        }
    }
}

/* Add the new side to the end of the list of sides - this keeps our */
/* list traversals going from top to bottom (the things we do to keep */
/* users happy...). */

void link_in_side(Side *side) {

    Side *head, *last;

    if (sidelist == NULL) {
        sidelist = side;
        neutral_placeholder.next = side;
    } else {
        for_all_sides(head) {
            if (head->next == NULL) last = head;
        }
        last->next = side;
    }
    side->next = NULL;
    side->unithead = create_unit(-1, (char *) NULL);
}

/* Create an object representing a side. Checking to make sure all human */
/* players have displays has been done by now, so problems mean bugs. */

Side *create_side(char *name, bool person, char *host) {

    int s, i;
    Side *newside;

    /* Can't have humans without displays */
    if (person && host == NULL) abort();
    if (name == NULL) name = "???";
    for (s = 0; s < MAXSIDES; ++s) {
        if (sides[s].name == NULL) {
            newside = &(sides[s]);
            newside->name = copy_string(name);
            newside->humanp = person;
            if (host == NULL || strcmp(host, "*") == 0) {
                newside->host = NULL;
            } else {
                newside->host = copy_string(host);
            }
            newside->lost = FALSE;
            for_all_unit_types(i) {
                newside->counts[i] = 1;
                newside->unitslist[i].units = 0;
                newside->unitslist[i].building = 0;
            }
            for_all_resource_types(i) {
                newside->resources[i] = 0;
            }
            newside->appdata.showmode = BORDERHEX; /* overridden by X11.c
                                              get_resources() */
            newside->timetaken = 0;
            newside->timedout = FALSE;
            strcpy(newside->timetakenbuf, "00:00:00");
            newside->itertime = 100;
            newside->startbeeptime = DEFAULT_STARTBEEPTIME;
            newside->followaction = FALSE;
            newside->view =
                (viewdata *) malloc(world.width*world.height*sizeof(viewdata));
#ifdef PREVVIEW
            newside->prevview =
                (viewdata *) malloc(world.width*world.height*sizeof(viewdata));
            newside->viewtimestamp =
                (short *) malloc(world.width*world.height*sizeof(short));
#endif
            newside->coverage =
                (short *) malloc(world.width*world.height*sizeof(short));
            init_view(newside);
            newside->deadunits = NULL;
            newside->display = NULL;
            newside->side_dialog = NULL;
            newside->dialog = NULL;
            link_in_side(newside);
            ++numsides;
            return newside;
        }
    }
    fprintf(stderr, "Cannot have more than %d sides total!\n", MAXSIDES);
    abort();
    return NULL;
}

void set_side_view(Side *s, int x, int y, viewdata v) {

#ifdef PREVVIEW
  s->viewtimestamp[world.width*y+x] = global.time;
/*   if (v != side_view(s,x,y)) */
    s->prevview[world.width*y+x] = side_view(s,x,y);
#endif
  s->view[world.width*y+x] = v;
}

/* The inverse function - given a number, figure out which side it is. */
/* Return NULL for failure; hopefully callers will check on this! */

Side *side_n(int n) {

    return ((n >= 0 && n < numsides) ? &sides[n] : NULL);
}

/* Utility to clean up images of units from a lost side. */

void remove_images(Side *side, int n) {

    int x, y;
    viewdata view;

    for (x = 0; x < world.width; ++x) {
        for (y = 0; y < world.height; ++y) {
            view = side_view(side, x, y);
            if (view != EMPTY && view != UNSEEN && vside(view) == n) {
                set_side_view(side, x, y, EMPTY);
                draw_hex(side, x, y, TRUE);
            }
        }
    }
}
