// (c) 2004 Max Howell (max.howell@methylblue.com)
// See COPYING file for licensing information

#ifndef ANALYZER_H
#define ANALYZER_H

#ifdef __FreeBSD__
   #include <sys/types.h>
#endif

#include "fht.h"
#include <tqpixmap.h> //stack allocated and convenience
#include <tqtimer.h>  //stack allocated
#include <tqwidget.h> //baseclass
#include <vector>    //included for convenience

namespace Analyzer
{
   typedef std::vector<float> Scope;

   template<class W> class Base : public W
   {
   public:
      uint timeout() const { return m_timeout; }

   protected:
      Base( TQWidget*, uint, uint = 7 );
      ~Base() { delete m_fht; }

      void drawFrame();
      virtual void transform( Scope& ) = 0;
      virtual void analyze( const Scope& ) = 0;
      virtual void demo();

   private:
      virtual bool event( TQEvent* );

   protected:
      TQTimer m_timer;
      uint   m_timeout;
      FHT   *m_fht;
   };

   class Base2D : public Base<TQWidget>
   {
   TQ_OBJECT
   public:
      const TQPixmap *background() const { return &m_background; }
      const TQPixmap *canvas() const { return &m_canvas; }

   private slots:
      void draw() { drawFrame(); bitBlt(this, 0, 0, canvas()); }

   protected:
      Base2D( TQWidget*, uint timeout, uint scopeSize = 7 );

      TQPixmap *background() { return &m_background; }
      TQPixmap *canvas() { return &m_canvas; }

      void eraseCanvas()
      {
        bitBlt(canvas(), 0, 0, background());
      }

      void paintEvent( TQPaintEvent* ) override { if( !m_canvas.isNull() ) bitBlt( this, 0, 0, canvas() ); }
      void resizeEvent( TQResizeEvent* ) override;
      void paletteChange( const TQPalette& ) override;

   private:
      TQPixmap m_background;
      TQPixmap m_canvas;
   };

   class Block : public Analyzer::Base2D
   {
   public:
      explicit Block( TQWidget* );

      static constexpr int HEIGHT      = 2;
      static constexpr int FADE_SIZE   = 90;
      static constexpr int MIN_COLUMNS = 32;
      static constexpr int MAX_COLUMNS = 256;
      static constexpr int MIN_ROWS    = 3;
      static constexpr int WIDTH       = 4;

   protected:
      virtual void transform( Analyzer::Scope& );
      virtual void analyze( const Analyzer::Scope& );
      void paletteChange(const TQPalette&) override;
      void resizeEvent(TQResizeEvent *) override;

      void determineStep();
      void drawBackground();

   private:
      TQPixmap *bar()
      {
         return &m_barPixmap;
      }

     // So we don't create a vector each frame.
     Scope m_scope;
     TQPixmap m_barPixmap;
     TQPixmap m_topBarPixmap;

     // Current bar heights
     std::vector<float> m_store;
     std::vector<float> m_yScale;

     std::vector<TQPixmap> m_fadeBars;
     std::vector<int>      m_fadeIntensity;
     std::vector<unsigned> m_fadePos;

     // Number of rows and columns of blocks
     unsigned m_columns;
     unsigned m_rows;

     // y-offset from top of widget
     unsigned m_y;

     // rows to fall per-step
     float m_step;
   };

   void interpolate(const Scope&, Scope&);
}

#endif
