/***************************************************************************
 *   Copyright (C) 2005 by David Saxton                                    *
 *   david@bluehaze.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "solidshape.h"
#include "libraryitem.h"
#include "resizeoverlay.h"

#include <cmath>
#include <kiconloader.h>
#include <tdelocale.h>
#include <tqpainter.h>


//BEGIN class DPRectangle
Item * DPRectangle::construct( ItemDocument *itemDocument, bool newItem, const char *id )
{
	return new DPRectangle( itemDocument, newItem, id );
}

LibraryItem* DPRectangle::libraryItem()
{
	return new LibraryItem(
			TQString("dp/rectangle"),
	i18n("Rectangle"),
	i18n("Other"),
	TDEGlobal::iconLoader()->loadIcon( "text", TDEIcon::Small ),
	LibraryItem::lit_drawpart,
	DPRectangle::construct );
}

DPRectangle::DPRectangle( ItemDocument *itemDocument, bool newItem, const char *id )
	: DrawPart( itemDocument, newItem, id ? id : "rectangle" )
{
	m_pRectangularOverlay = new RectangularOverlay(this);
	m_name = i18n("Rectangle");
	
	createProperty( "background", Variant::Type::Bool );
	property("background")->setValue(false);
	property("background")->setCaption( i18n("Display Background") );
	property("background")->setAdvanced(true);
	
	createProperty( "background-color", Variant::Type::Color );
	property("background-color")->setValue(TQt::white);
	property("background-color")->setCaption( i18n("Background Color") );
	property("background-color")->setAdvanced(true);
	
	createProperty( "line-color", Variant::Type::Color );
	property("line-color")->setValue(TQt::black);
	property("line-color")->setCaption( i18n("Line Color") );
	property("line-color")->setAdvanced(true);
	
	createProperty( "line-width", Variant::Type::Int );
	property("line-width")->setCaption( i18n("Line Width") );
	property("line-width")->setMinValue(1);
	property("line-width")->setMaxValue(1000);
	property("line-width")->setValue(1);
	property("line-width")->setAdvanced(true);
	
	createProperty( "line-style", Variant::Type::PenStyle );
	property("line-style")->setAdvanced(true);
	setDataPenStyle( "line-style", TQt::SolidLine );
}

DPRectangle::~DPRectangle()
{
}

void DPRectangle::setSelected( bool yes )
{
	if ( yes == isSelected() )
		return;
	
	DrawPart::setSelected(yes);
	m_pRectangularOverlay->showResizeHandles(yes);
}


void DPRectangle::dataChanged()
{
	bool displayBackground = dataBool("background");
	TQColor line_color = dataColor("line-color");
	unsigned width = unsigned( dataInt("line-width") );
	TQt::PenStyle style = getDataPenStyle("line-style");
	
	setPen( TQPen( line_color, width, style ) );
	
	if (displayBackground)
		setBrush( dataColor("background-color") );
	else
		setBrush( TQt::NoBrush );
	
	postResize();
	update();
}


TQSize DPRectangle::minimumSize() const
{
	int side = TQMAX(16, pen().width()+2);
	return TQSize( side, side );
}


void DPRectangle::postResize()
{
	setItemPoints( m_sizeRect, false );
}


TQRect DPRectangle::drawRect() const
{
	int lw = pen().width();
	
	if ( lw > m_sizeRect.width() )
		lw = m_sizeRect.width();
	
	if ( lw > m_sizeRect.height() )
		lw = m_sizeRect.height();
	
	return TQRect( int(x() + m_sizeRect.x()+lw/2), int(y() + m_sizeRect.y()+lw/2),
				  m_sizeRect.width()-lw, m_sizeRect.height()-lw );
}


void DPRectangle::drawShape( TQPainter & p )
{
	p.drawRect(drawRect());
}
//END class DPRectangle


//BEGIN class DPEllipse
Item * DPEllipse::construct( ItemDocument *itemDocument, bool newItem, const char *id )
{
	return new DPEllipse( itemDocument, newItem, id );
}

LibraryItem* DPEllipse::libraryItem()
{
	return new LibraryItem(
			TQString("dp/ellipse"),
	i18n("Ellipse"),
	i18n("Other"),
	TDEGlobal::iconLoader()->loadIcon( "text", TDEIcon::Small ),
	LibraryItem::lit_drawpart,
	DPEllipse::construct );
}

DPEllipse::DPEllipse( ItemDocument *itemDocument, bool newItem, const char *id )
	: DPRectangle( itemDocument, newItem, id ? id : "ellipse" )
{
	m_name = i18n("Ellipse");
}

DPEllipse::~DPEllipse()
{
}


void DPEllipse::postResize()
{
	TQRect br = m_sizeRect;
	
	// Make octagon that roughly covers ellipse
	TQPointArray pa(8);
	pa[0] = TQPoint( br.x() + br.width()/4,		br.y() );
	pa[1] = TQPoint( br.x() + 3*br.width()/4,	br.y() );
	pa[2] = TQPoint( br.x() + br.width(),		br.y() + br.height()/4 );
	pa[3] = TQPoint( br.x() + br.width(),		br.y() + 3*br.height()/4 );
	pa[4] = TQPoint( br.x() + 3*br.width()/4,	br.y() + br.height() );
	pa[5] = TQPoint( br.x() + br.width()/4,		br.y() + br.height() );
	pa[6] = TQPoint( br.x(),						br.y() + 3*br.height()/4 );
	pa[7] = TQPoint( br.x(),						br.y() + br.height()/4 );
	
	setItemPoints( pa, false );
}


void DPEllipse::drawShape( TQPainter & p )
{
	p.drawEllipse(drawRect());
}
//END class SolidShape


