/*  This file is part of the KDE KMobile library
    Copyright (C) 2003 Helge Deller <deller@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include <sys/types.h>
#include <sys/stat.h>

#include <sys/types.h>
#include <unistd.h>
#include <time.h>

#include <tqregexp.h>

#include <kdebug.h>
#include <tdelocale.h>
#include <kinstance.h>
#include <tdeabc/vcardconverter.h>

#include "tdeio_mobile.h"

#include <tdemacros.h>

using namespace TDEIO;

#define TDEIO_MOBILE_DEBUG_AREA 7126
#define PRINT_DEBUG kdDebug(TDEIO_MOBILE_DEBUG_AREA) << "tdeio_mobile: "

extern "C" { TDE_EXPORT int kdemain(int argc, char **argv); }

/**
 * The main program.
 */
int kdemain(int argc, char **argv)
{
  TDEInstance instance( "tdeio_mobile" );

  PRINT_DEBUG << "Starting " << getpid() << endl;

  if (argc != 4) {
	fprintf(stderr, "Usage tdeio_mobile protocol pool app\n");
	return -1;
  }
  // let the protocol class do its work
  KMobileProtocol slave(argv[2], argv[3]);

  slave.dispatchLoop();

  PRINT_DEBUG << "Done" << endl;
  return 0;
}


/**
 * Initialize the mobile slave
 */
KMobileProtocol::KMobileProtocol(const TQCString &pool, const TQCString &app)
  : SlaveBase( "mobile", pool, app)
{
}

KMobileProtocol::~KMobileProtocol()
{
}

/*
 * getDeviceAndRessource("mobile:/<devicename>/<resource>/...") - split
 */
int KMobileProtocol::getDeviceAndRessource(const TQString &_path,
	TQString &devName, TQString &resource, TQString &devPath, 
	KMobileDevice::Capabilities &devCaps)
{
//  PRINT_DEBUG << TQString("###getDeviceAndRessource### %1\n").arg(_path);
  TQStringList path = TQStringList::split('/', _path, false);

  devName = resource = devPath = TQString();
  devCaps = KMobileDevice::hasNothing;

  if (path.count() >= 1)  { devName = path[0];  path.pop_front(); };
  if (path.count() >= 1)  { resource = path[0]; path.pop_front(); };
  if (path.count() >= 1)  devPath = path.join("/");

  if (devName.isEmpty())
	return 0;

  int _caps = m_dev.capabilities(devName);

  if (resource.isEmpty()) {
	devCaps = (KMobileDevice::Capabilities) _caps;
	return 0;
  }

  for (int i=0; i<31; i++) {
	int cap = 1UL << i;
	if ((_caps & cap) == 0)
		continue;
	TQString capname = m_dev.nameForCap(devName,cap);
	if (capname != resource)
		continue;
	devCaps = (KMobileDevice::Capabilities) cap;
	return 0;
  }

  return TDEIO::ERR_DOES_NOT_EXIST;
}


static
void addAtom(TDEIO::UDSEntry& entry, unsigned int ID, long l, const TQString& s = TQString())
{
	TDEIO::UDSAtom	atom;
	atom.m_uds = ID;
	atom.m_long = l;
	atom.m_str = s;
	entry.append(atom);
}

static
void createDirEntry(TDEIO::UDSEntry& entry, const TQString& name, const TQString& url, const TQString& mime)
{
	entry.clear();
	addAtom(entry, TDEIO::UDS_NAME, 0, name);
	addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFDIR);
	addAtom(entry, TDEIO::UDS_ACCESS, 0500);
	addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, mime);
	addAtom(entry, TDEIO::UDS_URL, 0, url);
	addAtom(entry, TDEIO::UDS_USER, 0, getenv("USER"));
	addAtom(entry, TDEIO::UDS_GROUP, 0, getenv("USER"));
	PRINT_DEBUG << TQString("createDirEntry: File: %1  MIME: %2  URL: %3\n").arg(name).arg(mime).arg(url);
//	addAtom(entry, TDEIO::UDS_SIZE, 0);
	addAtom(entry, TDEIO::UDS_GUESSED_MIME_TYPE, 0, mime);
}

static
void createFileEntry(TDEIO::UDSEntry& entry, const TQString& name, const TQString& url, const TQString& mime, 
		const unsigned long size = 0)
{
	entry.clear();
	addAtom(entry, TDEIO::UDS_NAME, 0, name);
	addAtom(entry, TDEIO::UDS_FILE_TYPE, S_IFREG);
	addAtom(entry, TDEIO::UDS_URL, 0, url);
	addAtom(entry, TDEIO::UDS_ACCESS, 0400);
	addAtom(entry, TDEIO::UDS_USER, 0, getenv("USER"));
	addAtom(entry, TDEIO::UDS_GROUP, 0, getenv("USER"));
	addAtom(entry, TDEIO::UDS_MIME_TYPE, 0, mime);
	if (size) addAtom(entry, TDEIO::UDS_SIZE, size);
	addAtom(entry, TDEIO::UDS_GUESSED_MIME_TYPE, 0, mime);
	PRINT_DEBUG << TQString("createFileEntry: File: %1, Size: %2,  MIME: %3\n").arg(name).arg(size).arg(mime);
}


/**
 * Get the information contained in the URL.
 */
void KMobileProtocol::get(const KURL &url)
{
  PRINT_DEBUG << "###############################\n";
  PRINT_DEBUG << TQString("get(%1)\n").arg(url.path());

  KMobileDevice::Capabilities devCaps;
  TQString devName, resource, devPath;

  int err = getDeviceAndRessource(url.path(), devName, resource, devPath, devCaps);
  if (err) {
	error(err, url.path());
	return;
  }

  if (devName.isEmpty() || resource.isEmpty()) {
	error(TDEIO::ERR_DOES_NOT_EXIST, url.path());
	return;
  }
  
  // collect the result
  TQCString result;
  TQString mime;
  switch (devCaps) {
    case KMobileDevice::hasAddressBook:	err = getVCard(devName, result, mime, devPath);
					break;
    case KMobileDevice::hasCalendar:	err = getCalendar(devName, result, mime, devPath);
					break;
    case KMobileDevice::hasNotes:	err = getNote(devName, result, mime, devPath);
					break;
    case KMobileDevice::hasFileStorage: err = getFileStorage(devName, result, mime, devPath);
					break;
    default:				err = TDEIO::ERR_CANNOT_ENTER_DIRECTORY; /* TODO */
  }

  if (err) {
	error(err, url.path());
	return;
  }

  // tell the mimetype
  mimeType(mime);

  // tell the length
  TDEIO::filesize_t processed_size = result.length();
  totalSize(processed_size);

  // tell the contents of the URL
  TQByteArray array;
  array.setRawData( result.data(), result.length() );
  data(array);
  array.resetRawData( result.data(), result.length() );
  processedSize( processed_size );
  // tell we are finished
  data(TQByteArray());

  // tell we are finished
  finished();
}


/*
 * listRoot() - gives listing of all devices
 */
void KMobileProtocol::listRoot(const KURL& url)
{
  PRINT_DEBUG << TQString("########## listRoot(%1) for %2:/\n").arg(url.path()).arg(url.protocol());

  TDEIO::UDSEntry entry;

  TQStringList deviceNames = m_dev.deviceNames();
  unsigned int dirs = deviceNames.count();
  totalSize(dirs);

  int classMask = KMobileDevice::Unclassified;
  /* handle all possible protocols here and just add a <protocol>.protocol file */
  if (url.protocol() == "cellphone")		// cellphone:/
	classMask = KMobileDevice::Phone;
  if (url.protocol() == "organizer" ||		// organizer:/
      url.protocol() == "pda")			// pda:/
	classMask = KMobileDevice::Organizer;
  if (url.protocol() == "phonecamera")		// camera:/
	classMask = KMobileDevice::Camera;

  for (unsigned int i=0; i<dirs; i++) {

	TQString devName = deviceNames[i];

	if (classMask != KMobileDevice::Unclassified && 
	    m_dev.classType(devName) != classMask)
		continue;	

        createDirEntry(entry, devName, "mobile:/"+devName, 
			KMOBILE_MIMETYPE_DEVICE_KONQUEROR(devName));
        listEntry(entry, false);

	processedSize(i+1);
  }
  listEntry(entry, true);
  finished();
}


/*
 * folderMimeType() - returns mimetype of the folder itself
 */
TQString KMobileProtocol::folderMimeType(int cap)
{
  TQString mimetype;
  switch (cap) {
    case KMobileDevice::hasAddressBook:	mimetype = KMOBILE_MIMETYPE_INODE "addressbook";
					break;
    case KMobileDevice::hasCalendar:	mimetype = KMOBILE_MIMETYPE_INODE "calendar";
					break;
    case KMobileDevice::hasNotes:	mimetype = KMOBILE_MIMETYPE_INODE "notes";
					break;
    case KMobileDevice::hasFileStorage:
    default:				mimetype = "inode/directory";
  }
  return mimetype;
}

/*
 * entryMimeType() - returns mimetype of the entries in the given folder
 */
TQString KMobileProtocol::entryMimeType(int cap)
{
  TQString mimetype;
  switch (cap) {
    case KMobileDevice::hasAddressBook:	mimetype = "text/x-vcard";
					break;
    case KMobileDevice::hasCalendar:	mimetype = "text/x-vcalendar";
					break;
    case KMobileDevice::hasNotes:	mimetype = "text/plain";
					break;
    case KMobileDevice::hasFileStorage:
    default:				mimetype = "text/plain";
  }
  return mimetype;
}

/*
 * listTopDeviceDir("mobile:/<devicename>") - sub-directory of a devices
 */

void KMobileProtocol::listTopDeviceDir(const TQString &devName)
{
  PRINT_DEBUG << TQString("listTopDeviceDir(%1)\n").arg(devName);

  TDEIO::UDSEntry entry;
  unsigned int caps = m_dev.capabilities(devName);

  for (int i=0; i<31; i++) {
	unsigned int cap = 1UL<<i;
	if ((caps & cap) == 0)
		continue;

	TQString filename = m_dev.nameForCap(devName, cap);
	TQString mimetype = folderMimeType(cap);
	
        createDirEntry(entry, filename, TQString("mobile:/%1/%2/").arg(devName).arg(filename), mimetype);
        listEntry(entry, false);
  }
  listEntry(entry, true);
  finished();
}


/*
 * listEntries("mobile:/<devicename>/<resource>") - resources of a device
 */
void KMobileProtocol::listEntries(const TQString &devName, 
	const TQString &resource, const TQString &devPath,
	const KMobileDevice::Capabilities devCaps)
{
  PRINT_DEBUG << TQString("listEntries(%1,%2,%3)\n").arg(devName).arg(resource).arg(devPath);
  switch (devCaps) {
    case KMobileDevice::hasAddressBook:	listAddressBook(devName, resource);
					break;
    case KMobileDevice::hasCalendar:	listCalendar(devName, resource);
					break;
    case KMobileDevice::hasNotes:	listNotes(devName, resource);
					break;
    case KMobileDevice::hasFileStorage: listFileStorage(devName, resource, devPath);
					break;
    default:				error( ERR_CANNOT_ENTER_DIRECTORY,
					  TQString("/%1/%2").arg(devName).arg(resource) );
  }
}

/*
 * listAddressBook("mobile:/<devicename>/Addressbook) - list the addressbook
 */
void KMobileProtocol::listAddressBook(const TQString &devName, const TQString &resource)
{
  PRINT_DEBUG << TQString("listAddressBook(%1)\n").arg(devName);
  
  TDEIO::UDSEntry entry;

  int fieldwidth;
  int entries = m_dev.numAddresses(devName);
  if (entries>=1000) fieldwidth=4; else 
   if (entries>=100) fieldwidth=3; else
    if (entries>=10) fieldwidth=2; else fieldwidth=1;
  totalSize(entries);
//  TQRegExp rx; rx.setPattern( ".*FN:([\\w\\s]*)[\\n\\r]{2}.*" );
  TQString name;
  for (int i=0; i<entries; i++) {

#if 0
  	TQString content = m_dev.readAddress(devName, i);
        if ( rx.search( content ) < 0 )
		name = TQString();
	else
		name = "_" + rx.cap(1);
#endif

	TQString filename = TQString("%1%2.vcf").arg(i,fieldwidth).arg(name);
	for (int p=0; p<fieldwidth; p++) { 
		if (filename[p]==' ') filename[p]='0'; else break; 
	}
	TQString url = TQString("mobile:/%1/%2/%3").arg(devName).arg(resource).arg(filename);

	createFileEntry(entry, filename, url, entryMimeType(KMobileDevice::hasAddressBook),
			400 /*content.utf8().length()*/ );
        listEntry(entry, false);

	processedSize(i+1);
  }
  listEntry(entry, true);
  finished();
}

/*
 * getVCard() - gives the vCard of the given file
 */
int KMobileProtocol::getVCard( const TQString &devName, TQCString &result, TQString &mime, const TQString &path )
{
  PRINT_DEBUG << TQString("getVCard(%1)\n").arg(path);

  int index = path.find('.');
  if (index>0)
	index = path.left(index).toInt();
  if (index<0 || index>=m_dev.numAddresses(devName))
	return TDEIO::ERR_DOES_NOT_EXIST;

  TQString str = m_dev.readAddress(devName, index);
  if (str.isEmpty())
	return TDEIO::ERR_INTERNAL;
  result = str.utf8();
  mime = entryMimeType(KMobileDevice::hasAddressBook);
//  setMetaData("plugin", "const TQString &key, const TQString &value);
  return 0;
}

/*
 * listCalendar("mobile:/<devicename>/Calendar) - list the calendar entries
 */
void KMobileProtocol::listCalendar( const TQString &devName, const TQString &resource)
{
  PRINT_DEBUG << TQString("listCalendar(%1)\n").arg(devName);
  
  TDEIO::UDSEntry entry;

  int entries = m_dev.numCalendarEntries(devName);
  totalSize(entries);
  for (int i=0; i<entries; i++) {

	TQString filename = TQString("%1_%2.vcs").arg(i).arg(i18n("calendar"));
	TQString url = TQString("mobile:/%1/%2/%3").arg(devName).arg(resource).arg(filename);

	createFileEntry(entry, filename, url, entryMimeType(KMobileDevice::hasCalendar));
        listEntry(entry, false);

	processedSize(i+1);
  }
  listEntry(entry, true);
  finished();
}

/*
 * getCalendar() - reads a calendar entry
 */
int KMobileProtocol::getCalendar( const TQString &devName, TQCString &result, TQString &mime, const TQString &path)
{
  PRINT_DEBUG << TQString("getCalendar(%1, #%2)\n").arg(devName).arg(path);

  /* TODO */
  Q_UNUSED(result);
  Q_UNUSED(mime);
  return TDEIO::ERR_CANNOT_ENTER_DIRECTORY;
}


/*
 * listNotes("mobile:/<devicename>/Notes) - list the notes
 */
void KMobileProtocol::listNotes( const TQString &devName, const TQString &resource)
{
  PRINT_DEBUG << TQString("listNotes(%1)\n").arg(devName);
  
  TDEIO::UDSEntry entry;

  int entries = m_dev.numNotes(devName);
  totalSize(entries);
  for (int i=0; i<entries; i++) {

        TQString note /*= m_dev.readNote(devName, i)*/;

	TQString filename = TQString("%1_%2.txt").arg(i).arg(i18n("note"));
	TQString url = TQString("mobile:/%1/%2/%3").arg(devName).arg(resource).arg(filename);

	createFileEntry(entry, filename, url, entryMimeType(KMobileDevice::hasNotes),
			0 /*note.utf8().length()*/);
        listEntry(entry, false);

	processedSize(i+1);
  }
  listEntry(entry, true);
  finished();
}

/*
 * getNote() - gives the Note of the given file
 */
int KMobileProtocol::getNote( const TQString &devName, TQCString &result, TQString &mime, const TQString &path )
{
  PRINT_DEBUG << TQString("getNote(%1)\n").arg(path);

  int index = path.find('_');
  if (index>0)
	index = path.left(index).toInt();
  if (index<0 || index>=m_dev.numNotes(devName))
	return TDEIO::ERR_DOES_NOT_EXIST;

  TQString note = m_dev.readNote(devName, index);
  if (note.isEmpty())
	return TDEIO::ERR_DOES_NOT_EXIST;

  result = note.utf8();
  mime = entryMimeType(KMobileDevice::hasNotes);
  return 0;
}

/*
 * listFileStorage("mobile:/<devicename>/Files) - list the files on the device
 */
void KMobileProtocol::listFileStorage(const TQString &devName, const TQString &resource, const TQString &devPath)
{
  PRINT_DEBUG << TQString("listFileStorage(%1,%2)\n").arg(devName).arg(devPath);
  
  /* TODO */
  error( TDEIO::ERR_DOES_NOT_EXIST, TQString("/%1/%2/%3").arg(devName).arg(resource).arg(devPath) );
}

/*
 * getFileStorage() - gives the file contents of the given file
 */
int KMobileProtocol::getFileStorage(const TQString &devName, TQCString &result, TQString &mime, const TQString &path)
{
  PRINT_DEBUG << TQString("getFileStorage(%1)\n").arg(path);

  /* TODO */
  Q_UNUSED(devName);
  Q_UNUSED(result);
  Q_UNUSED(mime);
  return TDEIO::ERR_CANNOT_ENTER_DIRECTORY;
}


/**
 * Test if the url contains a directory or a file.
 */
void KMobileProtocol::stat( const KURL &url )
{
  PRINT_DEBUG << "###############################\n";
  PRINT_DEBUG << TQString("stat(%1)\n").arg(url.path());

  KMobileDevice::Capabilities devCaps;
  TQString devName, resource, devPath;

  int err = getDeviceAndRessource(url.path(), devName, resource, devPath, devCaps);
  if (err) {
	error(err, url.path());
	return;
  }

  TQStringList path = TQStringList::split('/', url.path(), false);
  TQString filename = (path.count()>0) ? path[path.count()-1] : "/";
  TQString fullPath = path.join("/");
  TQString fullUrl = TQString("mobile:/%1").arg(fullPath);

  UDSEntry entry;

  bool isDir = devPath.isEmpty();

  if (isDir) {
     createDirEntry(entry, filename, fullUrl, folderMimeType(devCaps));
  } else {
     createFileEntry(entry, filename, fullUrl, entryMimeType(devCaps));
  }

  statEntry(entry);
  finished();
}

/**
 * Get the mimetype.
 */
void KMobileProtocol::mimetype(const KURL &url)
{
  PRINT_DEBUG << "###############################\n";
  PRINT_DEBUG << TQString("mimetype(%1)\n").arg(url.path());

  KMobileDevice::Capabilities devCaps;
  TQString devName, resource, devPath;

  int err = getDeviceAndRessource(url.path(), devName, resource, devPath, devCaps);
  if (err) {
	error(err, url.path());
	return;
  }

  // tell the mimetype
  mimeType(entryMimeType(devCaps));
  finished();
}

/**
 * List the contents of a directory.
 */
void KMobileProtocol::listDir(const KURL &url)
{
  PRINT_DEBUG << "###############################\n";
  PRINT_DEBUG << TQString("listDir(%1)\n").arg(url.path());

  if (!m_dev.isKMobileAvailable()) {
	error( TDEIO::ERR_CONNECTION_BROKEN, i18n("TDE Mobile Device Manager") );
	return;
  }

  KMobileDevice::Capabilities devCaps;
  TQString devName, resource, devPath;

  int err = getDeviceAndRessource(url.path(), devName, resource, devPath, devCaps);
  if (err) {
	error(err, url.path());
	return;
  }

  if (devName.isEmpty()) {
	listRoot(url);
	return;
  }

#if 0  
  if (!dev) {
	error( TDEIO::ERR_DOES_NOT_EXIST, TQString("/%1").arg(devName) );
	return;
  }
#endif

  if (resource.isEmpty()) {
	listTopDeviceDir(devName);
	return;
  }

  listEntries(devName, resource, devPath, devCaps);
}
