/* This file is part of the TDE project
 * Copyright (C) 2025 mio <stigma@disroot.org>
 *
 * Portions of code based off tdefile_mp3.cpp
 *  Copyright (C) 2001, 2002 Rolf Magnus <ramagnus@kde.org>
 *  Copyright (C) 2002 Ryan Cumming <bodnar42@phalynx.dhs.org>
 *  Copyright (C) 2003 Scott Wheeler <wheeler@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include "config.h"
#include "tdefile_mp4.h"

// TQt
#include <tqfile.h>

// TDE
#include <kgenericfactory.h>
#include <kstringvalidator.h>

// TagLib
#include <id3v1genres.h>
#include <id3v2tag.h>
#include <mp4file.h>
#include <tfilestream.h>
#include <tpropertymap.h>
#include <tstring.h>

#define TStringToTQString(s) TQString::fromUtf8(s.toCString(true))

using Mp4Factory = KGenericFactory<TDEMp4Plugin>;

K_EXPORT_COMPONENT_FACTORY(tdefile_mp4, Mp4Factory("tdefile_mp4"))

/**
 * Do translation between KFileMetaInfo items and TagLib::String in a tidy way.
 */
class Translator
{
public:
	Translator(const KFileMetaInfo &info) : m_info(info) {}

	TagLib::String operator[](const char *key) const
	{
		return QStringToTString(m_info["id3"][key].value().toString());
	}

	int toInt(const char *key) const
	{
		return m_info["id3"][key].value().toInt();
	}

private:
	const KFileMetaInfo &m_info;
};

class Validator : public KStringListValidator
{
public:
	Validator(const TQStringList& list, bool rejecting, bool fixupEnabled,
			TQObject *parent, const char *name)
		: KStringListValidator(list, rejecting, fixupEnabled, parent, name)
	{

	}

	TQValidator::State validate(TQString&, int&) const override
	{
		return TQValidator::Acceptable;
	}
};

TDEMp4Plugin::TDEMp4Plugin(TQObject *parent, const char *name, const TQStringList& args)
	: KFilePlugin(parent, name, args)
{
	KFileMimeTypeInfo *info = addMimeTypeInfo(name);

	// ID3 group
	KFileMimeTypeInfo::GroupInfo *group = addGroupInfo(info, "id3", i18n("ID3 Tag"));
	setAttributes(group, KFileMimeTypeInfo::Addable | KFileMimeTypeInfo::Removable);

	KFileMimeTypeInfo::ItemInfo *item;

	item = addItemInfo(group, "Title", i18n("Title"), TQVariant::String);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);
	setHint(item, KFileMimeTypeInfo::Name);

	item = addItemInfo(group, "Artist", i18n("Artist"), TQVariant::String);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);
	setHint(item, KFileMimeTypeInfo::Author);

	item = addItemInfo(group, "Album", i18n("Album"), TQVariant::String);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);

	item = addItemInfo(group, "Date", i18n("Year"), TQVariant::String);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);

	item = addItemInfo(group, "Comment", i18n("Comment"), TQVariant::String);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);
	setHint(item, KFileMimeTypeInfo::Description);

	item = addItemInfo(group, "Tracknumber", i18n("Track"), TQVariant::Int);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);

	item = addItemInfo(group, "Genre", i18n("Genre"), TQVariant::String);
	setAttributes(item, KFileMimeTypeInfo::Modifiable);

	// Technical group.
	group = addGroupInfo(info, "Technical", i18n("Technical Details"));

	item = addItemInfo(group, "Length", i18n("Length"), TQVariant::Int);
	setAttributes(item,  KFileMimeTypeInfo::Cummulative);
	setUnit(item, KFileMimeTypeInfo::Seconds);

	item = addItemInfo(group, "Bitrate", i18n("Average Bitrate"), TQVariant::Int);
	setAttributes(item, KFileMimeTypeInfo::Averaged);
	setHint(item, KFileMimeTypeInfo::Bitrate);
	setSuffix(item, i18n(" kbps"));

	item = addItemInfo(group, "Sample Rate", i18n("Sample Rate"), TQVariant::Int);
	setSuffix(item, i18n("Hz"));

	item = addItemInfo(group, "Channels", i18n("Channels"), TQVariant::Int);
}

TQValidator* TDEMp4Plugin::createValidator(const TQString& mimeType, const TQString& group,
		const TQString& key, TQObject *parent, const char* name) const
{
	if (key == "Tracknumber" || key == "Date")
	{
		return new TQIntValidator(0, 9999, parent, name);
	}

	if (key == "Genre")
	{
		TQStringList genres;
		for (const auto& genre : TagLib::ID3v1::genreList())
		{
			genres.append(TStringToTQString(genre));
		}

		return new Validator(genres, false, true, parent, name);
	}

	return nullptr;
}

bool TDEMp4Plugin::readInfo(KFileMetaInfo& info, unsigned what)
{
	using What = KFileMetaInfo::What;

	bool readID3 = false;
	bool readTech = false;

	if (what & (What::Fastest | What::DontCare | What::ContentInfo))
	{
		readID3 = true;
	}

	if (what & (What::Fastest | What::DontCare | What::TechnicalInfo))
	{
		readTech = true;
	}

	if (!readID3 && !readTech)
	{
		return true;
	}

	if (info.path().isNull())
	{
		// Remote file.
		return false;
	}

	TagLib::MP4::File file(TQFile::encodeName(info.path()), readTech);

	if (!file.isOpen())
	{
		kdWarning(7034) << "Could not open " << file.name() << endl;
		return false;
	}

	if (readID3)
	{
		KFileMetaInfoGroup id3group = appendGroup(info, "id3");
		const TagLib::MP4::Tag* tag = file.tag();

		if (!tag->title().isEmpty())
		{
			appendItem(id3group, "Title", TStringToTQString(tag->title()).stripWhiteSpace());
		}
		if (!tag->artist().isEmpty())
		{
			appendItem(id3group, "Artist", TStringToTQString(tag->artist()).stripWhiteSpace());
		}
		if (!tag->album().isEmpty())
		{
			appendItem(id3group, "Album", TStringToTQString(tag->album()).stripWhiteSpace());
		}
		if (!tag->comment().isEmpty())
		{
			appendItem(id3group, "Comment", TStringToTQString(tag->comment()).stripWhiteSpace());
		}
		if (!tag->genre().isEmpty())
		{
			appendItem(id3group, "Genre", TStringToTQString(tag->genre()).stripWhiteSpace());
		}

		TQString date = (tag->year() > 0) ? TQString::number(tag->year()) : TQString::null;
		TQString track = (tag->track() > 0) ? TQString::number(tag->track()) : TQString::null;

		appendItem(id3group, "Date", date);
		appendItem(id3group, "Tracknumber", track);
	}

	if (readTech)
	{
		KFileMetaInfoGroup techGroup = appendGroup(info, "Technical");
		const TagLib::AudioProperties* audioProperties = file.audioProperties();

		appendItem(techGroup, "Length", audioProperties->lengthInSeconds());
		appendItem(techGroup, "Bitrate", audioProperties->bitrate());
		appendItem(techGroup, "Sample Rate", audioProperties->sampleRate());
		appendItem(techGroup, "Channels", audioProperties->channels());
	}

	return true;
}

bool TDEMp4Plugin::writeInfo(const KFileMetaInfo& info) const
{
	TagLib::ID3v2::FrameFactory::instance()->setDefaultTextEncoding(TagLib::String::UTF8);
	TagLib::FileStream stream(TQFile::encodeName(info.path()), false);

	if (!stream.isOpen() || stream.readOnly())
	{
		kdDebug(7034) << "couldn't open " << info.path() << " for writing" << endl;
		return false;
	}

	TagLib::MP4::File file(&stream, false);

	if (!file.isValid())
	{
		return false;
	}

	Translator t(info);

	file.tag()->setTitle(t["Title"]);
	file.tag()->setArtist(t["Artist"]);
	file.tag()->setAlbum(t["Album"]);
	file.tag()->setYear(t.toInt("Date"));
	file.tag()->setComment(t["Comment"]);
	file.tag()->setTrack(t.toInt("Tracknumber"));
	file.tag()->setGenre(t["Genre"]);

	file.save();

	return true;
}

#include "tdefile_mp4.moc"
