/* kpalmdoc_dlg.cpp
**
** Copyright (C) 2003 by Reinhold Kainhofer
**
** This is the main dialog of the KDE PalmDOC converter.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA 02110-1301, USA.
*/

/*
** Bug reports and questions can be sent to kde-pim@kde.org
*/
#include "options.h"

#include <stdlib.h>
#include <stdio.h>
#include <iostream>

#include <tqtabwidget.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>
#include <tqbuttongroup.h>
#include <tqlabel.h>
#include <tqcombobox.h>

#include <tdelocale.h>
#include <tdeconfig.h>
#include <tdeaboutapplication.h>
#include <tdeapplication.h>
#include <kurlrequester.h>
#include <tdemessagebox.h>
#include <kcharsets.h>

#include <pilotLocalDatabase.h>

#include "kpalmdoc_dlg.h"
#include "kpalmdoc_dlgbase.h"
#include "DOC-converter.h"
#include "kpalmdocSettings.h"


ConverterDlg::ConverterDlg( TQWidget *parent, const TQString& caption)
   : KDialogBase( parent, "converterdialog", false, caption, KDialogBase::Close|KDialogBase::Help|KDialogBase::User1,
	   KDialogBase::Close, true, i18n("&About"))
{
	TQWidget *page = makeHBoxMainWidget();
	dlg=new ConverterDlgBase(page);
	TQStringList l = TDEGlobal::charsets()->descriptiveEncodingNames();
	for ( TQStringList::Iterator it = l.begin(); it != l.end(); ++it)
	{
		dlg->fEncoding->insertItem(*it);
	}

  setHelp("conduit-palmdoc", "kpilot/config-conduits.html");

	readSettings();

	connect(dlg->fDirectories, TQ_SIGNAL(toggled(bool)),
		this, TQ_SLOT(slotDirectories(bool)));
	connect(dlg->fTextToPDB, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotToPDB()));
	connect(dlg->fPDBToText, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotToText()));

	resize(minimumSize());
}

ConverterDlg::~ConverterDlg()
{
    // no need to delete child widgets, TQt does it all for us
}
void ConverterDlg::writeSettings()
{
	// General page
	KPalmDocSettings::setTXTFolder( dlg->fTXTDir->url() );
	KPalmDocSettings::setPDBFolder( dlg->fPDBDir->url() );
	KPalmDocSettings::setSyncFolders( dlg->fDirectories->isChecked() );
	KPalmDocSettings::setAskOverwrite( dlg->fAskOverwrite->isChecked() );
	KPalmDocSettings::setVerboseMessages( dlg->fVerbose->isChecked() );
	KPalmDocSettings::setEncoding( dlg->fEncoding->currentText() );

	// PC->Handheld page
	KPalmDocSettings::setCompress( dlg->fCompress->isChecked() );
	KPalmDocSettings::setConvertBookmarks( dlg->fConvertBookmarks->isChecked() );
	KPalmDocSettings::setBookmarksInline( dlg->fBookmarksInline->isChecked() );
	KPalmDocSettings::setBookmarksEndtags( dlg->fBookmarksEndtags->isChecked() );
	KPalmDocSettings::setBookmarksBmk( dlg->fBookmarksBmk->isChecked() );

	// Handheld->PC page
	KPalmDocSettings::setBookmarksToPC( dlg->fPCBookmarks->id(dlg->fPCBookmarks->selected()) );

	KPalmDocSettings::self()->writeConfig();
}

void ConverterDlg::readSettings()
{
	FUNCTIONSETUP;

	KPalmDocSettings::self()->readConfig();

	// General Page:
	dlg->fTXTDir->setURL(KPalmDocSettings::tXTFolder());
	dlg->fPDBDir->setURL(KPalmDocSettings::pDBFolder());
	bool dir=KPalmDocSettings::syncFolders();
	dlg->fDirectories->setChecked(dir);
	slotDirectories(dir);
	dlg->fAskOverwrite->setChecked( KPalmDocSettings::askOverwrite() );
	dlg->fVerbose->setChecked( KPalmDocSettings::verboseMessages() );
	TQString encoding = KPalmDocSettings::encoding();
#ifdef DEBUG
	DEBUGKPILOT << fname << ": Encoding=" << encoding << endl;
#endif
	dlg->fEncoding->setCurrentText( KPalmDocSettings::encoding() );

	// PC->Handheld page
	dlg->fCompress->setChecked(KPalmDocSettings::compress() );
	dlg->fConvertBookmarks->setChecked(KPalmDocSettings::convertBookmarks());
	dlg->fBookmarksInline->setChecked(KPalmDocSettings::bookmarksInline());
	dlg->fBookmarksEndtags->setChecked(KPalmDocSettings::bookmarksEndtags());
	dlg->fBookmarksBmk->setChecked(KPalmDocSettings::bookmarksBmk());

	// Handheld->PC page
	dlg->fPCBookmarks->setButton(KPalmDocSettings::bookmarksToPC() );
}

void ConverterDlg::slotClose()
{
  writeSettings();
  kapp->quit();
  delete this;
}

void ConverterDlg::slotToText()
{
	FUNCTIONSETUP;
	// First, get the settings from the controls and initialize
	// the converter object
	int bmks=dlg->fPCBookmarks->id(dlg->fPCBookmarks->selected());
	DOCConverter conv;
	switch(bmks) {
		case 0: conv.setBookmarkTypes(DOCConverter::eBmkNone); break;
		case 1: conv.setBookmarkTypes(DOCConverter::eBmkInline); break;
		case 2: conv.setBookmarkTypes(DOCConverter::eBmkEndtags); break;
		case 3: conv.setBookmarkTypes(DOCConverter::eBmkDefaultBmkFile); break;
		default:
			break;
	}

	askOverwrite=dlg->fAskOverwrite->isChecked();
	verbose=dlg->fVerbose->isChecked();


	bool dir=dlg->fDirectories->isChecked();
	TQString txturl=dlg->fTXTDir->url();
	TQString pdburl=dlg->fPDBDir->url();

	TQFileInfo txtinfo(txturl);
	TQFileInfo pdbinfo(pdburl);

	if (dir)
	{
		if (pdbinfo.isFile())
		{
			int res=KMessageBox::questionYesNo(this,
				i18n("<qt>You selected to sync folders, "
				"but gave a filename instead (<em>%1</em>)."
				"<br>Use folder <em>%2</em> instead?</qt>").arg(pdburl)
				.arg(pdbinfo.dirPath(true)), TQString(), i18n("Use Folder"), KStdGuiItem::cancel());
			if (res==KMessageBox::Yes)
			{
				pdburl=pdbinfo.dirPath(true);
				pdbinfo.setFile(pdburl);
			}
			else return;
		}

		if (!pdbinfo.isDir())
		{
			// no directory, so error message and return
			KMessageBox::sorry(this,
				i18n("<qt>The folder <em>%1</em> for "
				"the handheld database files is not a valid "
				"folder.</qt>").arg(pdburl));
			return;
		}

		if (!pdbinfo.exists())
		{
			KMessageBox::sorry(this,
				i18n("<qt>The folder <em>%1</em> for "
				"the handheld database files is not a "
				"valid directory.</qt>").arg(pdburl));
			return;
		}


		// Now check the to directory:
		if (txtinfo.isFile())
		{
			int res=KMessageBox::questionYesNo(this,
				i18n("<qt>You selected to sync folders, "
				"but gave a filename instead (<em>%1</em>)."
				"<br>Use folder <em>%2</em> instead?</qt>").arg(txturl)
				.arg(txtinfo.dirPath(true)), TQString(), i18n("Use Folder"), KStdGuiItem::cancel());
			if (res==KMessageBox::Yes) {
				txturl=txtinfo.dirPath(true);
				txtinfo.setFile(txturl);
			}
			else return;
		}

		// Now that we have a directory path, try to create it:
		if (!txtinfo.isDir()) {
			txtinfo.dir().mkdir(txturl, true);
		}
		if (!txtinfo.isDir()) {
			KMessageBox::sorry(this,
				i18n("<qt>The folder <em>%1</em> for "
				"the text files could not be created.</qt>").arg(txturl));
			return;
		}


		// Now that we have both directories, create the converter object
		DEBUGKPILOT<<"Pdbinfo.dir="<<pdbinfo.dir().absPath()<<endl;
		DEBUGKPILOT<<"txtinfo.dir="<<txtinfo.dir().absPath()<<endl;
		TQStringList pdbfiles(pdbinfo.dir().entryList(CSL1("*.pdb")));
		TQStringList converted_Files;

		DEBUGKPILOT<<"Length of filename list: "<<pdbfiles.size()<<endl;
		for ( TQStringList::Iterator it = pdbfiles.begin(); it != pdbfiles.end(); ++it )
		{
			TQString txtfile=TQFileInfo(*it).baseName(true)+CSL1(".txt");
			DEBUGKPILOT<<"pdbfile="<<*it<<", pdbdir="<<pdburl<<", txtfile="<<txtfile<<", txtdir="<<txturl<<endl;
			if (convertPDBtoTXT(pdburl, *it, txturl, txtfile, &conv))
			{
				converted_Files.append(*it);
			}
		}
		if (converted_Files.size()>0) {
			KMessageBox::informationList(this, i18n("The following texts were "
					"successfully converted:"), converted_Files, i18n("Conversion Successful"));
		}
		else
		{
			KMessageBox::sorry(this, i18n("No text files were converted correctly"));
		}


	} else { // no dir


		// Check the from file
		if (!pdbinfo.isFile() || !pdbinfo.exists())
		{
			KMessageBox::sorry(this, i18n("<qt>The file <em>%1</em> does not "
				"exist.</qt>").arg(pdburl));
			return;
		}

		// Now check the to file
/*		// I can't check if a given filename is a valid filename
		if (!txtinfo.isFile())
		{
			KMessageBox::sorry(this, i18n("<qt>The filename <em>%1</em> for the "
				"text is not a valid filename.</qt>").arg(txturl));
			return;
		}*/
		if (convertPDBtoTXT(pdbinfo.dirPath(true), pdbinfo.fileName(),
				txtinfo.dirPath(true), txtinfo.fileName(), &conv) )
		{
			KMessageBox::information(this, i18n("Conversion of file %1 successful.").arg(pdburl));
		}

	}

}

void ConverterDlg::slotToPDB()
{
	FUNCTIONSETUP;
	// First, get the settings from the controls and initialize
	// the converter object
	bool compress=dlg->fCompress->isChecked();
	int bmks=0;
	if (dlg->fConvertBookmarks->isChecked())
	{
		if (dlg->fBookmarksInline->isChecked()) bmks|=DOCConverter::eBmkInline;
		if (dlg->fBookmarksEndtags->isChecked()) bmks|=DOCConverter::eBmkEndtags;
		if(dlg->fBookmarksBmk->isChecked()) bmks|=DOCConverter::eBmkDefaultBmkFile;
	}
	DOCConverter conv;
	conv.setBookmarkTypes(bmks);
	conv.setCompress(compress);
	conv.setSort(DOCConverter::eSortName);


	askOverwrite=dlg->fAskOverwrite->isChecked();
	verbose=dlg->fVerbose->isChecked();


	bool dir=dlg->fDirectories->isChecked();
	TQString txturl=dlg->fTXTDir->url();
	TQString pdburl=dlg->fPDBDir->url();

	TQFileInfo txtinfo(txturl);
	TQFileInfo pdbinfo(pdburl);

	if (dir)
	{
		if (txtinfo.isFile())
		{
			int res=KMessageBox::questionYesNo(this,
				i18n("<qt>You selected to sync folders, "
				"but gave a filename instead (<em>%1</em>)."
				"<br>Use folder <em>%2</em> instead?</qt>").arg(txturl)
				.arg(txtinfo.dirPath(true)), TQString(), i18n("Use Folder"), KStdGuiItem::cancel());
			if (res==KMessageBox::Yes)
			{
				txturl=txtinfo.dirPath(true);
				txtinfo.setFile(txturl);
			}
			else return;
		}

		if (!txtinfo.isDir() || !txtinfo.exists())
		{
			KMessageBox::sorry(this,
				i18n("<qt>The folder <em>%1</em> for "
				"the text files is not a valid folder.</qt>").arg(txturl));
			return;
		}


		// Now check the to directory:
		if (pdbinfo.isFile())
		{
			int res=KMessageBox::questionYesNo(this,
				i18n("<qt>You selected to sync folders, "
				"but gave a filename instead (<em>%1</em>)."
				"<br>Use folder <em>%2</em> instead?</qt>")
				.arg(pdburl)
				.arg(pdbinfo.dirPath(true)), TQString(), i18n("Use Folder"), KStdGuiItem::cancel());
			if (res==KMessageBox::Yes) {
				pdburl=pdbinfo.dirPath(true);
				pdbinfo.setFile(pdburl);
			}
			else return;
		}

		// Now that we have a directory path, try to create it:
		if (!pdbinfo.isDir()) {
			pdbinfo.dir().mkdir(pdburl, true);
		}
		if (!pdbinfo.isDir()) {
			KMessageBox::sorry(this, i18n("<qt>The folder <em>%1</em> for "
				"the PalmDOC files could not be created.</qt>").arg(pdburl));
			return;
		}


		// Now that we have both directories, create the converter object
		DEBUGKPILOT<<"Pdbinfo.dir="<<pdbinfo.dir().absPath()<<endl;
		DEBUGKPILOT<<"txtinfo.dir="<<txtinfo.dir().absPath()<<endl;
		TQStringList txtfiles(txtinfo.dir().entryList(CSL1("*.txt")));
		TQStringList converted_Files;

		DEBUGKPILOT<<"Length of filename list: "<<txtfiles.size()<<endl;
		for ( TQStringList::Iterator it = txtfiles.begin(); it != txtfiles.end(); ++it )
		{
			TQString pdbfile=TQFileInfo(*it).baseName(true)+CSL1(".pdb");
			DEBUGKPILOT<<"pdbfile="<<pdbfile<<", pdbdir="<<pdburl<<", txtfile="<<*it<<", txtdir="<<txturl<<endl;
			if (convertTXTtoPDB(txturl, *it, pdburl, pdbfile, &conv))
			{
				converted_Files.append(*it);
			}
		}
		if (converted_Files.size()>0) {
			KMessageBox::informationList(this, i18n("The following texts were "
					"successfully converted:"), converted_Files, i18n("Conversion Successful"));
		}
		else
		{
			KMessageBox::sorry(this, i18n("No text files were converted correctly"));
		}


	} else { // no dir


		// Check the from file
		if (!txtinfo.isFile() || !txtinfo.exists())
		{
			KMessageBox::sorry(this, i18n("<qt>The file <em>%1</em> does not "
				"exist.</qt>").arg(txturl));
			return;
		}

		if (convertTXTtoPDB(txtinfo.dirPath(true), txtinfo.fileName(),
				pdbinfo.dirPath(true), pdbinfo.fileName(), &conv) )
		{
			KMessageBox::information(this, i18n("Conversion of file %1 successful.").arg(txturl));
		}

	}

}


void ConverterDlg::slotUser1()
{
	TDEAboutApplication ab(TDEGlobal::instance()->aboutData(), this);
	ab.show();
	ab.exec();
	return;
}

void ConverterDlg::slotDirectories(bool dir)
{
	FUNCTIONSETUP;
	DEBUGKPILOT<<"Slot Directories: "<<dir<<endl;
	if (dir)
	{
		dlg->fTextLabel->setText(i18n("&Text folder:"));
		dlg->fPdbLabel->setText(i18n("&PalmDOC folder:"));
		dlg->fTXTDir->setMode(KFile::LocalOnly | KFile::Directory);
		dlg->fPDBDir->setMode(KFile::LocalOnly | KFile::Directory);
	} else {
		dlg->fTextLabel->setText(i18n("&Text file:"));
		dlg->fPdbLabel->setText(i18n("&DOC file:"));
		dlg->fTXTDir->setMode(KFile::LocalOnly | KFile::File);
		dlg->fPDBDir->setMode(KFile::LocalOnly | KFile::File);
	}
}

bool ConverterDlg::convertTXTtoPDB(TQString txtdir, TQString txtfile,
		TQString pdbdir, TQString pdbfile, DOCConverter*conv)
{
	FUNCTIONSETUP;
	bool res=false;
	TQFileInfo dbfileinfo(pdbdir, pdbfile);
	DEBUGKPILOT<<"Working  on file "<<pdbfile<<endl;
	if (!dbfileinfo.exists() || !askOverwrite ||
			(KMessageBox::Yes==KMessageBox::questionYesNo(this,
			i18n("<qt>The database file <em>%1</em> already exists. Overwrite it?</qt>")
			.arg(dbfileinfo.filePath()), TQString(), i18n("Overwrite"), KStdGuiItem::cancel() ) ))
	{
		PilotLocalDatabase*pdbdb=new PilotLocalDatabase(pdbdir, TQFileInfo(pdbfile).baseName(), false);
		if (pdbdb)
		{
			if (!pdbdb->isOpen())
			{
#ifdef DEBUG
				DEBUGKPILOT<<pdbfile<<" does not yet exist. Creating it"<<endl;
#endif
				if (!pdbdb->createDatabase(get_long("REAd"), get_long("TEXt")) ) {
				}
			}

			if (pdbdb->isOpen())
			{
				conv->setPDB(pdbdb);
				conv->setTXTpath(txtdir, txtfile);
				DEBUGKPILOT<<"Converting "<<txtfile<<" (dir "<<txtdir<<") to "<<dbfileinfo.filePath()<<endl;
				if (conv->convertTXTtoPDB()) res=true;
			}
			delete pdbdb;
		}
		if ( !res && verbose )
		{
			KMessageBox::sorry(this, i18n("<qt>Error while converting the text %1.</qt>").arg(txtfile));
		}
	}
	else
	{
		DEBUGKPILOT<<"Ignoring the file "<<txtfile<<endl;
	}
	return res;
}

bool ConverterDlg::convertPDBtoTXT(TQString pdbdir, TQString pdbfile,
		TQString txtdir, TQString txtfile, DOCConverter*conv)
{
	FUNCTIONSETUP;
	bool res=false;
	TQFileInfo txtfileinfo(txtdir, txtfile);
	DEBUGKPILOT<<"Working  on file "<<txtfile<<endl;
	if (!txtfileinfo.exists() || !askOverwrite ||
			(KMessageBox::Yes==KMessageBox::questionYesNo(this,
			i18n("<qt>The text file <em>%1</em> already exists. Overwrite it?</qt>")
			.arg(txtfileinfo.filePath()), TQString(), i18n("Overwrite"), KStdGuiItem::cancel() ) ))
	{
		PilotLocalDatabase*pdbdb=new PilotLocalDatabase(pdbdir, TQFileInfo(pdbfile).baseName(), false);
		if (pdbdb)
		{
			if (pdbdb->isOpen())
			{
				conv->setPDB(pdbdb);
				conv->setTXTpath(txtdir, txtfile);
				DEBUGKPILOT<<"Converting "<<txtfile<<" (dir "<<txtdir<<") from "<<pdbfile<<" (dir "<<pdbdir<<")"<<endl;
				if (conv->convertPDBtoTXT()) res=true;
			}
			delete pdbdb;
		}
		if ( !res && verbose )
		{
			KMessageBox::sorry(this, i18n("<qt>Error while converting the text %1.</qt>").arg(pdbfile));
		}
	}
	else
	{
		DEBUGKPILOT<<"Ignoring the file "<<pdbfile<<endl;
	}
	return res;

}

#include "kpalmdoc_dlg.moc"
