/* The TdeGtk Theming Engine for Gtk+.
 * Copyright (C) 2012 Timothy Pearson <kb9vqf@pearsoncomputing.net>
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */

#include <cairo.h>
#include <gtk/gtk.h>
#include <math.h>

#include <tqapplication.h>
#include <tqpainter.h>
#include <tqstyle.h>
#include <tqpopupmenu.h>
#include <tqlistview.h>
#include <tqdrawutil.h>
#include <tqslider.h>

#include <kiconloader.h>
#include <kicontheme.h>

#include "tdegtk-cairo-support.h"
#include "tdegtk-draw.h"
#include "tdegtk-support.h"
#include "tdegtk-types.h"
#include "tdegtk-widgetlookup.h"
#include "tdegtk-utils.h"
#include "tdegtk-animations.h"

#include "tqtcairopainter.h"

#define DEBUG_FILL_BACKGROUND_WITH_COLOR(p,x,y,z)	TQBrush brush2( TQColor(x,y,z), TQBrush::SolidPattern );	\
							p.setBrush( brush2 );						\
							p.setPen( TQt::NoPen );						\
							p.drawRect( 0,0, width*2,height*2 );				\
							p.setBrush(TQt::NoBrush);

#define DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p,x)		p.setBrush(x);				\
								p.setPen(TQt::NoPen);			\
								p.drawRect(0, 0, width, height);	\
								p.setBrush(TQt::NoBrush);

#define DRAW_FILLED_RECTANGLE_OVER_SPECIFIC_AREA(p,b,x,y,w,h)	p.setBrush(b);				\
								p.setPen(TQt::NoPen);			\
								p.drawRect(x, y, w, h);			\
								p.setBrush(TQt::NoBrush);

extern int m_scrollBarSubLineWidth;
extern bool m_scrollbarBack1;
extern bool m_scrollbarForward1;
extern bool m_scrollbarBack2;
extern bool m_scrollbarForward2;

WidgetLookup m_widgetLookup;
Animations m_animations;

//#define DRAW_SCROLLBAR_PIECES_INSTEAD_OF_ENTIRE_BAR 1

// Keep this in sync with gtkToTQPaletteColorGroup() below
static TQColorGroup gtkToTQtColorGroup(GtkThemingEngine* engine, GtkStateFlags state) {
// 	GdkRGBA *background_color;
// 	GdkRGBA *foreground_color;
// 	gtk_theming_engine_get(engine, state, GTK_STYLE_PROPERTY_BACKGROUND_COLOR, &background_color, NULL);
// 	gtk_theming_engine_get(engine, state, GTK_STYLE_PROPERTY_COLOR, &foreground_color, NULL);

// 	GTK_STATE_FLAG_ACTIVE
// 	GTK_STATE_FLAG_PRELIGHT
// 	GTK_STATE_FLAG_SELECTED
// 	GTK_STATE_FLAG_INSENSITIVE
// 	GTK_STATE_FLAG_INCONSISTENT
// 	GTK_STATE_FLAG_FOCUSED

	TQColorGroup cg;
// 	if (state & GTK_STATE_FLAG_INCONSISTENT) {
// 		cg = tqApp->palette().inactive();
// 	}
	/*else*/ if (state & GTK_STATE_FLAG_INSENSITIVE) {
		cg = tqApp->palette().disabled();
	}
	else {
		cg = tqApp->palette().active();
	}
// 	cg.setColor(TQColorGroup::Background, TQColor((background_color->red*255.0),(background_color->green*255.0),(background_color->blue*255.0)));
// 	cg.setColor(TQColorGroup::Foreground, TQColor((foreground_color->red*255.0),(foreground_color->green*255.0),(foreground_color->blue*255.0)));

	return cg;
}

// Keep this in sync with gtkToTQtColorGroup() above
static TQPalette::ColorGroup gtkToTQPaletteColorGroup(GtkThemingEngine* engine, GtkStateFlags state) {
	TQPalette::ColorGroup cg = TQPalette::Active;

	if (state & GTK_STATE_FLAG_INSENSITIVE) {
		cg = TQPalette::Disabled;
	}
	else {
		cg = TQPalette::Active;
	}

	return cg;
}

// Keep this in sync with gtkToTQtColorGroup() above
static TQColorGroup gtkToTQColorGroup(GtkThemingEngine* engine, GtkStateFlags state, TQPalette &basePalette) {
	TQColorGroup cg = basePalette.active();

	if (state & GTK_STATE_FLAG_INSENSITIVE) {
		cg = basePalette.disabled();
	}
	else {
		cg = basePalette.active();
	}

	return cg;
}

static TQStyle::SFlags gtkToTQtStyleFlags(GtkThemingEngine* engine, GtkStateFlags state, TQt3WidgetType wt, GtkWidget* widget) {
	TQStyle::SFlags sflags = TQStyle::Style_Default;

	gboolean active, inconsistent, mousedown, prelight, focused, disabled, in_menu, has_focus;
	in_menu = gtk_theming_engine_has_class (engine, GTK_STYLE_CLASS_MENUITEM);
	prelight = (state & GTK_STATE_FLAG_PRELIGHT) != 0;
	focused = (state & GTK_STATE_FLAG_FOCUSED) != 0;
	disabled = (state & GTK_STATE_FLAG_INSENSITIVE) != 0;
	inconsistent = (state & GTK_STATE_FLAG_INCONSISTENT) != 0;
	mousedown = (state & GTK_STATE_FLAG_SELECTED) != 0;
	active = (state & GTK_STATE_FLAG_ACTIVE) != 0;
	has_focus = gtk_widget_has_visible_focus(widget);

	if ((wt == TQT3WT_TQRadioButton) || (wt == TQT3WT_TQCheckBox) || (wt == TQT3WT_TQPushButton)) {
		GtkCheckButton* checkButton = (widget)?((GTK_IS_CHECK_BUTTON(widget))?GTK_CHECK_BUTTON(widget):0L):0L;
		if ((!inconsistent) && (!mousedown)) {
			if (active) {
				sflags |= TQStyle::Style_On;
			}
			else {
				sflags |= TQStyle::Style_Off;
			}
		}
		if (mousedown) {
			if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkButton))) {
				sflags |= TQStyle::Style_On;
				sflags |= TQStyle::Style_Down;
			}
			else {
				sflags |= TQStyle::Style_Off;
				sflags |= TQStyle::Style_Down;
			}
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_TQToolButton) {
		if (!inconsistent) {
			if (active) {
				sflags |= TQStyle::Style_On;
			}
			else {
				sflags |= TQStyle::Style_Off;
			}
		}
		if (mousedown) {
			sflags |= TQStyle::Style_Down;
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
			sflags |= TQStyle::Style_Raised;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_GTKTreeViewCell) {
		if (!inconsistent) {
			if (active) {
				sflags |= TQStyle::Style_On;
			}
			else {
				sflags |= TQStyle::Style_Off;
			}
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_TQGroupBox) {
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_TQMenuItem) {
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
			sflags |= TQStyle::Style_Active;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_TQMenuBarItem) {
		if (prelight) {
			sflags |= TQStyle::Style_Active;
			sflags |= TQStyle::Style_Down;
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
	}
	else if (wt == TQT3WT_TQSlider) {
		if (active) {
			sflags |= TQStyle::Style_Active;
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_TQScrollBar) {
		if (active) {
			sflags |= TQStyle::Style_Active;
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}
	else if (wt == TQT3WT_TQLineEdit) {
		if (active) {
			sflags |= TQStyle::Style_Down;
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
		sflags |= TQStyle::Style_Sunken;
	}
	else {
		if (active) {
			sflags |= TQStyle::Style_Down;
		}
		if (prelight) {
			sflags |= TQStyle::Style_MouseOver;
		}
		if (focused) {
			sflags |= TQStyle::Style_HasFocus;
		}
		if (!disabled) {
			sflags |= TQStyle::Style_Enabled;
		}
		if (has_focus) {
			sflags |= TQStyle::Style_HasFocus;
		}
	}

	return sflags;
}

static TQColorGroup::ColorRole backgroundModeToColorRole(TQt::BackgroundMode mode) {
	TQColorGroup::ColorRole ret = TQColorGroup::Background;

	if (mode == TQt::PaletteForeground) {
		ret = TQColorGroup::Foreground;
	}
	else if (mode == TQt::PaletteBackground) {
		ret = TQColorGroup::Background;
	}
	else if (mode == TQt::PaletteButton) {
		ret = TQColorGroup::Button;
	}
	else if (mode == TQt::PaletteLight) {
		ret = TQColorGroup::Light;
	}
	else if (mode == TQt::PaletteMidlight) {
		ret = TQColorGroup::Midlight;
	}
	else if (mode == TQt::PaletteDark) {
		ret = TQColorGroup::Dark;
	}
	else if (mode == TQt::PaletteMid) {
		ret = TQColorGroup::Mid;
	}
	else if (mode == TQt::PaletteText) {
		ret = TQColorGroup::Text;
	}
	else if (mode == TQt::PaletteBrightText) {
		ret = TQColorGroup::BrightText;
	}
	else if (mode == TQt::PaletteButtonText) {
		ret = TQColorGroup::ButtonText;
	}
	else if (mode == TQt::PaletteBase) {
		ret = TQColorGroup::Base;
	}
	else if (mode == TQt::PaletteShadow) {
		ret = TQColorGroup::Shadow;
	}
	else if (mode == TQt::PaletteHighlight) {
		ret = TQColorGroup::Highlight;
	}
	else if (mode == TQt::PaletteHighlightedText) {
		ret = TQColorGroup::HighlightedText;
	}
	else if (mode == TQt::PaletteLink) {
		ret = TQColorGroup::Link;
	}
	else if (mode == TQt::PaletteLinkVisited) {
		ret = TQColorGroup::LinkVisited;
	}

	return ret;
}

void gtkRangeToCeData(GtkRange* rangeWidget, TQStyleControlElementData &ceData) {
	gint slider_start_pos;
	gint slider_end_pos;
	GtkAdjustment* adjustment = gtk_range_get_adjustment(rangeWidget);
	gtk_range_get_slider_range(rangeWidget, &slider_start_pos, &slider_end_pos);

	ceData.minSteps = gtk_adjustment_get_lower(adjustment);
	ceData.maxSteps = gtk_adjustment_get_upper(adjustment);
// 	ceData.tickMarkSetting =
// 	ceData.tickInterval =
	ceData.currentStep = gtk_adjustment_get_value(adjustment);
	ceData.startStep = slider_start_pos;
	ceData.lineStep = gtk_adjustment_get_step_increment(adjustment);
	ceData.pageStep = gtk_adjustment_get_page_increment(adjustment);
}

void gtkScaleToSliderCeData(GtkScale* scaleWidget, TQStyleControlElementData &ceData) {
	gtkRangeToCeData(GTK_RANGE(scaleWidget), ceData);

// 	GtkPositionType valuePosition = gtk_scale_get_value_pos(scaleWidget);
// 	if (valuePosition == GTK_POS_LEFT) {
// 		ceData.tickMarkSetting = TQSlider::Left;
// 	}
// 	else if (valuePosition == GTK_POS_RIGHT) {
// 		ceData.tickMarkSetting = TQSlider::Right;
// 	}
// 	else if (valuePosition == GTK_POS_TOP) {
// 		ceData.tickMarkSetting = TQSlider::Above;
// 	}
// 	else if (valuePosition == GTK_POS_BOTTOM) {
// 		ceData.tickMarkSetting = TQSlider::Below;
// 	}
// 	else {
		ceData.tickMarkSetting = TQSlider::NoMarks;
// 	}
}

void gtkScrollbarToScrollbarCeData(GtkScrollbar* scrollbarWidget, TQStyleControlElementData &ceData) {
	GtkRange* rangeWidget = GTK_RANGE(scrollbarWidget);
	gtkRangeToCeData(rangeWidget, ceData);

	gint slider_start_pos;
	gint slider_end_pos;
	GtkAdjustment* adjustment = gtk_range_get_adjustment(rangeWidget);
	gtk_range_get_slider_range(rangeWidget, &slider_start_pos, &slider_end_pos);

	ceData.minSteps = gtk_adjustment_get_lower(adjustment);
	ceData.maxSteps = (gtk_adjustment_get_upper(adjustment)-(gtk_adjustment_get_page_increment(adjustment)+gtk_adjustment_get_step_increment(adjustment)));
	ceData.currentStep = gtk_adjustment_get_value(adjustment);
	ceData.startStep = slider_start_pos;
	ceData.lineStep = gtk_adjustment_get_step_increment(adjustment);
	ceData.pageStep = gtk_adjustment_get_page_increment(adjustment);

	// Convert the GTK slider length into a correct pagestep that TQt3 can use to obtain the same slider length value
	bool threeButtonScrollBar = false;
	if (m_scrollbarForward1 | m_scrollbarBack2) threeButtonScrollBar = true;
	TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
	int gtkSliderLength = slider_end_pos - slider_start_pos;
	uint range = ceData.maxSteps - ceData.minSteps;
	// HACK
	// GTK3 does not handle odd-sized steppers correctly
	if ((m_scrollBarSubLineWidth % 2) != 0) {
		gtkSliderLength = gtkSliderLength + 4;
	}
	int sbextent = tqApp->style().pixelMetric(TQStyle::PM_ScrollBarExtent, ceData, elementFlags, NULL);
	int maxlen = ((ceData.orientation == TQt::Horizontal) ? ceData.rect.width() : ceData.rect.height()) - (sbextent * (threeButtonScrollBar ? 3.0 : 2.0));
	int gtkPageStep=ceil((-1.0)*(range*gtkSliderLength)/(gtkSliderLength-maxlen));
	ceData.pageStep = gtkPageStep;
}

static void
draw_combobox_frame(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget) {
	cairo_save(cr);

	cairo_matrix_t gtk_matrix;
	cairo_get_matrix(cr, &gtk_matrix);
	gtk_matrix.x0 = 0;
	gtk_matrix.y0 = 0;
	cairo_set_matrix(cr, &gtk_matrix);

	GtkWidget* parent(widget?gtk_widget_get_parent(widget):0L);
	if (GTK_IS_COMBO_BOX(parent)) {
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(parent);
		TQRect boundingRect(0, 0, allocation.width, allocation.height);
		TQt3CairoPaintDevice pd(NULL, allocation.x, allocation.y, allocation.width, allocation.height, cr);
		TQPainter p(&pd);

		TQStringList objectTypes;
		objectTypes.append("TQComboBox");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY)) {
			elementFlags = elementFlags | TQStyle::CEF_IsEditable;
		}

		ceData.rect = boundingRect;

		if (gtk_widget_has_visible_focus(widget)) {
			elementFlags = elementFlags | TQStyle::CEF_HasFocus;
		}

		// Draw item
		//tqApp->style().drawComplexControl(TQStyle::CC_ComboBox, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget), TQStyle::SC_ComboBoxFrame, TQStyle::SC_None);
		tqApp->style().drawComplexControl(TQStyle::CC_ComboBox, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget), TQStyle::SC_All, TQStyle::SC_None);
	}

	cairo_restore(cr);
}

static void
draw_progressbar_frame(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget) {
	cairo_save(cr);

	cairo_matrix_t gtk_matrix;
	cairo_get_matrix(cr, &gtk_matrix);
	gtk_matrix.x0 = 0;
	gtk_matrix.y0 = 0;
	cairo_set_matrix(cr, &gtk_matrix);

	if (GTK_IS_PROGRESS_BAR(widget)) {
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(widget);
		TQRect boundingRect(0, 0, allocation.width, allocation.height);
		TQt3CairoPaintDevice pd(NULL, allocation.x, allocation.y, allocation.width, allocation.height, cr);
		TQPainter p(&pd);

		TQStringList objectTypes;
		objectTypes.append("TQProgressBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);
	
		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;
	
		GtkProgressBar* progressBar = GTK_PROGRESS_BAR(widget);
		gdouble gtkProgressBarFraction = gtk_progress_bar_get_fraction(progressBar);
		ceData.totalSteps = 8192;
		ceData.currentStep = gtkProgressBarFraction*8192;
		ceData.percentageVisible = false;

		TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget);
		sflags = sflags | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default);

		TQRect progressBarGrooveRect;
		TQRect progressBarContentsRect;

		if (gtk_widget_has_visible_focus(widget)) {
			elementFlags = elementFlags | TQStyle::CEF_HasFocus;
		}

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)

		// Draw frame
		progressBarGrooveRect = TQStyle::visualRect(tqApp->style().subRect(TQStyle::SR_ProgressBarGroove, ceData, elementFlags, NULL), ceData, elementFlags);
		progressBarContentsRect = TQStyle::visualRect(tqApp->style().subRect(TQStyle::SR_ProgressBarContents, ceData, elementFlags, NULL), ceData, elementFlags);

		tqApp->style().drawControl(TQStyle::CE_ProgressBarGroove, &p, ceData, elementFlags, progressBarGrooveRect, ((state & GTK_STATE_FLAG_INSENSITIVE)?objectPalette.disabled():objectPalette.active()), sflags);

		// Draw contents
		if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL)) {
			// If vertical, apply a 90 degree rotation matrix to the painter
			// This is required to make TQt draw a vertical progress bar
			TQWMatrix m;

// 			// Upside down
// 			m.rotate(90.0);
// 			m.translate(0, (allocation.width)*(-1.0));

			// Right side up
			m.rotate(-90.0);
			m.translate((allocation.height)*(-1.0), 0);

			p.setWorldMatrix(m, true);

			boundingRect = TQRect(0, 0, allocation.height, allocation.width);
			ceData.rect = boundingRect;
		}

		progressBarGrooveRect = TQStyle::visualRect(tqApp->style().subRect(TQStyle::SR_ProgressBarGroove, ceData, elementFlags, NULL), ceData, elementFlags);
		progressBarContentsRect = TQStyle::visualRect(tqApp->style().subRect(TQStyle::SR_ProgressBarContents, ceData, elementFlags, NULL), ceData, elementFlags);

		tqApp->style().drawControl(TQStyle::CE_ProgressBarContents, &p, ceData, elementFlags, progressBarContentsRect, ((state & GTK_STATE_FLAG_INSENSITIVE)?objectPalette.disabled():objectPalette.active()), sflags);
	}

	cairo_restore(cr);
}

static void
draw_slider_frame(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget) {
	cairo_save(cr);

	cairo_matrix_t gtk_matrix;
	cairo_get_matrix(cr, &gtk_matrix);
	gtk_matrix.x0 = 0;
	gtk_matrix.y0 = 0;
	cairo_set_matrix(cr, &gtk_matrix);

	if (GTK_IS_SCALE(widget)) {
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(widget);

		GdkRectangle rangeRect;
		gtk_range_get_range_rect(GTK_RANGE(widget), &rangeRect);

		TQRect boundingRect(0, 0, rangeRect.width, rangeRect.height);
		TQt3CairoPaintDevice pd(NULL, allocation.x + rangeRect.x, allocation.y + rangeRect.y, rangeRect.width, rangeRect.height, cr);
		TQPainter p(&pd);

		bool mousedown = (state & GTK_STATE_FLAG_ACTIVE) != 0;
		bool prelight = (state & GTK_STATE_FLAG_PRELIGHT) != 0;

		GtkScale* scaleWidget = GTK_SCALE(widget);

		TQStringList objectTypes;
		objectTypes.append("TQSlider");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		gtkScaleToSliderCeData(scaleWidget, ceData);

		if (gtk_widget_has_visible_focus(widget)) {
			elementFlags = elementFlags | TQStyle::CEF_HasFocus;
		}

		if (mousedown) {
			elementFlags = elementFlags | TQStyle::CEF_IsActive;
		}

		if (prelight) {
			elementFlags = elementFlags | TQStyle::CEF_HasMouse;
		}

		// Draw item
		tqApp->style().drawComplexControl(TQStyle::CC_Slider, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQSlider, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default), TQStyle::SC_All, ((mousedown)?TQStyle::SC_All:TQStyle::SC_None));
	}

	cairo_restore(cr);
}

TQStyle::SubControl lastSliderActiveSubControl = TQStyle::SC_None;

static void
draw_scrollbar_frame(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget) {
	cairo_save(cr);

	cairo_matrix_t gtk_matrix;
	cairo_get_matrix(cr, &gtk_matrix);
	gtk_matrix.x0 = 0;
	gtk_matrix.y0 = 0;
	cairo_set_matrix(cr, &gtk_matrix);

	if (GTK_IS_SCROLLBAR(widget)) {
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(widget);

		GdkRectangle rangeRect;
		gtk_range_get_range_rect(GTK_RANGE(widget), &rangeRect);

		TQRect boundingRect(0, 0, rangeRect.width, rangeRect.height);
		TQt3CairoPaintDevice pd(NULL, allocation.x + rangeRect.x, allocation.y + rangeRect.y, rangeRect.width, rangeRect.height, cr);
		TQPainter p(&pd);

		GtkStateFlags widgetStateFlags = gtk_widget_get_state_flags(widget);

		bool prelight = (state & GTK_STATE_FLAG_PRELIGHT) != 0;

		GtkScrollbar* scrollbarWidget = GTK_SCROLLBAR(widget);

		TQStringList objectTypes;
		objectTypes.append("TQSlider");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		gtkScrollbarToScrollbarCeData(scrollbarWidget, ceData);

		if (gtk_widget_has_visible_focus(widget)) {
			elementFlags = elementFlags | TQStyle::CEF_HasFocus;
		}

		if (prelight) {
			elementFlags = elementFlags | TQStyle::CEF_HasMouse;
		}

		TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_TQScrollBar, widget);
		sflags = sflags | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default);

		if (gtk_widget_get_sensitive(widget)) {
			sflags |= TQStyle::Style_Enabled;
			state = (GtkStateFlags)(state & (~GTK_STATE_FLAG_INSENSITIVE));
		}
		else {
			sflags &= ~TQStyle::Style_Enabled;
			state = (GtkStateFlags)(state | GTK_STATE_FLAG_INSENSITIVE);
		}

		sflags &= ~TQStyle::Style_On;
		sflags &= ~TQStyle::Style_Down;
		sflags &= ~TQStyle::Style_Active;

		// Determine active subcontrols
		gint cursor_x = -1;
		gint cursor_y = -1;
		TQStyle::SubControl activeSubControl = TQStyle::SC_None;
		if (gtk_widget_get_realized(widget)) {
			GdkDevice *device;
			device = gdk_device_manager_get_client_pointer(gdk_display_get_device_manager(gtk_widget_get_display(widget)));
			gdk_window_get_device_position(gtk_widget_get_window(widget), device, &cursor_x, &cursor_y, NULL);
			cursor_x = cursor_x - allocation.x;
			cursor_y = cursor_y - allocation.y;

			TQPoint cursor_pos(cursor_x, cursor_y);
			TQRect addline, subline, addpage, subpage, slider, first, last;		
			subline = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarSubLine);
			addline = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarAddLine);
			subpage = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarSubPage);
			addpage = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarAddPage);
			slider  = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarSlider);
			first   = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarFirst);
			last    = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarLast);

			bool threeButtonScrollBar = false;
			if (m_scrollbarForward1 | m_scrollbarBack2) threeButtonScrollBar = true;
			if (threeButtonScrollBar) {
				if (ceData.orientation == TQt::Horizontal) {
					addline = TQRect(allocation.width-(subline.width()*2), subline.y(), subline.width()*2, subline.height());
				}
				else {
					addline = TQRect(subline.x(), allocation.height-(subline.height()*2), subline.width(), subline.height()*2);
				}
			}

			// HACK
			// This may not be 100% reliable!
			GdkModifierType mouseStateFlags;
			gdk_device_get_state (device, gtk_widget_get_window(widget), NULL, &mouseStateFlags);
			bool mousedown = (mouseStateFlags & GDK_BUTTON1_MASK);

			if (mousedown) {
				if (lastSliderActiveSubControl != TQStyle::SC_None) {
					activeSubControl = (TQStyle::SubControl)(activeSubControl | lastSliderActiveSubControl);
				}
				else {
					if (subline.contains(cursor_pos)) {
						activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarSubLine);
						lastSliderActiveSubControl = TQStyle::SC_ScrollBarSubLine;
					}
					else if (addline.contains(cursor_pos)) {
						if (threeButtonScrollBar) {
							// Not so fast...the addline region may contain a subline control!
							TQRect internalSubLine;
							if (ceData.orientation == TQt::Horizontal) {
								internalSubLine = TQRect(addline.x(), addline.y(), addline.width()/2, addline.height());
							}
							else {
								internalSubLine = TQRect(addline.x(), addline.y(), addline.width(), addline.height()/2);
							}
							if (internalSubLine.contains(cursor_pos)) {
								activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarSubLine);
								lastSliderActiveSubControl = TQStyle::SC_ScrollBarSubLine;
							}
							else {
								activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarAddLine);
								lastSliderActiveSubControl = TQStyle::SC_ScrollBarAddLine;
							}
						}
						else {
							activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarAddLine);
							lastSliderActiveSubControl = TQStyle::SC_ScrollBarAddLine;
						}
					}
					else if (subpage.contains(cursor_pos)) {
						activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarSubPage);
						lastSliderActiveSubControl = TQStyle::SC_ScrollBarSubPage;
					}
					else if (addpage.contains(cursor_pos)) {
						activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarAddPage);
						lastSliderActiveSubControl = TQStyle::SC_ScrollBarAddPage;
					}
					else if (slider.contains(cursor_pos)) {
						activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarSlider);
						lastSliderActiveSubControl = TQStyle::SC_ScrollBarSlider;
					}
					else if (first.contains(cursor_pos)) {
						activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarFirst);
						lastSliderActiveSubControl = TQStyle::SC_ScrollBarFirst;
					}
					else if (last.contains(cursor_pos)) {
						activeSubControl = (TQStyle::SubControl)(activeSubControl | TQStyle::SC_ScrollBarLast);
						lastSliderActiveSubControl = TQStyle::SC_ScrollBarLast;
					}
				}
			}
			else {
				lastSliderActiveSubControl = TQStyle::SC_None;
			}
		}

		// Draw item
		tqApp->style().drawComplexControl(TQStyle::CC_ScrollBar, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), sflags, TQStyle::SC_All, activeSubControl);
	}

	cairo_restore(cr);
}

static void
draw_checkbox(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget, TQt3WidgetType tqt3WidgetType) {
	TQStringList objectTypes;
	objectTypes.append("TQCheckBox");
	TQPalette objectPalette = tqApp->palette(objectTypes);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	TQStyleControlElementData ceData;
	TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
	ceData.widgetObjectTypes = objectTypes;
	ceData.rect = boundingRect;

	if (gtk_widget_has_visible_focus(widget)) {
		elementFlags = elementFlags | TQStyle::CEF_HasFocus;
	}

	// Draw item
	tqApp->style().drawControl(TQStyle::CE_CheckBox, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, tqt3WidgetType, widget));
}

static void
draw_radiobutton(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget, TQt3WidgetType tqt3WidgetType) {
	TQStringList objectTypes;
	objectTypes.append("TQCheckBox");
	TQPalette objectPalette = tqApp->palette(objectTypes);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	TQStyleControlElementData ceData;
	TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
	ceData.widgetObjectTypes = objectTypes;
	ceData.rect = boundingRect;

	if (gtk_widget_has_visible_focus(widget)) {
		elementFlags = elementFlags | TQStyle::CEF_HasFocus;
	}

	// Draw item
	tqApp->style().drawControl(TQStyle::CE_RadioButton, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, tqt3WidgetType, widget));
}

static void
draw_pushbutton(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget, TQt3WidgetType tqt3WidgetType) {
	TQStringList objectTypes;
	objectTypes.append("TQCheckBox");
	TQPalette objectPalette = tqApp->palette(objectTypes);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	TQStyleControlElementData ceData;
	TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
	ceData.widgetObjectTypes = objectTypes;
	ceData.rect = boundingRect;

	if (gtk_widget_has_default(widget)) {
		elementFlags = elementFlags | TQStyle::CEF_IsDefault;
	}
	if (gtk_widget_has_visible_focus(widget)) {
		elementFlags = elementFlags | TQStyle::CEF_HasFocus;
	}

	// Draw item
	tqApp->style().drawControl(TQStyle::CE_PushButton, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, tqt3WidgetType, widget));
}

static void
draw_toolbar_button(DRAW_ARGS, const GtkWidgetPath* path, GtkStateFlags state, GtkWidget* widget, TQt3WidgetType tqt3WidgetType) {
	bool mousedown = (state & GTK_STATE_FLAG_ACTIVE) != 0;

	TQStringList objectTypes;
	objectTypes.append("TQToolButton");
	TQPalette objectPalette = tqApp->palette(objectTypes);

	TQStyleControlElementData ceData;
	TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
	ceData.widgetObjectTypes = objectTypes;
	ceData.colorGroup = ((state & GTK_STATE_FLAG_INSENSITIVE)?objectPalette.disabled():objectPalette.active());
	ceData.orientation = TQt::Horizontal;

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	ceData.rect = boundingRect;

	GtkWidget* parent = (widget)?Gtk::gtk_widget_find_parent(widget, GTK_TYPE_TOOLBAR):0L;
	if (parent) {
		TQStringList parentObjectTypes;
		parentObjectTypes.append("TQToolBar");
		TQPalette parentObjectPalette = tqApp->palette(parentObjectTypes);
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(parent);

		ceData.parentWidgetData.widgetObjectTypes = parentObjectTypes;
		ceData.parentWidgetData.rect = TQRect(0, 0, allocation.width, allocation.height);
		ceData.parentWidgetData.colorGroup = ((state & GTK_STATE_FLAG_INSENSITIVE)?parentObjectPalette.disabled():parentObjectPalette.active());
	}

	// Draw item
	tqApp->style().drawComplexControl(TQStyle::CC_ToolButton, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, tqt3WidgetType, widget), TQStyle::SC_ToolButton, ((mousedown)?TQStyle::SC_ToolButton:TQStyle::SC_None));
}

/* draw a texture placed on the centroid */
static gboolean
draw_centroid_texture (GtkThemingEngine *engine,
                       cairo_t          *cr,
                       gdouble           x,
                       gdouble           y,
                       gdouble           width,
                       gdouble           height)
{
	GtkStateFlags state;
	GValue value = { 0, };
	cairo_pattern_t *texture = NULL;
	cairo_surface_t *surface = NULL;
	gboolean retval = FALSE;
	
	state = gtk_theming_engine_get_state (engine);
	
	gtk_theming_engine_get_property (engine, "-tdegtk-centroid-texture", state, &value);
	
	if (!G_VALUE_HOLDS_BOXED (&value)) {
		return FALSE;
	}
	
	texture = (cairo_pattern_t*)g_value_dup_boxed (&value);
	g_value_unset (&value);
	
	if (texture != NULL) {
		cairo_pattern_get_surface (texture, &surface);
	}
	
	if (surface != NULL) {
		cairo_save (cr);
		
		cairo_set_source_surface (cr, surface, (gint) (x + width / 2 - cairo_image_surface_get_width (surface) / 2), (gint) (y + height / 2 - cairo_image_surface_get_height (surface) / 2));
		cairo_paint (cr);
		
		cairo_restore (cr);
		
		retval = TRUE;
	}
	
	if (texture != NULL) {
		cairo_pattern_destroy (texture);
	}
	
	return retval;
}

static void
tdegtk_draw_activity (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_PROGRESS_BAR)) {
#if 0
		TQStringList objectTypes;
		objectTypes.append("TQProgressBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget);
		sflags = sflags | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default);

		TQRect progressBarRect = TQStyle::visualRect(tqApp->style().subRect(TQStyle::SR_ProgressBarContents, ceData, elementFlags, NULL), ceData, elementFlags);

		// Draw item
		tqApp->style().drawControl(TQStyle::CE_ProgressBarContents, &p, ceData, elementFlags, progressBarRect, ((state & GTK_STATE_FLAG_INSENSITIVE)?objectPalette.disabled():objectPalette.active()), sflags);
#else
		// Do nothing
#endif
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,128,0,128);
		printf("[WARNING] tdegtk_draw_activity() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_arrow (GtkThemingEngine *engine,
                  cairo_t          *cr,
                  gdouble           angle,
                  gdouble           x,
                  gdouble           y,
                  gdouble           size)
{
	cairo_save(cr);
	cairo_reset_clip(cr);

	TQRect boundingRect(0, 0, size, size);
	TQt3CairoPaintDevice *pd = NULL;
	TQPainter *p = NULL;

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	GtkWidget* parent(widget?gtk_widget_get_parent(widget):0L);

	GtkArrowType arrow_direction;

	if ((angle <= ((G_PI/2)-(G_PI/4))) || (angle > ((3*(G_PI/2))+(G_PI/4)))) {
		arrow_direction = GTK_ARROW_UP;
	}
	else if ((angle <= ((G_PI)-(G_PI/4))) && (angle > ((0)+(G_PI/4)))) {
		arrow_direction = GTK_ARROW_RIGHT;
	}
	else if ((angle <= ((3*(G_PI/2))-(G_PI/4))) && (angle > ((G_PI/2)+(G_PI/4)))) {
		arrow_direction = GTK_ARROW_DOWN;
	}
	else {
		arrow_direction = GTK_ARROW_LEFT;
	}

#if 0
	if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_COMBO_BOX)) {
		// Do nothing
	}

	else
#endif
	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCROLLBAR)) {
#ifdef DRAW_SCROLLBAR_PIECES_INSTEAD_OF_ENTIRE_BAR
		TQStringList objectTypes;
		objectTypes.append("TQScrollBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		GtkRange* rangeWidget = GTK_RANGE(widget);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.orientation = ((arrow_direction == GTK_ARROW_UP) || (arrow_direction == GTK_ARROW_DOWN))?TQt::Vertical:TQt::Horizontal;
		bool subline = ((arrow_direction == GTK_ARROW_DOWN) || (arrow_direction == GTK_ARROW_RIGHT))?false:true;

		bool combine_addlineregion_drawing_areas = tqApp->style().styleHint(TQStyle::SH_ScrollBar_CombineAddLineRegionDrawingAreas);
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(widget);

		TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget);
		sflags = sflags | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default);

		if (combine_addlineregion_drawing_areas) {
			int newx = x;
			int newy = y;
			int neww = size;
			int newh = size;
			if (!subline) {
				if (ceData.orientation == TQt::Horizontal) {
					if ((x+m_scrollBarSubLineWidth) == allocation.width) {
						newx = x-m_scrollBarSubLineWidth;
					}
					boundingRect.setWidth(m_scrollBarSubLineWidth*2);
					neww = m_scrollBarSubLineWidth*2;
				}
				else {
					if ((y+m_scrollBarSubLineWidth) == allocation.height) {
						newy = y-m_scrollBarSubLineWidth;
					}
					boundingRect.setHeight(m_scrollBarSubLineWidth*2);
					newh = m_scrollBarSubLineWidth*2;
				}
			}
			pd = new TQt3CairoPaintDevice(NULL, newx, newy, neww, newh, cr);
			p = new TQPainter(pd);

			if (!subline) {
				if (ceData.orientation == TQt::Horizontal) {
					p->setClipRect(TQRect(0+m_scrollBarSubLineWidth, 0, m_scrollBarSubLineWidth, newh));
				}
				else {
					p->setClipRect(TQRect(0, 0+m_scrollBarSubLineWidth, neww, m_scrollBarSubLineWidth));
				}
			}
		}
		else {
			pd = new TQt3CairoPaintDevice(NULL, x, y, size, size, cr);
			p = new TQPainter(pd);
		}

		ceData.rect = boundingRect;

		gtkRangeToCeData(rangeWidget, ceData);

		// Draw slider arrow buttons
		TQRect scrollpagerect = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, (subline)?TQStyle::SC_ScrollBarSubLine:TQStyle::SC_ScrollBarAddLine, sflags);
		if (ceData.orientation == TQt::Vertical) {
			scrollpagerect.setY(ceData.rect.y());
			scrollpagerect.setHeight(ceData.rect.height());
		}
		else {
			scrollpagerect.setX(ceData.rect.x());
			scrollpagerect.setWidth(ceData.rect.width());
		}
		tqApp->style().drawPrimitive((subline)?TQStyle::PE_ScrollBarSubLine:TQStyle::PE_ScrollBarAddLine, p, scrollpagerect, gtkToTQtColorGroup(engine, state), sflags);
#else
		// Draw nothing
#endif
	}

	else if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_COMBO_BOX) && (!(GTK_IS_MENU_ITEM(widget) || GTK_IS_MENU_ITEM(parent) || (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUITEM))))) {
		// Do nothing
	}

	else {
		pd = new TQt3CairoPaintDevice(NULL, x, y, size, size, cr);
		p = new TQPainter(pd);

		// Draw arrow
		TQStyle::PrimitiveElement pe;
		if (arrow_direction == GTK_ARROW_UP) {
			pe = TQStyle::PE_ArrowUp;
		}
		else if (arrow_direction == GTK_ARROW_DOWN) {
			pe = TQStyle::PE_ArrowDown;
		}
		else if (arrow_direction == GTK_ARROW_LEFT) {
			pe = TQStyle::PE_ArrowLeft;
		}
		else {
			pe = TQStyle::PE_ArrowRight;
		}

		TQt3WidgetType tqt3WidgetType = TQT3WT_NONE;
		TQStringList objectTypes;
		objectTypes.append("TQWidget");
		if ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENU)) || (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUITEM))) {
			GtkWidget* parent(widget?gtk_widget_get_parent(widget):0L);
			if (!GTK_IS_MENU_BAR(parent)) {
				objectTypes.clear();
				objectTypes.append("TQPopupMenu");
				tqt3WidgetType = TQT3WT_TQMenuItem;
			}
		}
		TQPalette objectPalette = tqApp->palette(objectTypes);

		tqApp->style().drawPrimitive(pe, p, boundingRect, gtkToTQColorGroup(engine, state, objectPalette), gtkToTQtStyleFlags(engine, state, tqt3WidgetType, widget));
	}
	
	if (p) {
		p->end();
		delete p;
	}
	if (pd) {
		delete pd;
	}

	cairo_restore(cr);
}

static void
tdegtk_draw_cell_background (DRAW_ARGS,
                            GtkRegionFlags flags)
{
	cairo_save(cr);
	cairo_reset_clip(cr);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_TREE_VIEW)) {
		TQStringList objectTypes;
		objectTypes.append("TQListView");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		if ((state & GTK_STATE_FLAG_SELECTED) != 0) {
			ceData.viewportData.bgBrush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Highlight);
		}
		else {
			ceData.viewportData.bgBrush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
		}

		TQListViewItem tqt3ListViewItem;
		TQStyleOption listViewItemOpt(&tqt3ListViewItem);

		// Draw item
		tqApp->style().drawComplexControl(TQStyle::CC_ListView, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget), TQStyle::SC_ListView, TQStyle::SC_All, listViewItemOpt);
	}

	// FIXME
	// GtkCellRenderer backgrounds should be drawn here, however GTK3 does not provide any means for a GtkCellRenderer to call style engine methods!
	// See upstream GTK bug #687677

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,255,128,64);
		printf("[WARNING] tdegtk_draw_cell_background() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();

	cairo_restore(cr);
}

static void
tdegtk_draw_cell_frame (DRAW_ARGS,
                       GtkRegionFlags flags)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_TREE_VIEW)) {
#if 0
		TQStringList objectTypes;
		objectTypes.append("TQListView");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		if ((state & GTK_STATE_FLAG_SELECTED) != 0) {
			ceData.viewportData.bgBrush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Highlight);
		}
		else {
			ceData.viewportData.bgBrush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
		}

		TQListViewItem tqt3ListViewItem;
		TQStyleOption listViewItemOpt(&tqt3ListViewItem);

		TQRect paintDeviceRect;
		TQRect backgroundRect;

		// Draw item
		tqApp->style().drawComplexControl(TQStyle::CC_ListView, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget), TQStyle::SC_ListView, TQStyle::SC_All, listViewItemOpt);
#endif
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,255,64,64);
		printf("[WARNING] tdegtk_draw_cell_background() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_check (DRAW_ARGS)
{
	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_TREE_VIEW)) {
		draw_checkbox(engine, cr, x, y, width, height, path, state, widget, TQT3WT_GTKTreeViewCell);
	}
	else {
		draw_checkbox(engine, cr, x, y, width, height, path, state, widget, TQT3WT_TQCheckBox);
	}
}

static void
tdegtk_draw_common (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	DEBUG_FILL_BACKGROUND_WITH_COLOR(p,0,255,255);
	printf("[WARNING] tdegtk_draw_common() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);

	GtkStateFlags state;
	state = gtk_theming_engine_get_state(engine);
	//tqApp->style().drawPrimitive(TQStyle::PE_Indicator, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
	
	if (p.isActive()) p.end();
}

static void
tdegtk_draw_common_background (DRAW_ARGS)
{
	cairo_save(cr);
	cairo_reset_clip(cr);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	//TQString widgetPath(Gtk::gtk_widget_path(widget));
	TQString widgetThemingEnginePath(gtk_widget_path_to_string(gtk_theming_engine_get_path(engine)));

	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCROLLBAR)) {
#ifdef DRAW_SCROLLBAR_PIECES_INSTEAD_OF_ENTIRE_BAR
		TQStringList objectTypes;
		objectTypes.append("TQScrollBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		GtkRange* rangeWidget = GTK_RANGE(widget);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		gtkRangeToCeData(rangeWidget, ceData);

		// Draw frame
		tqApp->style().drawPrimitive(TQStyle::PE_PanelLineEdit, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
#else
		draw_scrollbar_frame(engine, cr, x, y, width, height, path, state, widget);
#endif
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_SCROLLED_WINDOW)) {
		if ((x == 0) && (y == 0)) {
			TQStringList objectTypes;
			objectTypes.append("TQFrame");
			TQPalette objectPalette = tqApp->palette(objectTypes);
	
			int groupBoxLineWidth = 1;
	
			int lineWidth = 0;
			int midLineWidth = 0;
			TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_TQGroupBox, widget);
	
			lineWidth = groupBoxLineWidth;
			midLineWidth = 0;
			sflags = sflags | TQStyle::Style_Sunken;
	
			TQStyleOption opt(lineWidth, midLineWidth);
	
			// Draw background
			TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
			DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	
			// Draw group box
			tqApp->style().drawPrimitive(TQStyle::PE_PanelGroupBox, &p, boundingRect, gtkToTQtColorGroup(engine, state), sflags, opt);
		}
	}

	else if ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_FRAME) || gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY) || gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_BUTTON)) && (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_COMBO_BOX))) {
		draw_combobox_frame(engine, cr, x, y, width, height, path, state, widget);
	}

	else if ((gtk_widget_path_is_type(path, GTK_TYPE_ENTRY))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_TEXT_VIEW))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_TEXT_BUFFER))
		|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY))
		|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_COMBOBOX_ENTRY))
		) {
		TQStringList objectTypes;
		objectTypes.append("TQLineEdit");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

#if 0
	else if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_COMBO_BOX)) {
		bool mousedown = (state & GTK_STATE_FLAG_ACTIVE) != 0;

		TQStringList objectTypes;
		objectTypes.append("TQComboBox");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY)) {
			elementFlags = elementFlags | TQStyle::CEF_IsEditable;
		}

		ceData.rect = boundingRect;

		// Draw item
		tqApp->style().drawComplexControl(TQStyle::CC_ComboBox, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget), TQStyle::SC_ComboBoxEditField, TQStyle::SC_None);
	}
#endif

	else if (gtk_widget_path_is_type(path, GTK_TYPE_BUTTON)) {
#if 0
		TQStringList objectTypes;
		objectTypes.append("TQButton");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_TOOLBAR)) {
			// Draw background
			TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
			DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
		}
		else if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_TREE_VIEW)) {
			// Draw background
			TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
			DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
		}
		else {
			draw_pushbutton(engine, cr, x, y, width, height, path, state, widget, TQT3WT_TQPushButton);
		}
#else
		// Draw nothing!
#endif
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_TOOLBAR)) {
		TQStringList objectTypes;
		objectTypes.append("TQToolBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUBAR)) {
		TQStringList objectTypes;
		objectTypes.append("TQMenuBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_TOOLTIP)) {
		TQStringList objectTypes;
		objectTypes.append("TQTipLabel");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else if ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENU)) || (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUITEM))) {
		GtkWidget* parent(widget?gtk_widget_get_parent(widget):0L);
		if (!GTK_IS_MENU_BAR(parent)) {
			// FIXME
			// Mouse movements in the popup menu should be tracked similarly to mouse movements in the tab bar
			// Many TQt3 styles highlight inactive menu items on hover; not providing this visual feedback may cause users to briefly think their GTK application has frozen up
	
// 			// Register menu item with animation engine
// 			m_animations.menuItemEngine().registerWidget(widget);
// 	
// 			// Check tab properties
// 			int tabIndex = Gtk::gtk_notebook_find_tab(widget, x+width/2, y+height/2);
// 			m_animations.tabWidgetEngine().updateTabRect(widget, tabIndex, x, y, width, height);
// 			bool prelight = (tabIndex == m_animations.tabWidgetEngine().hoveredTab(widget));
// 			GtkMenuItem* menuitem = GTK_MENU_ITEM(widget);
// 			const int currentPage = gtk_notebook_get_current_page(notebook);
// 			const int numPages = gtk_notebook_get_n_pages(notebook);
// 			bool selectedTab = (tabIndex==currentPage);

			TQStringList objectTypes;
			objectTypes.append("TQPopupMenu");
			TQPalette objectPalette = tqApp->palette(objectTypes);

			bool pressed = false;
			bool prelight = false;

			TQStyleControlElementData ceData;
			TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
			ceData.widgetObjectTypes = objectTypes;
			ceData.rect = boundingRect;

			TQMenuItem tqt3MenuItem;
			int maxAcceleratorTextWidth = 0;
			int maxIconWidth = IconSize(TDEIcon::Small) + 4;
			TQStyleOption menuOpt(&tqt3MenuItem, maxIconWidth, maxAcceleratorTextWidth);

			TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_TQMenuItem, widget);
			sflags = sflags | ((pressed)?TQStyle::Style_Down:TQStyle::Style_Default) | ((prelight)?TQStyle::Style_MouseOver:TQStyle::Style_Default);

// 			if (???) elementFlags = elementFlags | TQStyle::TQStyle::CEF_IsCheckable;

			TQRect paintDeviceRect;
			TQRect backgroundRect;

			// Draw item
			tqApp->style().drawControl(TQStyle::CE_PopupMenuItem, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), sflags, menuOpt);
		}
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_NOTEBOOK)) {
		TQStringList objectTypes;
		objectTypes.append("TQTabWidget");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_PROGRESS_BAR)) {
		// Do nothing; backround is drawn in frame drawing routine
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_HANDLE_BOX)) {
		TQStringList objectTypes;
		objectTypes.append("TQDockWindowHandle");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_CELL)) {
		TQStringList objectTypes;
		objectTypes.append("TQListView");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		if ((state & GTK_STATE_FLAG_SELECTED) != 0) {
			ceData.viewportData.bgBrush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Highlight);
		}
		else {
			ceData.viewportData.bgBrush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
		}

		TQListViewItem tqt3ListViewItem;
		TQStyleOption listViewItemOpt(&tqt3ListViewItem);

		// Draw item
		tqApp->style().drawComplexControl(TQStyle::CC_ListView, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget), TQStyle::SC_ListView, TQStyle::SC_All, listViewItemOpt);
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCALE) /*&& gtk_theming_engine_has_class (engine, GTK_STYLE_CLASS_TROUGH)*/) {
		draw_slider_frame(engine, cr, x, y, width, height, path, state, widget);
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_LABEL)) {
		if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_FRAME)) {
			TQStringList objectTypes;
			objectTypes.append("TQFrame");
			TQPalette objectPalette = tqApp->palette(objectTypes);
	
			// Draw background
			TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
			DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
		}
		else {
			// Do nothing
		}
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_IMAGE)) {
		// Do nothing
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_SWITCH)) {
		TQStringList objectTypes;
		objectTypes.append("TQScrollBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;
		ceData.startStep = 0;

		// Draw frame
		tqApp->style().drawPrimitive(TQStyle::PE_PanelLineEdit, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
	}

#ifdef GTK_TYPE_LEVEL_BAR
	else if (gtk_widget_path_is_type(path, GTK_TYPE_LEVEL_BAR)) {
		TQStringList objectTypes;
		objectTypes.append("TQProgressBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		if (gtk_theming_engine_has_class(engine, "fill-block")) {
			// Draw background
			TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Highlight);
			DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
		}
		if (gtk_theming_engine_has_class(engine, "empty-fill-block")) {
			// Draw background
			TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
			DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
		}
	}
#else // GTK_TYPE_LEVEL_BAR
#warning GTK_TYPE_LEVEL_BAR not available in this version of GTK!  Level bars will not be drawn as a result!
#endif // GTK_TYPE_LEVEL_BAR

	else if (gtk_widget_path_is_type(path, GTK_TYPE_COMBO_BOX)) {
		// Do nothing
	}

	else if (widgetThemingEnginePath.endsWith("GtkTextHandle")) {
		// Do nothing
		// FIXME
		// Where is this widget actually used?
	}

	else if (widgetThemingEnginePath.endsWith("GtkEventBox")) {
		// Do nothing
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_BACKGROUND)) {
		// Do nothing
	}

	else if (widgetThemingEnginePath.endsWith("GtkTreeView.view")) {
		TQStringList objectTypes;
		objectTypes.append("TQListView");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Base);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else if ((gtk_widget_path_is_type(path, GTK_TYPE_EXPANDER))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_BOX))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_VIEWPORT))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_SCROLLED_WINDOW))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_PANED))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_FRAME))
		) {
		TQStringList objectTypes;
		objectTypes.append("TQWidget");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,0,0,255);
		printf("[WARNING] tdegtk_draw_common_background() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();

	cairo_restore(cr);
}

static void
tdegtk_draw_common_frame (DRAW_ARGS)
{
	cairo_save(cr);
	cairo_reset_clip(cr);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	TQString widgetPath(Gtk::gtk_widget_path(widget).c_str());

	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCROLLBAR)) {
#ifdef DRAW_SCROLLBAR_PIECES_INSTEAD_OF_ENTIRE_BAR
		if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_BUTTON)) {
			// Scrollbar buttons are drawn in the arrow handler
		}
		else {
			TQStringList objectTypes;
			objectTypes.append("TQScrollBar");
			TQPalette objectPalette = tqApp->palette(objectTypes);

			GtkRange* rangeWidget = GTK_RANGE(widget);
	
			TQStyleControlElementData ceData;
			TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
			ceData.widgetObjectTypes = objectTypes;
			ceData.rect = boundingRect;
			ceData.orientation = ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal);

			gtkRangeToCeData(rangeWidget, ceData);
	
			// Draw background
			// HACK
			// PE_ScrollBarAddPage and PE_ScrollBarSubPage are separate in TQt3
			// Apparently there is no such distinction in GTK3!
			TQRect scrollpagerect = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarAddPage, gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
			if (ceData.orientation == TQt::Vertical) {
				scrollpagerect.setY(ceData.rect.y());
				scrollpagerect.setHeight(ceData.rect.height());
			}
			else {
				scrollpagerect.setX(ceData.rect.x());
				scrollpagerect.setWidth(ceData.rect.width());
			}
			tqApp->style().drawPrimitive(TQStyle::PE_ScrollBarAddPage, &p, scrollpagerect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		}
#else
		// Do nothing...
#endif
	}

	else if ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_FRAME) || gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY) || gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_BUTTON)) && (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_COMBO_BOX))) {
		draw_combobox_frame(engine, cr, x, y, width, height, path, state, widget);
	}

	else {
		if (gtk_widget_path_is_type(path, GTK_TYPE_BUTTON)) {
			if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_TOOLBAR)) {
				draw_toolbar_button(engine, cr, x, y, width, height, path, state, widget, TQT3WT_TQToolButton);
			}
			else if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_TREE_VIEW)) {
				// Draw frame
				tqApp->style().drawPrimitive(TQStyle::PE_HeaderSection, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
			}
			else {
				draw_pushbutton(engine, cr, x, y, width, height, path, state, widget, TQT3WT_TQPushButton);
			}
		}

		else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUITEM)) {
			GtkWidget* parent(widget?gtk_widget_get_parent(widget):0L);
			if (GTK_IS_MENU_BAR(parent)) {
				TQStringList objectTypes;
				objectTypes.append("TQMenuBar");
				TQPalette objectPalette = tqApp->palette(objectTypes);
		
				TQStyleControlElementData ceData;
				TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
				ceData.widgetObjectTypes = objectTypes;
				ceData.rect = boundingRect;
		
				TQMenuItem tqt3MenuItem;
				int maxAcceleratorTextWidth = 0;
				int maxIconWidth = IconSize(TDEIcon::Small) + 4;
				TQStyleOption menuOpt(&tqt3MenuItem, maxIconWidth, maxAcceleratorTextWidth);
		
				TQRect paintDeviceRect;
				TQRect backgroundRect;

				TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_TQMenuBarItem, widget);
				sflags = sflags | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default);
		
				// Draw item
				tqApp->style().drawControl(TQStyle::CE_MenuBarItem, &p, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), sflags, menuOpt);
			}
		}

#if 0
		// FIXME
		// Certain styles (highcolor) paint over the color swatches if this is enabled, rendering the picker useless
		else if (widgetPath.endsWith(".GtkColorSwatch")) {
			TQStringList objectTypes;
			objectTypes.append("TQLineEdit");
			TQPalette objectPalette = tqApp->palette(objectTypes);
	
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_PanelLineEdit, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQLineEdit, widget));
		}
#endif

		else if (widgetPath.endsWith(".add-color-button")) {
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_ButtonBevel, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		}

		else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY)) {
			TQStringList objectTypes;
			objectTypes.append("TQLineEdit");
			TQPalette objectPalette = tqApp->palette(objectTypes);
	
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_PanelLineEdit, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQLineEdit, widget));
		}

		else if (gtk_widget_path_is_type(path, GTK_TYPE_SPIN_BUTTON)) {
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_ButtonBevel, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQPushButton, widget));
		}

		else if ((gtk_widget_path_is_type(path, GTK_TYPE_HANDLE_BOX))
			|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_TOOLBAR))
			) {
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_PanelDockWindow, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		}

		else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUBAR)) {
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_PanelMenuBar, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		}

		else if (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_STATUSBAR)) {
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_StatusBarSection, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		}

		else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENU)) {
			TQStringList objectTypes;
			objectTypes.append("TQPopupMenu");
			TQPalette objectPalette = tqApp->palette(objectTypes);

			TQStyleControlElementData ceData;
			TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
			ceData.widgetObjectTypes = objectTypes;
			ceData.rect = boundingRect;
			ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_PanelPopup, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
		}

		else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_TOOLTIP)) {
			TQStringList objectTypes;
			objectTypes.append("TQToolTip");
			TQPalette objectPalette = tqApp->palette(objectTypes);
	
			int groupBoxLineWidth = 1;
	
			int lineWidth = 0;

			lineWidth = groupBoxLineWidth;

			// Draw tooltip frame
			qDrawPlainRect(&p, boundingRect, ((state & GTK_STATE_FLAG_INSENSITIVE)?objectPalette.disabled():objectPalette.active()).foreground(), lineWidth);
		}

		else if (gtk_widget_path_is_type(path, GTK_TYPE_PROGRESS_BAR)) {
			draw_progressbar_frame(engine, cr, x, y, width, height, path, state, widget);
		}

		else if (gtk_widget_path_is_type(path, GTK_TYPE_IMAGE)) {
			// Do nothing
		}

		else if (gtk_widget_path_is_type(path, GTK_TYPE_LABEL)) {
			// Do nothing
		}

		else if (gtk_widget_path_is_type(path, GTK_TYPE_NOTEBOOK)) {
			// Draw frame
			tqApp->style().drawPrimitive(TQStyle::PE_PanelTabWidget, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		}

#ifdef GTK_TYPE_LEVEL_BAR
		else if (gtk_widget_path_is_type(path, GTK_TYPE_LEVEL_BAR)) {
			const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(widget);

			p.end(); // Restore original Cairo context before creating a new TQt3 paint device based on that context
			boundingRect = TQRect(0, 0, allocation.width, allocation.height);
			TQt3CairoPaintDevice pd2(NULL, 0, 0, allocation.width, allocation.height, cr);
			TQPainter p2(&pd2);

			TQStringList objectTypes;
			objectTypes.append("TQProgressBar");
			TQPalette objectPalette = tqApp->palette(objectTypes);

			TQStyleControlElementData ceData;
			TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
			ceData.widgetObjectTypes = objectTypes;
			ceData.rect = boundingRect;
			ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;
	
			if (gtk_theming_engine_has_class(engine, "fill-block")) {
				// Do nothing
			}
			if (gtk_theming_engine_has_class(engine, "empty-fill-block")) {
				// Draw frame
				p2.setClipRect(TQRect(x, y, width, height));
				tqApp->style().drawPrimitive(TQStyle::PE_PanelLineEdit, &p2, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
			}
		}
#else // GTK_TYPE_LEVEL_BAR
#warning GTK_TYPE_LEVEL_BAR not available in this version of GTK!  Level bars will not be drawn as a result!
#endif // GTK_TYPE_LEVEL_BAR

		else if ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_BACKGROUND))
			|| (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_BOX))
			|| (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_EXPANDER))
			|| (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_VIEWPORT))
			|| (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_SCROLLED_WINDOW))
			|| (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_PANED))
			) {
			// Do nothing
		}

		else {
			DEBUG_FILL_BACKGROUND_WITH_COLOR(p,128,64,255);
			printf("[WARNING] tdegtk_draw_common_frame() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
		}
	}

	if (p.isActive()) p.end();

	cairo_restore(cr);
}

static void
tdegtk_draw_expander (DRAW_ARGS)
{
#if 0
	// FIXME
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	DEBUG_FILL_BACKGROUND_WITH_COLOR(p,255,255,0);
	printf("[WARNING] tdegtk_draw_expander() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);

	if (p.isActive()) p.end();
#else
	GtkStateFlags state;
	GdkRGBA color;
	gint size;
	gdouble angle = G_PI_2;
	
	state = gtk_theming_engine_get_state (engine);
	
	gtk_theming_engine_get_color (engine, state, &color);
	
	cairo_save (cr);
	
	/* use floor function to adjust doubles */
	size = floor (MIN (width, height));
	
	x += (gint) (width / 2) - size / 2;
	y += (gint) (height / 2) - size / 2;
	
	if ((state & GTK_STATE_FLAG_ACTIVE) == 0)
		angle = 0;
	
	cairo_translate (cr, x + size / 2.0 + 0.5, y + size / 2.0 + 0.5);
	cairo_rotate (cr, angle);
	cairo_translate (cr, size / 4.0, 0);
	
	/* FIXME this + 1/- 1 is done to fix blurred diagonal lines.
	* I know it's not nice at all, but it fix a visual bug */
	cairo_move_to (cr, - size / 2.0, - size / 2.0);
	cairo_rel_line_to (cr, size / 2.0 + 1, size / 2.0);
	cairo_rel_line_to (cr, - size / 2.0 - 1, size / 2.0);
	cairo_close_path (cr);
	
	cairo_set_source_rgba (cr, color.red, color.green, color.blue, color.alpha * 0.75);
	cairo_fill_preserve (cr);
	
	gdk_cairo_set_source_rgba (cr, &color);
	cairo_stroke (cr);
	
	cairo_restore (cr);
#endif
}

static void
tdegtk_draw_extension (DRAW_ARGS,
                      GtkPositionType gap_side)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_NOTEBOOK)) {
		// Register tabbar with animation engine
		m_animations.tabWidgetEngine().registerWidget(widget);

		// Check tab properties
		int tabIndex = Gtk::gtk_notebook_find_tab(widget, x+width/2, y+height/2);
		m_animations.tabWidgetEngine().updateTabRect(widget, tabIndex, x, y, width, height);
		bool prelight = (tabIndex == m_animations.tabWidgetEngine().hoveredTab(widget));
		GtkNotebook* notebook = GTK_NOTEBOOK(widget);
// 		bool firstTab = (tabIndex == 0);
// 		bool lastTab = (tabIndex == gtk_notebook_get_n_pages(notebook)-1);
		const int currentPage = gtk_notebook_get_current_page(notebook);
		const int numPages = gtk_notebook_get_n_pages(notebook);
		bool selectedTab = (tabIndex==currentPage);

		TQStringList objectTypes;
		objectTypes.append("TQTabBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;

		TQTab tqt3Tab;
		tqt3Tab.setIdentifier(tabIndex);
		TQStyleOption tabOpt(&tqt3Tab, (prelight)?&tqt3Tab:(TQTab*)NULL);

		elementFlags = elementFlags | TQStyle::CEF_HasParentWidget;
		ceData.parentWidgetData.widgetObjectTypes.append("TQTabBar");

		int tab_overlap = tqApp->style().pixelMetric(TQStyle::PM_TabBarTabOverlap);
		int tab_to_frame_gap_height = 2;

		ceData.tabBarData.tabCount = numPages;
		ceData.tabBarData.identIndexMap[tqt3Tab.identifier()] = tabIndex;

		TQRect paintDeviceRect;
		TQRect backgroundRect;

		switch (gap_side) {
			default:
			case GTK_POS_TOP:
				ceData.tabBarData.shape = TQTabBar::RoundedBelow;

				paintDeviceRect = TQRect(x-tab_overlap, y-tab_to_frame_gap_height, width+(tab_overlap*2), height+tab_to_frame_gap_height);
				boundingRect = TQRect(0, 0, width+tab_overlap, height+tab_to_frame_gap_height);
				backgroundRect = TQRect(tab_overlap, 0, width, height+2);
				break;
			case GTK_POS_LEFT:
				ceData.tabBarData.shape = TQTabBar::RoundedAbove;
				// FIXME
				// TQt3 does not know how to draw these
				// For now, draw a single unconnected tab in this location
				tabIndex = 0;
				ceData.tabBarData.tabCount = 1;
				tqt3Tab.setIdentifier(tabIndex);
				ceData.tabBarData.identIndexMap[tqt3Tab.identifier()] = tabIndex;

				paintDeviceRect = TQRect(x-tab_to_frame_gap_height, y, width+tab_to_frame_gap_height, height);
				boundingRect = TQRect(0, 0, width, height);
				backgroundRect = TQRect(0, 0, width, height);
				break;
			case GTK_POS_BOTTOM:
				ceData.tabBarData.shape = TQTabBar::RoundedAbove;

				paintDeviceRect = TQRect(x-tab_overlap, y, width+(tab_overlap*2), height+tab_to_frame_gap_height);
				boundingRect = TQRect(0, 0, width+tab_overlap, height+tab_to_frame_gap_height);
				backgroundRect = TQRect(tab_overlap, 0, width, height+2);
				break;
			case GTK_POS_RIGHT:
				ceData.tabBarData.shape = TQTabBar::RoundedAbove;
				// FIXME
				// TQt3 does not know how to draw these
				// For now, draw a single unconnected tab in this location
				tabIndex = 0;
				ceData.tabBarData.tabCount = 1;
				tqt3Tab.setIdentifier(tabIndex);
				ceData.tabBarData.identIndexMap[tqt3Tab.identifier()] = tabIndex;

				paintDeviceRect = TQRect(x, y, width+tab_to_frame_gap_height, height);
				boundingRect = TQRect(0, 0, width, height);
				backgroundRect = TQRect(0, 0, width, height);
				break;
		}

		cairo_save(cr);
		cairo_reset_clip(cr);

		p.end(); // Restore original Cairo context before creating a new TQt3 paint device based on that context
		TQt3CairoPaintDevice pd2(NULL, paintDeviceRect.x(), paintDeviceRect.y(), paintDeviceRect.width(), paintDeviceRect.height(), cr);
		TQPainter p2(&pd2);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_SPECIFIC_AREA(p2, brush, backgroundRect.x(), backgroundRect.y(), backgroundRect.width(), backgroundRect.height())

		// Draw tab
		tqApp->style().drawControl(TQStyle::CE_TabBarTab, &p2, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((selectedTab)?TQStyle::Style_Selected:TQStyle::Style_Default) | ((prelight)?TQStyle::Style_MouseOver:TQStyle::Style_Default), tabOpt);

		cairo_restore(cr);
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,255,255,0);
		printf("[WARNING] tdegtk_draw_extension() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_focus (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_FRAME) || gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY) || gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_BUTTON)) && (Gtk::gtk_widget_path_has_type(path, GTK_TYPE_COMBO_BOX))) {
		// Draw nothing!
	}

	else if ((gtk_widget_path_is_type(path, GTK_TYPE_ENTRY))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_TEXT_VIEW))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_TEXT_BUFFER))
		|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY))
		|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_COMBOBOX_ENTRY))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_TREE_VIEW))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_BUTTON))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_EXPANDER))
		) {
		TQStyle::SFlags sflags = TQStyle::Style_Default;

		TQStringList objectTypes;
		if ((gtk_widget_path_is_type(path, GTK_TYPE_ENTRY))
			|| (gtk_widget_path_is_type(path, GTK_TYPE_TEXT_VIEW))
			|| (gtk_widget_path_is_type(path, GTK_TYPE_TEXT_BUFFER))
			|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_ENTRY))
			|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_COMBOBOX_ENTRY))
			) {
			objectTypes.append("TQLineEdit");
		}
		else if (gtk_widget_path_is_type(path, GTK_TYPE_TREE_VIEW)) {
			objectTypes.append("TQListView");
		}
		else if (gtk_widget_path_is_type(path, GTK_TYPE_BUTTON)) {
			objectTypes.append("TQPushButton");
		}
		else {
			objectTypes.append("TQWidget");
		}

		TQPalette objectPalette = tqApp->palette(objectTypes);
		TQColor bgColor;
	
		if (state & GTK_STATE_FLAG_INSENSITIVE) {
			bgColor = objectPalette.color(TQPalette::Disabled, TQColorGroup::Background);
		}
		else {
			bgColor = objectPalette.color(TQPalette::Active, TQColorGroup::Background);
		}

		TQRect focusBoundingRect = boundingRect;
		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;

		if (gtk_widget_path_is_type(path, GTK_TYPE_BUTTON)) {
			TQStringList objectTypes;
			objectTypes.append("TQButton");
			ceData.widgetObjectTypes = objectTypes;
			ceData.rect = boundingRect;

			focusBoundingRect = TQStyle::visualRect(tqApp->style().subRect(TQStyle::SR_PushButtonFocusRect, ceData, elementFlags, NULL), ceData, elementFlags);
		}

		TQStyleOption opt(bgColor);
		// FIXME
		// There are only a few TQt3 widgets that draw a PE_FocusRect directly
		// Those specific widgets need to be isolated and the draw routine added here
		// All other widget should not trigger any drawing here!
		//tqApp->style().drawPrimitive(TQStyle::PE_FocusRect, &p, focusBoundingRect, gtkToTQtColorGroup(engine, state), sflags, opt);
	}

	else if ((gtk_widget_path_is_type(path, GTK_TYPE_ENTRY))
		|| (gtk_widget_path_is_type(path, GTK_TYPE_NOTEBOOK))
		|| (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCALE))
		) {
		// Draw nothing
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,128,255,64);
		printf("[WARNING] tdegtk_draw_focus() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_frame_gap (DRAW_ARGS,
                      GtkPositionType gap_side,
                      gdouble         xy0_gap,
                      gdouble         xy1_gap)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_FRAME)) {
		GtkFrame* frame = GTK_FRAME(widget);
		GtkShadowType gtkShadowType = gtk_frame_get_shadow_type(frame);

		TQStringList objectTypes;
		objectTypes.append("TQGroupBox");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		int groupBoxLineWidth = 1;

		int lineWidth = 0;
		int midLineWidth = 0;
		TQStyle::SFlags sflags = gtkToTQtStyleFlags(engine, state, TQT3WT_TQGroupBox, widget);

		if (gtkShadowType == GTK_SHADOW_NONE) {
			lineWidth = 0;
			midLineWidth = 0;
		}
		else if (gtkShadowType == GTK_SHADOW_IN) {
			lineWidth = groupBoxLineWidth;
			midLineWidth = 0;
			sflags = sflags | TQStyle::Style_Sunken;
		}
		else if (gtkShadowType == GTK_SHADOW_OUT) {
			lineWidth = groupBoxLineWidth;
			midLineWidth = 0;
			sflags = sflags | TQStyle::Style_Raised;
		}
		else if (gtkShadowType == GTK_SHADOW_ETCHED_IN) {
			lineWidth = groupBoxLineWidth;
			midLineWidth = groupBoxLineWidth;
			sflags = sflags | TQStyle::Style_Sunken;
		}
		else if (gtkShadowType == GTK_SHADOW_ETCHED_OUT) {
			lineWidth = groupBoxLineWidth;
			midLineWidth = groupBoxLineWidth;
			sflags = sflags | TQStyle::Style_Raised;
		}

		TQStyleOption opt(lineWidth, midLineWidth);

		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)

		// Draw group box
		tqApp->style().drawPrimitive(TQStyle::PE_PanelGroupBox, &p, boundingRect, gtkToTQtColorGroup(engine, state), sflags, opt);

		// Draw background behind groupbox label
		DRAW_FILLED_RECTANGLE_OVER_SPECIFIC_AREA(p, brush, xy0_gap, x, (xy1_gap-xy0_gap), (height/2))
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,255,128,0);
		printf("[WARNING] tdegtk_draw_frame_gap() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}
	
	if (p.isActive()) p.end();
}

static void
tdegtk_draw_grip (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	TQStringList objectTypes;
	objectTypes.append("TQStatusBar");
	TQPalette objectPalette = tqApp->palette(objectTypes);

	// Draw background
	TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
	DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)

	cairo_save(cr);
	cairo_reset_clip(cr);

	// Draw grip
	tqApp->style().drawPrimitive(TQStyle::PE_SizeGrip, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));

	cairo_restore(cr);

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_handle (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_HANDLE_BOX)) {
		TQStringList objectTypes;
		objectTypes.append("TQDockWindowHandle");
		TQPalette objectPalette = tqApp->palette(objectTypes);
	
		// Draw handle
		tqApp->style().drawPrimitive(TQStyle::PE_DockWindowHandle, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQStyle::Style_Default:TQStyle::Style_Horizontal));
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_PANED)) {
		TQStringList objectTypes;
		objectTypes.append("TQSplitter");
		TQPalette objectPalette = tqApp->palette(objectTypes);
	
		// Draw handle
		tqApp->style().drawPrimitive(TQStyle::PE_Splitter, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQStyle::Style_Default:TQStyle::Style_Horizontal));
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,0,128,255);
		printf("[WARNING] tdegtk_draw_handle() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_line (GtkThemingEngine *engine,
                 cairo_t          *cr,
                 gdouble           x0,
                 gdouble           y0,
                 gdouble           x1,
                 gdouble           y1)
{
	// FIXME
	// This should be using TQt to draw the line with an appropriate line color from the current widget type palette

	// line endings
	if (y0 == y1) {
		y0 += 0.5;
		y1 += 0.5;
		x0 += 0.5;
		x1 -= 0.5;
	}
	else if (x0 == x1) {
		x0 += 0.5;
		x1 += 0.5;
		y0 += 0.5;
		y1 -= 0.5;
	}
	
	cairo_move_to (cr, x0, y0);
	cairo_line_to (cr, x1, y1);
	tdegtk_cairo_set_source_border (engine, cr, MAX (x1 - x0, 1), MAX (y1 - y0, 1));
	cairo_stroke (cr);
}

static void
tdegtk_draw_notebook (DRAW_ARGS,
                     GtkPositionType gap_side,
                     gdouble         xy0_gap,
                     gdouble         xy1_gap)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_NOTEBOOK)) {
		TQStringList objectTypes;
		objectTypes.append("TQTabWidget");
		TQPalette objectPalette = tqApp->palette(objectTypes);
	
		// Draw background
		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)
	
		// Draw tab frame
		tqApp->style().drawPrimitive(TQStyle::PE_PanelTabWidget, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,0,255,0);
		printf("[WARNING] tdegtk_draw_notebook() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}
	
	if (p.isActive()) p.end();
}

static void
tdegtk_draw_radio (DRAW_ARGS)
{
	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_TREE_VIEW)) {
		draw_radiobutton(engine, cr, x, y, width, height, path, state, widget, TQT3WT_GTKTreeViewCell);
	}
	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_MENUITEM)) {
		const GtkAllocation allocation = Gtk::gtk_widget_get_allocation(widget);

		TQRect boundingRect(0, 0, allocation.width, allocation.height);
		TQt3CairoPaintDevice pd(NULL, 0, 0, allocation.width, allocation.height, cr);
		TQPainter p(&pd);

		bool checked = ((state & GTK_STATE_FLAG_ACTIVE) != 0);
		int maxAcceleratorTextWidth = 0;
		int maxIconWidth = IconSize(TDEIcon::Small) + 4;

		TQStringList objectTypes;
		objectTypes.append("TQPopupMenu");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;

		TQMenuItem tqt3MenuItem;
		tqt3MenuItem.setChecked(checked);
		TQStyleOption menuOpt(&tqt3MenuItem, maxIconWidth, maxAcceleratorTextWidth);
		elementFlags = elementFlags | TQStyle::TQStyle::CEF_IsCheckable;

		if (checked) {
			tqApp->style().drawPrimitive(TQStyle::PE_MenuItemIndicatorFrame, &p, ceData, elementFlags, ceData.rect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQRadioButton, widget), menuOpt);
			tqApp->style().drawPrimitive(TQStyle::PE_MenuItemIndicatorCheck, &p, ceData, elementFlags, ceData.rect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQRadioButton, widget), menuOpt);
		}
	}
	else {
		draw_radiobutton(engine, cr, x, y, width, height, path, state, widget, TQT3WT_TQRadioButton);
	}
}

static void
tdegtk_draw_separator (DRAW_ARGS)
{
	cairo_save(cr);
	cairo_reset_clip(cr);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_widget_path_is_type(path, GTK_TYPE_SEPARATOR_MENU_ITEM)) {
		TQStringList objectTypes;
		objectTypes.append("TQPopupMenu");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		bool pressed = false;
		bool prelight = false;

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;

		TQMenuItem tqt3MenuItem;
		tqt3MenuItem.setSeparator(true);
		int maxAcceleratorTextWidth = 0;
		int maxIconWidth = IconSize(TDEIcon::Small) + 4;
		TQStyleOption menuOpt(&tqt3MenuItem, maxIconWidth, maxAcceleratorTextWidth);

// 		if (???) elementFlags = elementFlags | TQStyle::TQStyle::CEF_IsCheckable;

		TQRect paintDeviceRect;
		TQRect backgroundRect;

// 		// Draw background
// 		TQBrush brush = objectPalette.brush(gtkToTQPaletteColorGroup(engine, state), TQColorGroup::Background);
// 		DRAW_FILLED_RECTANGLE_OVER_ENTIRE_AREA(p, brush)

		// HACK
		// GTK 3.7
		// Gtk3 attempts to draw two separators instead of just one
		if ((x==0) && (y==0)) {
			// HACK
			// GTK 3.7
			// Related to the previous hack; the separator should therefore be drawn at the bottom of the provided rectangle instead of at the top
			if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL)) {
				x = x + (width/4);
			}
			else {
				y = y + (height/4);
			}
			p.end(); // Restore original Cairo context before creating a new TQt3 paint device based on that context
			TQt3CairoPaintDevice pd2(NULL, x, y, width, height, cr);
			TQPainter p2(&pd2);

			// Draw item
			tqApp->style().drawControl(TQStyle::CE_PopupMenuItem, &p2, ceData, elementFlags, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_TQMenuItem, widget), menuOpt);
		}
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_SEPARATOR)) {
		TQt::Orientation orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;
		tqApp->style().drawPrimitive(TQStyle::PE_Separator, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_SEPARATOR_TOOL_ITEM)) {
		TQt::Orientation orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;
		tqApp->style().drawPrimitive(TQStyle::PE_Separator, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,128,128,255);
		printf("[WARNING] tdegtk_draw_separator() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();

	cairo_restore(cr);
}

static void
tdegtk_draw_slider (DRAW_ARGS,
                   GtkOrientation orientation)
{
	cairo_save(cr);
	cairo_reset_clip(cr);

	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	const GtkWidgetPath* path;
	GtkStateFlags state;
	GtkWidget* widget;

	path = gtk_theming_engine_get_path(engine);
	state = gtk_theming_engine_get_state(engine);
	widget = m_widgetLookup.find(cr, path);

	if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCROLLBAR)) {
#ifdef DRAW_SCROLLBAR_PIECES_INSTEAD_OF_ENTIRE_BAR
		TQStringList objectTypes;
		objectTypes.append("TQScrollBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		GtkRange* rangeWidget = GTK_RANGE(widget);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;

		gtkRangeToCeData(rangeWidget, ceData);

		// Draw slider
		TQRect scrollpagerect = tqApp->style().querySubControlMetrics(TQStyle::CC_ScrollBar, ceData, elementFlags, TQStyle::SC_ScrollBarSlider, gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget));
		if (ceData.orientation == TQt::Vertical) {
			scrollpagerect.setY(ceData.rect.y());
			scrollpagerect.setHeight(ceData.rect.height());
		}
		else {
			scrollpagerect.setX(ceData.rect.x());
			scrollpagerect.setWidth(ceData.rect.width());
		}
		tqApp->style().drawPrimitive(TQStyle::PE_ScrollBarSlider, &p, scrollpagerect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
#else
// 		draw_scrollbar_frame(engine, cr, x, y, width, height, path, state, widget);
#endif
	}

	else if (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_SCALE)) {
		draw_slider_frame(engine, cr, x, y, width, height, path, state, widget);
	}

	else if (gtk_widget_path_is_type(path, GTK_TYPE_SWITCH)) {
		TQStringList objectTypes;
		objectTypes.append("TQScrollBar");
		TQPalette objectPalette = tqApp->palette(objectTypes);

		TQStyleControlElementData ceData;
		TQStyle::ControlElementFlags elementFlags = TQStyle::CEF_None;
		ceData.widgetObjectTypes = objectTypes;
		ceData.rect = boundingRect;
		ceData.orientation = (gtk_theming_engine_has_class(engine, GTK_STYLE_CLASS_VERTICAL))?TQt::Vertical:TQt::Horizontal;
		ceData.startStep = 0;

		// Draw slider
		//tqApp->style().drawPrimitive(TQStyle::PE_ScrollBarSlider, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
		tqApp->style().drawPrimitive(TQStyle::PE_ButtonBevel, &p, boundingRect, gtkToTQtColorGroup(engine, state), gtkToTQtStyleFlags(engine, state, TQT3WT_NONE, widget) | ((ceData.orientation == TQt::Horizontal)?TQStyle::Style_Horizontal:TQStyle::Style_Default));
	}

	else {
		DEBUG_FILL_BACKGROUND_WITH_COLOR(p,255,0,255);
		printf("[WARNING] tdegtk_draw_slider() nonfunctional for widget with path '%s'\n", gtk_widget_path_to_string (gtk_theming_engine_get_path(engine))); fflush(stdout);
	}

	if (p.isActive()) p.end();

	cairo_restore(cr);
}

static void
tdegtk_draw_spinbutton_background (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	DEBUG_FILL_BACKGROUND_WITH_COLOR(p,128,255,0);

	if (p.isActive()) p.end();
}

static void
tdegtk_draw_spinbutton_frame (DRAW_ARGS)
{
	TQRect boundingRect(0, 0, width, height);
	TQt3CairoPaintDevice pd(NULL, x, y, width, height, cr);
	TQPainter p(&pd);

	DEBUG_FILL_BACKGROUND_WITH_COLOR(p,128,255,128);

	if (p.isActive()) p.end();
}

void
tdegtk_register_style_default (TdeGtkStyleFunctions *functions)
{
	g_assert (functions);
	
	functions->draw_activity                      = tdegtk_draw_activity;
	functions->draw_arrow                         = tdegtk_draw_arrow;
	functions->draw_cell_background               = tdegtk_draw_cell_background;
	functions->draw_cell_frame                    = tdegtk_draw_cell_frame;
	functions->draw_check                         = tdegtk_draw_check;
	functions->draw_common                        = tdegtk_draw_common;
	functions->draw_common_background             = tdegtk_draw_common_background;
	functions->draw_common_frame                  = tdegtk_draw_common_frame;
	functions->draw_expander                      = tdegtk_draw_expander;
	functions->draw_extension                     = tdegtk_draw_extension;
	functions->draw_focus                         = tdegtk_draw_focus;
	functions->draw_frame_gap                     = tdegtk_draw_frame_gap;
	functions->draw_grip                          = tdegtk_draw_grip;
	functions->draw_handle                        = tdegtk_draw_handle;
	functions->draw_line                          = tdegtk_draw_line;
	functions->draw_notebook                      = tdegtk_draw_notebook;
	functions->draw_radio                         = tdegtk_draw_radio;
	functions->draw_separator                     = tdegtk_draw_separator;
	functions->draw_slider                        = tdegtk_draw_slider;
	functions->draw_spinbutton_background         = tdegtk_draw_spinbutton_background;
	functions->draw_spinbutton_frame              = tdegtk_draw_spinbutton_frame;
}
