/*  Ia Ora theme
 *  Copyright (C) 2003 Frederic Crozat - Mandriva
 *                1999 Olivier Fourdan (fourdan@xfce.org) for XFCE code
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <math.h>
#include <gtk/gtk.h>
#include <string.h>

#include "ia_ora_style.h"
#include "ia_ora_rc_style.h"

#define DETAIL(s)   ((detail) && (!strcmp(s, detail)))

#define M_PI 3.14159265358979323846


/* 
  Port to GTK+-2.0 based on Thinice port by Tim Gerla <timg@rrv.net>
      Tomas gren <stric@ing.umu.se> and  Richard Hult <rhult@codefactory.se>
  Portions based on BlueCurve by Alexander Larsson
  Portions based on Raleigh theme by Owen Taylor
  Portions based on Notif theme
  Portions based on Notif2 theme
  Portions based on original GTK theme
 */

static GtkStyleClass *ia_ora_parent_class;

#define RADIO_SIZE 13
#define PART_SIZE 13
#define SCALE_SIZE 5
#define RANGE_TROUGH_SIZE 6
#define SCALE_INCREMENT 12

static const char top_half_cross_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x0c, 0x06, 0x1c, 0x07, 0xb8, 0x03, 0xf0, 0x01,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00 };

static const char bottom_half_cross_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf0, 0x01, 0xb8, 0x03, 0x1c, 0x07, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00 };

static const char * menu_check_xpm[] = {
"9 9 3 1",
"       c None",
".      c #344351",
"+      c #000000",
"        .",
"       ..",
"      ...",
".    ....",
"..  .... ",
".......  ",
"++++++   ",
" ++++    ",
"  ++     "};

static const char * menu_check_active_xpm[] = {
"9 9 3 1",
"       c None",
".      c #FFFFFF",
"+      c #D3DCE3",
"        .",
"       ..",
"      ...",
".    ....",
"..  .... ",
".......  ",
"++++++   ",
" ++++    ",
"  ++     "};


static const char check_top_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x06, 0x00, 0x07, 0x84, 0x03,
   0xcc, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

static const char check_bottom_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xfc, 0x00, 0x78, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00};

static const char radio_up_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x00, 0xf0, 0x01, 0xf8, 0x03,
   0xf8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00 };

static const char radio_down_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xf8, 0x03, 0xf0, 0x01, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00 };

static const char circle_alpha[] = {
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x23,0x62,0x92,0xb3,0xb2,0x95,0x2b,0x00,0x00,0x00,
0x00,0x00,0x3e,0xab,0xc9,0xeb,0xf9,0xf5,0xfd,0xff,0x57,0x00,0x00,
0x00,0x1f,0xb5,0xd8,0xfc,0xff,0xff,0xff,0xff,0xff,0xff,0x2b,0x00,
0x00,0x67,0xb9,0xf2,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x9c,0x00,
0x00,0x9a,0xe2,0xfc,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xe5,0x00,
0x00,0xba,0xeb,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,
0x00,0xc0,0xfa,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xe5,0x00,
0x00,0x9b,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x9c,0x00,
0x00,0x2b,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x2b,0x00,
0x00,0x00,0x57,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x57,0x00,0x00,
0x00,0x00,0x00,0x2b,0x9c,0xe5,0xff,0xe5,0x9c,0x2b,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
};
static const char outline_alpha[] = {
0x00,0x00,0x00,0x4a,0xac,0xe9,0xff,0xe9,0xac,0x4a,0x00,0x00,0x00,
0x00,0x00,0x98,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x98,0x00,0x00,
0x00,0x98,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x98,0x00,
0x4a,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x4a,
0xac,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xac,
0xe9,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xe9,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xe9,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xe9,
0xac,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xac,
0x4a,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x4a,
0x00,0x98,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x98,0x00,
0x00,0x00,0x98,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x98,0x00,0x00,
0x00,0x00,0x00,0x4a,0xac,0xe9,0xff,0xe9,0xac,0x4a,0x00,0x00,0x00,
};

/* internal functions */
static void draw_hline(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x1, gint x2, gint y);
static void draw_vline(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint y1, gint y2, gint x);
static void draw_shadow(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height);

static void draw_box(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height);
static void draw_check(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height);
static void draw_option(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height);
static void draw_shadow_gap(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkPositionType gap_side, gint gap_x, gint gap_width);
static void draw_box_gap(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkPositionType gap_side, gint gap_x, gint gap_width);
static void draw_extension(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkPositionType gap_side);
static void draw_slider(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkOrientation orientation);
static void draw_handle(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkOrientation orientation);
static void draw_tab(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height);

static void ensure_radio_pixmaps (GtkStyle  *style, GtkStateType state, GtkWidget *widget);

static gboolean
ia_ora_object_is_a (const GObject * object, const gchar * type_name)
{
  gboolean result = FALSE;
 
  if ((object))
    {
      GType tmp = g_type_from_name (type_name);

      if (tmp)
        result = g_type_check_instance_is_a ((GTypeInstance *) object, tmp);
    }
 
  return result;
}



static gboolean 
sanitize_size (GdkWindow      *window,
	       gint           *width,
	       gint           *height)
{
  gboolean set_bg = FALSE;

  if ((*width == -1) && (*height == -1))
    {
      set_bg = GDK_IS_WINDOW (window);
      gdk_drawable_get_size (window, width, height);
    }
  else if (*width == -1)
    gdk_drawable_get_size (window, width, NULL);
  else if (*height == -1)
    gdk_drawable_get_size (window, NULL, height);

  return set_bg;
}


static GdkPixbuf *
generate_bit (const char alpha[], GdkColor *color, double mult)
{
  guint r, g, b;
  GdkPixbuf *pixbuf;
  unsigned char *pixels;
  int w, h, rs;
  int x, y;
  
  r = (color->red >> 8) * mult;
  r = MIN(r, 255);
  g = (color->green >> 8) * mult;
  g = MIN(g, 255);
  b = (color->blue >> 8) * mult;
  b = MIN(b, 255);

  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, RADIO_SIZE, RADIO_SIZE);

  w = gdk_pixbuf_get_width (pixbuf);
  h = gdk_pixbuf_get_height (pixbuf);
  rs = gdk_pixbuf_get_rowstride (pixbuf);
  pixels = gdk_pixbuf_get_pixels (pixbuf);


  for (y=0; y < h; y++)
    {
      for (x=0; x < w; x++)
	{
	  pixels[y*rs + x*4 + 0] = r;
	  pixels[y*rs + x*4 + 1] = g;
	  pixels[y*rs + x*4 + 2] = b;
	  if (alpha)
	    pixels[y*rs + x*4 + 3] = alpha[y*w + x];
	  else
	    pixels[y*rs + x*4 + 3] = 255;
	}
    }

  return pixbuf;
}

static void
rgb_to_hls (gdouble *r,
            gdouble *g,
            gdouble *b)
{
  gdouble min;
  gdouble max;
  gdouble red;
  gdouble green;
  gdouble blue;
  gdouble h, l, s;
  gdouble delta;
  
  red = *r;
  green = *g;
  blue = *b;
  
  if (red > green)
    {
      if (red > blue)
        max = red;
      else
        max = blue;
      
      if (green < blue)
        min = green;
      else
        min = blue;
    }
  else
    {
      if (green > blue)
        max = green;
      else
        max = blue;
      
      if (red < blue)
        min = red;
      else
        min = blue;
    }
  
  l = (max + min) / 2;
  s = 0;
  h = 0;
  
  if (max != min)
    {
      if (l <= 0.5)
        s = (max - min) / (max + min);
      else
        s = (max - min) / (2 - max - min);
      
      delta = max -min;
      if (red == max)
        h = (green - blue) / delta;
      else if (green == max)
        h = 2 + (blue - red) / delta;
      else if (blue == max)
        h = 4 + (red - green) / delta;
      
      h *= 60;
      if (h < 0.0)
        h += 360;
    }
  
  *r = h;
  *g = l;
  *b = s;
}

static void
hls_to_rgb (gdouble *h,
            gdouble *l,
            gdouble *s)
{
  gdouble hue;
  gdouble lightness;
  gdouble saturation;
  gdouble m1, m2;
  gdouble r, g, b;
  
  lightness = *l;
  saturation = *s;
  
  if (lightness <= 0.5)
    m2 = lightness * (1 + saturation);
  else
    m2 = lightness + saturation - lightness * saturation;
  m1 = 2 * lightness - m2;
  
  if (saturation == 0)
    {
      *h = lightness;
      *l = lightness;
      *s = lightness;
    }
  else
    {
      hue = *h + 120;
      while (hue > 360)
        hue -= 360;
      while (hue < 0)
        hue += 360;
      
      if (hue < 60)
        r = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
        r = m2;
      else if (hue < 240)
        r = m1 + (m2 - m1) * (240 - hue) / 60;
      else
        r = m1;
      
      hue = *h;
      while (hue > 360)
        hue -= 360;
      while (hue < 0)
        hue += 360;
      
      if (hue < 60)
        g = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
        g = m2;
      else if (hue < 240)
        g = m1 + (m2 - m1) * (240 - hue) / 60;
      else
        g = m1;
      
      hue = *h - 120;
      while (hue > 360)
        hue -= 360;
      while (hue < 0)
        hue += 360;
      
      if (hue < 60)
        b = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
        b = m2;
      else if (hue < 240)
        b = m1 + (m2 - m1) * (240 - hue) / 60;
      else
        b = m1;
      
      *h = r;
      *l = g;
      *s = b;
    }
}

static void
shade (GdkColor * a, GdkColor * b, float k)
{
  gdouble red;
  gdouble green;
  gdouble blue;
  
  red = (gdouble) a->red / 65535.0;
  green = (gdouble) a->green / 65535.0;
  blue = (gdouble) a->blue / 65535.0;
  
  rgb_to_hls (&red, &green, &blue);

  green *= k;
  if (green > 1.0)
    green = 1.0;
  else if (green < 0.0)
    green = 0.0;
  
  blue *= k;
  if (blue > 1.0)
    blue = 1.0;
  else if (blue < 0.0)
    blue = 0.0;

  hls_to_rgb (&red, &green, &blue);
  
  b->red = red * 65535.0;
  b->green = green * 65535.0;
  b->blue = blue * 65535.0;
}

static GtkRequisition default_option_indicator_size = { 7, 13 };
static GtkBorder default_option_indicator_spacing = { 7, 5, 2, 2 };

static void
option_menu_get_props (GtkWidget      *widget,
		       GtkRequisition *indicator_size,
		       GtkBorder      *indicator_spacing)
{
  GtkRequisition *tmp_size = NULL;
  GtkBorder *tmp_spacing = NULL;
  
  if (widget)
    gtk_widget_style_get (widget, 
			  "indicator_size", &tmp_size,
			  "indicator_spacing", &tmp_spacing,
			  NULL);

  if (tmp_size)
    {
      *indicator_size = *tmp_size;
      gtk_requisition_free (tmp_size);
    }
  else
    *indicator_size = default_option_indicator_size;

  if (tmp_spacing)
    {
      *indicator_spacing = *tmp_spacing;
      gtk_border_free (tmp_spacing);
    }
  else
    *indicator_spacing = default_option_indicator_spacing;
}

static void
ia_ora_draw_vgradient (GdkDrawable *drawable, GdkGC *gc, GdkColormap *colormap,
		int x, int y, int width, int height,
		GdkColor *top_color, GdkColor *bottom_color)
{
  int i;
  GdkColor col;
  int dr, dg, db;
  GdkGCValues old_values;

  if (height <= 1)
    return;

  col = *top_color;
  dr = (bottom_color->red - top_color->red) / (height-1);
  dg = (bottom_color->green - top_color->green) / (height-1);
  db = (bottom_color->blue - top_color->blue) / (height-1);
  
  gdk_gc_get_values (gc, &old_values);
  
  for (i = 0; i < height; i++)
    {
      gdk_rgb_find_color (colormap,
			  &col);
      
      gdk_gc_set_foreground (gc, &col);
      
      gdk_draw_line (drawable, gc,
		     x, y + i,
		     x + width - 1, y + i);
					   
      col.red += dr;
      col.green += dg;
      col.blue += db;
    }
  gdk_gc_set_foreground (gc, &old_values.foreground);

}

static void
ia_ora_draw_hgradient (GdkDrawable *drawable, GdkGC *gc, GdkColormap *colormap,
		int x, int y, int width, int height,
		GdkColor *left_color, GdkColor *right_color)
{
  int i;
  GdkColor col;
  int dr, dg, db;
  GdkGCValues old_values;

  if (width <= 1)
    return;

  col = *left_color;
  dr = (right_color->red - left_color->red) / (width-1);
  dg = (right_color->green - left_color->green) / (width-1);
  db = (right_color->blue - left_color->blue) / (width-1);
  
  gdk_gc_get_values (gc, &old_values);
  
  for (i = 0; i < width; i++)
    {
      gdk_rgb_find_color (colormap,
			  &col);
      
      gdk_gc_set_foreground (gc, &col);
      gdk_draw_line (drawable, gc,
		     x + i, y,
		     x + i, y + height - 1);
					   
      col.red += dr;
      col.green += dg;
      col.blue += db;
    }

  gdk_gc_set_foreground (gc, &old_values.foreground);
}

static void ia_ora_print_path (GtkWidget *widget) {
	char *path, *class_path;

	gtk_widget_path (widget, NULL, &path, NULL);
	gtk_widget_class_path (widget, NULL, &class_path, NULL);

	g_print("%s %s\n",path, class_path);
}

static void ia_ora_draw_point (Ia_OraStyle * ia_ora_style, GdkWindow * window, GtkStateType state_type, GdkRectangle *area, GtkWidget *widget, gint x, gint y, gboolean bicolor, gboolean vertical) {
    GdkGC *top_right_color_gc, *top_left_color_gc, *bottom_left_color_gc, *bottom_right_color_gc;
    int color;

    if (state_type == GTK_STATE_INSENSITIVE) {
	    top_left_color_gc = top_right_color_gc = ia_ora_style->gray_gc[5];
	    bottom_left_color_gc = ia_ora_style->gray_gc[2];
	    bottom_right_color_gc = ia_ora_style->gray_gc[2];
    }
    else {
	    if (bicolor) {
		    if (state_type == GTK_STATE_PRELIGHT) {
			    color = 3;
		    } else {
			    color = 2;
		    }

		    top_left_color_gc = ia_ora_style->blue_gc[4];
		    bottom_right_color_gc = ia_ora_style->blue_gc[color];
		    if (vertical) {
			    top_right_color_gc = ia_ora_style->blue_gc[color];
			    bottom_left_color_gc = ia_ora_style->blue_gc[4];

		    } else {
			    top_right_color_gc = ia_ora_style->blue_gc[4];
			    bottom_left_color_gc = ia_ora_style->blue_gc[color];
		    }
	    }
	    else {
		    color = 2;
	    	    top_left_color_gc = top_right_color_gc = ia_ora_style->blue_gc[4];
		    bottom_right_color_gc = ia_ora_style->blue_gc[2];
	    	    bottom_left_color_gc = ia_ora_style->blue_gc[1];
	    }
    }

    if (area) {
	    if (state_type == GTK_STATE_INSENSITIVE) {
    		gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], area);
	    	gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[5], area);
	    } else {
    		gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[1], area);
	    	gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[color], area);
	    	gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[4], area);
	    }
    }

    if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL) {
	    GdkGC *tmp_gc;
	    tmp_gc = bottom_right_color_gc;
	    bottom_right_color_gc = bottom_left_color_gc;
	    bottom_left_color_gc = tmp_gc;
	    tmp_gc = top_right_color_gc;
	    top_right_color_gc = top_left_color_gc;
	    top_left_color_gc = tmp_gc;
    }

    gdk_draw_point (window, top_left_color_gc, x+1, y);
    gdk_draw_line (window, top_left_color_gc, x, y+1, x+1, y+1);

    gdk_draw_point (window, top_right_color_gc, x+2, y);
    gdk_draw_line (window, top_right_color_gc, x+2, y+1, x+3, y+1);

    gdk_draw_line (window, bottom_left_color_gc, x, y+2, x+1, y+2);
    gdk_draw_point (window, bottom_left_color_gc, x+1, y+3);

    gdk_draw_line (window, bottom_right_color_gc, x+2, y+2, x+3, y+2);
    gdk_draw_point (window,bottom_right_color_gc, x+2, y+3);

    if (area) {
	    if (state_type == GTK_STATE_INSENSITIVE) {
	    	gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], NULL);
	    	gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[5], NULL);
	    } else {
	    	gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[1], NULL);
	    	gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[color], NULL);
	    	gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[4], NULL);
	    }
    }
}


static void ia_ora_draw_handlers(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GdkRectangle * area, GtkWidget * widget, gint x, gint y, gint width, gint height, GtkOrientation orientation)
{
    Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);

    int i;
    int delta;

    sanitize_size (window, &width, &height);

    if(orientation == GTK_ORIENTATION_HORIZONTAL) {
        if(width > 32)
        {
            delta = x + (width/2) - 11;
            for(i = 0; i < 3; i++) {
	      ia_ora_draw_point (ia_ora_style, window, state_type, area, widget, delta + 9*i,
				 y + MAX((height-4)/2,0), FALSE, FALSE);
            }
        }
	else {
	  ia_ora_draw_point (ia_ora_style, window,  state_type, area, widget, x + (width-4)/2,
				 y + MAX((height-4)/2,0), FALSE, FALSE);
	}
    }
    else {
        if(height > 32) {
	  delta = y + (height / 2) - 11;
            for(i = 0; i < 3; i++) {
	      ia_ora_draw_point (ia_ora_style, window, state_type, area, widget, x + MAX((width-4)/2,0), 
				 delta+9*i, FALSE, FALSE);
            }
        }
	else {
	  ia_ora_draw_point (ia_ora_style, window, state_type, area, widget, x + MAX((width-4)/2,0), 
			      y + (height-4) / 2, FALSE, FALSE);
	}
    }
}


static void draw_hline(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x1, gint x2, gint y)
{
    Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    if (area) {
        gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], area);
        gdk_gc_set_clip_rectangle(style->white_gc, area);
    }

    gdk_draw_line(window, ia_ora_style->gray_gc[2], x1, y, x2, y);
    gdk_draw_line(window, style->white_gc, x1, y + 1, x2, y+1);

    if(area) {
        gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], NULL);
        gdk_gc_set_clip_rectangle(style->white_gc, NULL);
    }
}


static void draw_vline(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint y1, gint y2, gint x)
{
    Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
    GdkGC *gc1, *gc2;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    if(area)
    {
        gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], area);
        gdk_gc_set_clip_rectangle(style->white_gc, area);
    }

    if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR) {
	    gc1 = ia_ora_style->gray_gc[2];
	    gc2 = style->white_gc;
    }
    else {
	  gc1 = style->white_gc;
	  gc2 = ia_ora_style->gray_gc[2];

    }
    gdk_draw_line(window, gc1, x, y1, x, y2);
    gdk_draw_line(window, gc2, x+1, y1, x+1, y2);

    if(area)
    {
        gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], NULL);
        gdk_gc_set_clip_rectangle(style->white_gc, NULL);
    }
}

static void ia_ora_draw_round_rectangle(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  int color1, color2;
  gboolean is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;

     if (state_type == GTK_STATE_INSENSITIVE || (widget && (GTK_IS_SPIN_BUTTON (widget) || GTK_IS_ENTRY (widget)) && GTK_WIDGET_STATE (widget) == GTK_STATE_INSENSITIVE )) {
	color1 = color2 = 4;
     }
     else {
	color1 = 5;
	color2 = 4;
     }

      if (area) { 
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color1], area);
	if (color1 != color2) 
		gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color2], area);
	gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
      }

    gdk_draw_line(window, ia_ora_style->gray_gc[color1], x+1, y, x+width-2, y);
    gdk_draw_line(window, ia_ora_style->gray_gc[is_ltr ? color1 : color2], x, y+1, x, y+height-2);

    gdk_draw_line(window, ia_ora_style->gray_gc[color2], x+1, y+height-1, x+width-2, y+height-1);
    gdk_draw_line(window, ia_ora_style->gray_gc[is_ltr ? color2 : color1], x+width-1, y+1, x+width-1, y+height-2);

    if (GTK_IS_ENTRY (widget)) {
    	gdk_draw_point(window, style->bg_gc[state_type],x,y);
    	gdk_draw_point(window, style->bg_gc[state_type],x+width-1,y);
    	gdk_draw_point(window, style->bg_gc[state_type],x,y+height-1);
    	gdk_draw_point(window, style->bg_gc[state_type],x+width-1,y+height-1);
    }

      if (area) { 
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color1], NULL);
	if (color1 != color2) 
		gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color2], NULL);
	gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
      }
}

static void ia_ora_draw_half_round_rectangle (GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  gboolean is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;
	int color1, color2;

     if (GTK_WIDGET_STATE (widget) == GTK_STATE_INSENSITIVE) {
	color1 = color2 = 4;
     }
     else {
	color1 = 5;
	color2 = 4;
     }

      if (area) { 
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color1], area);
	if (color1 != color2) 
		gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color2], area);
      }

    x += !is_ltr;

    gdk_draw_line(window, ia_ora_style->gray_gc[color1], x+!is_ltr, y, x+width-(1+is_ltr), y);
    gdk_draw_line(window, ia_ora_style->gray_gc[is_ltr ? color1 : color2], x, y+1, x, y+height-2);

    gdk_draw_line(window, ia_ora_style->gray_gc[color2], x+!is_ltr, y+height-1, x+width-(1+is_ltr), y+height-1);
    gdk_draw_line(window, ia_ora_style->gray_gc[is_ltr ? color2 : color1], x+width-1, y+1, x+width-1, y+height-2);

      if (area) { 
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color1], NULL);
	if (color1 != color2) 
		gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color2], NULL);
      }
}

static void ia_ora_draw_double_gradient_bar (GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkOrientation orientation, gboolean round_beginning, gboolean round_end)
{
	  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
	  GdkColor *top_color1, *bottom_color1,*top_color2, *bottom_color2;
	  GdkGC *top_gc, *bottom_gc;

	  if (!DETAIL("bar") && !DETAIL("menuitem") && state_type == GTK_STATE_PRELIGHT) {
	    top_color1 = &ia_ora_style->blue[0];
	    bottom_color1 = &ia_ora_style->blue[2];
	    top_color2 = &ia_ora_style->blue[4];
	    bottom_color2 = &ia_ora_style->blue[2];
	  }
	  else if (state_type == GTK_STATE_INSENSITIVE) {
	    top_color1 = &ia_ora_style->gray[1];
	    bottom_color1 = &ia_ora_style->gray[3];
	    top_color2 = &ia_ora_style->gray[4];
	    bottom_color2 = &ia_ora_style->gray[2];
	  }
	  else {
	    top_color1 = &ia_ora_style->blue[1];
	    bottom_color1 = &ia_ora_style->blue[3];
	    top_color2 = &ia_ora_style->blue[4];
	    bottom_color2 = &ia_ora_style->blue[2];
	  }

	if (DETAIL("hscale") || (DETAIL("vscale"))) {
	      top_gc = ia_ora_style->gray_gc[5];
	      bottom_gc = ia_ora_style->gray_gc[4];
	}
	else {
	  top_gc = bottom_gc = ia_ora_style->gray_gc[0];
	}

	if (area) {
		gdk_gc_set_clip_rectangle (style->base_gc[state_type], area);
		gdk_gc_set_clip_rectangle (top_gc, area);
		if (bottom_gc != top_gc) 
			gdk_gc_set_clip_rectangle (bottom_gc, area);

	}
	      switch (orientation) {
		      case GTK_ORIENTATION_HORIZONTAL:
			ia_ora_draw_vgradient (window, style->base_gc[state_type], style->colormap,
			        x, y, width, height/2,
		       		top_color1, bottom_color1);
			ia_ora_draw_vgradient (window, style->base_gc[state_type], style->colormap,
			        x, y+height/2, width, height/2+height % 2,
		       		top_color2, bottom_color2);
			break;
	              case GTK_ORIENTATION_VERTICAL:
	      		if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL) {
			  /* inverse gradient */
			  GdkColor *tmp_color;
			  tmp_color = top_color1;
			  top_color1 = bottom_color2;
			  bottom_color2 = tmp_color;
			  tmp_color = top_color2;
			  top_color2 = bottom_color1;
			  bottom_color1 = tmp_color;
			  }

			ia_ora_draw_hgradient (window, style->base_gc[state_type], style->colormap,
			       x, y, width/2, height,
			       top_color1, bottom_color1);
			ia_ora_draw_hgradient (window, style->base_gc[state_type], style->colormap,
			       x+width/2, y, width/2 + width % 2, height,
			       top_color2, bottom_color2);
			break;
		      default:
		      break;
	      }
	if (round_beginning) {
	      gdk_draw_point (window, top_gc,x,y);

		switch (orientation) {
		      case GTK_ORIENTATION_HORIZONTAL:
			      gdk_draw_point (window, bottom_gc,x,y+height-1);
			      break;
		      case GTK_ORIENTATION_VERTICAL:
			      gdk_draw_point (window, bottom_gc,x+width-1,y);
			      break;
		}
	}

	if (round_end) {
	      gdk_draw_point (window, bottom_gc,x+width-1,y+height-1);

		switch (orientation) {
		      case GTK_ORIENTATION_HORIZONTAL:
			      gdk_draw_point (window, top_gc,x+width-1,y);
			      break;
		      case GTK_ORIENTATION_VERTICAL:
			      gdk_draw_point (window, top_gc,x,y+height-1);
			      break;
		}
	}

	if (area) {
		gdk_gc_set_clip_rectangle (style->base_gc[state_type], NULL);
		gdk_gc_set_clip_rectangle (top_gc, NULL);
		if (bottom_gc != top_gc) 
			gdk_gc_set_clip_rectangle (bottom_gc, NULL);
	}
}

static
void ia_ora_treeview_get_header_index (GtkTreeView *tv, GtkWidget *header,
                                    gint *column_index, gint *columns,
                                    gboolean *resizable)
{
        GList *list, *list_start;
        *column_index = *columns = 0;
        list_start = list = gtk_tree_view_get_columns (tv);

        do
        {
                GtkTreeViewColumn *column = GTK_TREE_VIEW_COLUMN(list->data);
                if ( column->button == header )
                {
                        *column_index = *columns;
                        *resizable = column->resizable;
                }
                if ( column->visible )
                        (*columns)++;
        } while ((list = g_list_next(list)));

        g_list_free (list_start);
}

static
void ia_ora_clist_get_header_index (GtkCList *clist, GtkWidget *button,
                                 gint *column_index, gint *columns)
{
	int i;
	*columns = clist->columns;
	
	for (i=0; i<*columns; i++)
	{
		if (clist->column[i].button == button)
		{
			*column_index = i;
			break;
		}
	}
}



static void ia_ora_draw_button (GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{

	Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  	int xthick, ythick;
	gboolean is_combo_button, is_in_tree;
	int color1, color2;
	gboolean disable_gradient = FALSE;
	gboolean is_ltr;

    xthick = style->xthickness;
    ythick = style->ythickness;

    /* workaround for mozilla */
    if (width < 20) {
	xthick = ythick = 1;
    }

    is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;
 
    is_combo_button = (DETAIL("button") 
			&& (GTK_IS_COMBO(gtk_widget_get_parent(widget))
			   || GTK_IS_COMBO_BOX_ENTRY(gtk_widget_get_parent(widget))));

    if ((is_in_tree = (DETAIL ("button") && (GTK_IS_TREE_VIEW(gtk_widget_get_parent(widget)) ||
                  GTK_IS_CLIST (gtk_widget_get_parent(widget)) || ia_ora_object_is_a (G_OBJECT(widget), "ECanvas"))))) {
	xthick = ythick = 0;
    }


      if ((((width > 2*xthick) && (height > 2*ythick)  
	   && IA_ORA_RC_STYLE(style->rc_style)->enable_gradient))
      	&& !(GTK_IS_BUTTON(widget) && GTK_BUTTON (widget)->relief == GTK_RELIEF_NONE && state_type == GTK_STATE_NORMAL)
	&& (disable_gradient = gdk_color_equal(&style->bg[GTK_STATE_NORMAL],&ia_ora_style->gray[0]))) {
	  GdkColor top_color, bottom_color;
	  int is_big = (height > 38);

	  switch (state_type) {
	   case GTK_STATE_PRELIGHT:
	    top_color = style->white;
	    bottom_color = ia_ora_style->gray[1+is_big];
	    break;
	   case GTK_STATE_SELECTED:
	    top_color = ia_ora_style->blue[0];
	    bottom_color = ia_ora_style->blue[3+is_big];
	    break;
	   case GTK_STATE_INSENSITIVE:
	   case GTK_STATE_ACTIVE:
	   default:
	    top_color = ia_ora_style->gray[0];
	    bottom_color = ia_ora_style->gray[2+is_big];
	    break;
	  }

	  if (shadow_type == GTK_SHADOW_IN) {
		/* swap colors, to inverse gradient in this case */
		GdkColor tmp_color;
		
		tmp_color = top_color;
		top_color = bottom_color;
		bottom_color = tmp_color;
	  }

      if (area) { 
	gdk_gc_set_clip_rectangle (style->base_gc[state_type], area);
      }
	  ia_ora_draw_vgradient (window, style->base_gc[state_type], style->colormap, 
			  x+xthick,y+ythick, 
			  width-2*xthick, 
			  height-2*ythick, 
			  &top_color, 
			  &bottom_color);
      if (area) { 
	gdk_gc_set_clip_rectangle (style->base_gc[state_type], NULL);
      }
	}
      else {
	gtk_style_apply_default_background(style, window, TRUE, state_type, 
					   area, x+xthick, 
					   y+ythick, 
					   width-2*xthick, 
					   height-2*ythick);
      }

     if (state_type == GTK_STATE_INSENSITIVE) {
	color1 = color2 = 4;
     }
     else {
	color1 = 5;
	color2 = 4;
     }


      if (area) { 
	gdk_gc_set_clip_rectangle (style->white_gc, area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[1], area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[2], area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[4], area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], area);
      }

      if (xthick > 1 && !is_in_tree) {
	GdkGC *color_gc;

	if ((state_type == GTK_STATE_ACTIVE) || (state_type == GTK_STATE_INSENSITIVE && shadow_type == GTK_SHADOW_IN )) {
		color_gc = ia_ora_style->gray_gc[2];
	}
	else {
		color_gc = style->white_gc;
	}
	gdk_draw_line (window, ia_ora_style->gray_gc[color1],
			 x+!(is_combo_button && is_ltr)*xthick-(is_combo_button && !is_ltr),y,
			 x+width-1-xthick+is_combo_button+(is_combo_button && !is_ltr),y);
	gdk_draw_line (window, color_gc,
			x+xthick-(is_combo_button && is_ltr), y+1,x+width-1-xthick,y+1);
	if (!is_combo_button) {
		gdk_draw_point (window, ia_ora_style->gray_gc[is_ltr ? color1 : color2],
			x+1, y+1);
	}
	gdk_draw_line (window, ia_ora_style->gray_gc[is_ltr ? color1 : color2],
			x, y+xthick-is_combo_button, x, y+height-1-ythick+is_combo_button);
	gdk_draw_line (window, is_ltr ? color_gc : ia_ora_style->gray_gc[1],
			x+1, y+xthick-(is_combo_button && !is_ltr),x+1,y+height-1-ythick);
	if (!is_combo_button) {
		gdk_draw_point (window, ia_ora_style->gray_gc[color2],
			x+1, y+height-2);
	}
	gdk_draw_line (window, ia_ora_style->gray_gc[1],
			x+(is_combo_button ? 1 : xthick),y+height-2,x+width-1 - (is_combo_button ? 1 : xthick), y+height-2);
	gdk_draw_line (window, ia_ora_style->gray_gc[color2],
			 x+!(is_combo_button && is_ltr)*xthick-(is_combo_button && !is_ltr),y+height-1,
			 x+width-xthick-1+is_combo_button+(is_combo_button && !is_ltr),y+height-1);
	gdk_draw_line (window, is_ltr ? ia_ora_style->gray_gc[1] : color_gc,
			x+width-2,y+2-is_combo_button,x+width-2,y+height-ythick-1);
	gdk_draw_line (window, ia_ora_style->gray_gc[is_ltr ? color2 : color1],
			x+width-1, y+ythick-is_combo_button, x+width-1, y+height-ythick-1+is_combo_button);
	if (!is_combo_button) {
		gdk_draw_point (window, ia_ora_style->gray_gc[is_ltr ? color2 : color1],
			x+width-2, y+1);
		gdk_draw_point (window, ia_ora_style->gray_gc[color2],
			x+width-2, y+height-2);
	}
      }

      	if (DETAIL ("optionmenu")) {
	  GtkRequisition indicator_size;
	  GtkBorder indicator_spacing;
	  gint vline_x;
	  
	  option_menu_get_props (widget, &indicator_size, &indicator_spacing);
	  sanitize_size (window, &width, &height);

	  if (!is_ltr) {
	    vline_x = x + indicator_size.width + indicator_spacing.left 
	      + indicator_spacing.right;
	  }
	  else {
	    vline_x = x + width - (indicator_size.width + indicator_spacing.left
				   + indicator_spacing.right 
				   + xthick );
	  }
	  
	  gdk_draw_line (window, is_ltr ? ia_ora_style->gray_gc[2] : style->white_gc,
			 vline_x,y+4,
			 vline_x,y+height-5);
	  gdk_draw_line (window, is_ltr ? style->white_gc : ia_ora_style->gray_gc[2],
			 vline_x+1, y+4,
			 vline_x+1, y+height-5);
	}


      if (is_in_tree) {
		gint columns, column_index;
		gboolean resizable = TRUE;
		columns =  column_index = 1;

		if (GTK_IS_TREE_VIEW (widget->parent))
		{
			ia_ora_treeview_get_header_index (GTK_TREE_VIEW(widget->parent),
							   widget, &column_index, &columns,
							   &resizable);
		}
		else if (GTK_IS_CLIST (widget->parent))
		{
			ia_ora_clist_get_header_index (GTK_CLIST(widget->parent),
							widget, &column_index, &columns);
		}

		if ((is_ltr && (column_index != columns -1)) || (!is_ltr && (column_index != 0))) { /* not the last one */
			gdk_draw_line (window, is_ltr ? ia_ora_style->gray_gc[2] : style->white_gc,
					x+width-1, y, x+width-1, y+height-2);
		}

		if (((is_ltr && (column_index != 0))) || (!is_ltr && (column_index != columns -1))) { /* not the first one */
			gdk_draw_line (window, is_ltr ? style->white_gc : ia_ora_style->gray_gc[2],
					x, y, x, y+height-2);
		}
		gdk_draw_line (window, ia_ora_style->gray_gc[5],
				x, y+height-1, x+width-1, y+height-1);
      }

      if (area) { 
	gdk_gc_set_clip_rectangle (style->white_gc, NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[1], NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[2], NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[4], NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], NULL);
      }

      if (xthick == 1) {
	      ia_ora_draw_round_rectangle(style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);
	      if (disable_gradient) {
	      	if (area) {
			gdk_gc_set_clip_rectangle (style->bg_gc[state_type], area);
	      	}
	        gdk_draw_point(window, style->bg_gc[state_type],x,y);
		gdk_draw_point(window, style->bg_gc[state_type],x+width-1,y);
		gdk_draw_point(window, style->bg_gc[state_type],x,y+height-1);
		gdk_draw_point(window, style->bg_gc[state_type],x+width-1,y+height-1);
		if (area) {
			gdk_gc_set_clip_rectangle (style->bg_gc[state_type], NULL);
		}
	      }

      }

}

static void ia_ora_draw_spinbutton (GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  int xthick, ythick, color;
  GdkColor top_color, bottom_color;

  if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL) {
	      x += 1;
    }

    xthick = ythick = 1;
    if (DETAIL ("spinbutton_down") ) {
     if (state_type == GTK_STATE_INSENSITIVE) {
	color = 4;
     }
     else {
	color = 5;
     }
	if (area) 
		gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color], area);

	  gdk_draw_line (window, ia_ora_style->gray_gc[color], x+1,y,x+width-2,y);

	if (area) 
		gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[color], NULL);
      }
	  /* Make the top button overlap the first line of the bottom button */
	  if (DETAIL ("spinbutton_up")) {
		  	height += 1;
		  }

	  switch (state_type) {
		  case GTK_STATE_PRELIGHT:
	    top_color = style->white;
	    bottom_color = ia_ora_style->gray[1];
	    break;
		  case GTK_STATE_ACTIVE:
	      top_color = ia_ora_style->gray[2];
	      bottom_color = ia_ora_style->gray[0];
	      break;
		  case GTK_STATE_INSENSITIVE:
	    top_color = ia_ora_style->gray[1];
	    bottom_color = ia_ora_style->gray[2];
	    break;
		  default: 
	    top_color = ia_ora_style->gray[0];
	    bottom_color = ia_ora_style->gray[2];
	    break;
	    }

	if (area) 
		gdk_gc_set_clip_rectangle (style->base_gc[state_type], area);

	  ia_ora_draw_vgradient (window, style->base_gc[state_type], style->colormap, 
			  x+xthick,y+xthick,
			  width-2*xthick,
			  height-2*ythick, 
			  &top_color, 
			  &bottom_color);
	if (area) 
		gdk_gc_set_clip_rectangle (style->base_gc[state_type], NULL);

}

static void draw_shadow(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  GdkGC *gc1, *gc2;
  int xthick, ythick;
  gboolean is_ltr;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    if (shadow_type == GTK_SHADOW_NONE) {
      return;
    }

    xthick = style->xthickness;
    ythick = style->ythickness;

    sanitize_size (window, &width, &height);

    if (widget) {
	    is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;
    }
    else {
	    is_ltr = TRUE;
    }

    if ((shadow_type == GTK_SHADOW_ETCHED_IN) || 
	(shadow_type == GTK_SHADOW_ETCHED_OUT)) {

      if (area) {
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[3], area);
        gdk_gc_set_clip_rectangle (style->white_gc, area);
      }

      if (shadow_type == GTK_SHADOW_ETCHED_IN) {
	gc1 = ia_ora_style->gray_gc[3];
	gc2 = style->white_gc;
      }
      else { /*GTK_SHADOW_ETCHED_OUT */
	gc1 = style->white_gc;
	gc2 = ia_ora_style->gray_gc[3];
      }
      gdk_draw_line (window, gc1, x+1+(!is_ltr && (shadow_type == GTK_SHADOW_ETCHED_IN)),y,
		      x+width-2-(is_ltr && shadow_type == GTK_SHADOW_ETCHED_IN),y);
      gdk_draw_line (window, is_ltr ? gc1 : gc2, x,y+1,x,y+height-2);
      gdk_draw_line (window, is_ltr ? gc1 : gc2, x+width-2,y+1,x+width-2,y+height-2);
      gdk_draw_line (window, gc1, x+1,y+height-2,x+width-2,y+height-2);
      
      gdk_draw_line (window, gc2, x+2,y+1,x+width-3,y+1);
      gdk_draw_line (window, is_ltr ? gc2 : gc1, x+width-1,y+1,x+width-1,y+height-2);
      gdk_draw_line (window, gc2, x+1,y+height-1,x+width-2,y+height-1);
      gdk_draw_line (window, is_ltr ? gc2 : gc1, x+1,y+1,x+1,y+height-3);
      if (area) {
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[3], NULL);
        gdk_gc_set_clip_rectangle (style->white_gc, NULL);
      }
    }
    if (DETAIL("entry") || (DETAIL("frame") && GTK_IS_MENU(widget->parent)))
    {
  	int adjust_entry = 0;

	  	if (widget && (GTK_IS_COMBO (widget->parent) ||
		               GTK_IS_COMBO_BOX_ENTRY(widget->parent) ||
		               GTK_IS_SPIN_BUTTON (widget)))
		{
			if (is_ltr)
				width += xthick;
			else {
				x -= 1;
				width +=1;
				adjust_entry = 1;
			}
		}
	    ia_ora_draw_round_rectangle (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);

	    if (widget && DETAIL("entry") && GTK_WIDGET_HAS_FOCUS (widget)) {
      		if (area) {
			gdk_gc_set_clip_rectangle (ia_ora_style->blue_gc[2], area);
		}
		gdk_draw_rectangle (window, ia_ora_style->blue_gc[2], FALSE, x + 1 - 2*adjust_entry,y + 1,width-3 + adjust_entry,height-3);
      		if (area) {
			gdk_gc_set_clip_rectangle (ia_ora_style->blue_gc[2], NULL);
		}
	    }
    } else {

    if (area) {
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[1], area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[2], area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], area);
    }
    
    switch (shadow_type) {
        case GTK_SHADOW_IN:
	      if (widget && widget->parent && (GTK_IS_STATUSBAR (widget->parent))) {
		      draw_hline(style, window, state_type, area, widget, detail, x, x+width,  y);

	      }
	      else {
	  	if (DETAIL("scrolled_window")) 
			gc1 = ia_ora_style->gray_gc[5];
		else
			gc1 = ia_ora_style->gray_gc[2];


	      gdk_draw_line (window, gc1, x+1,y,x+width-2,y);
	      gdk_draw_line (window, gc1, x,y+1,x,y+height-2);
	      gdk_draw_line (window, gc1, x+1,y+height-1,x+width-2,y+height-1);
	      gdk_draw_line (window, gc1, x+width-1,y+1,x+width-1,y+height-2);
	      if (xthick > 1 || ythick > 1) {
		      if (area) {
        		gdk_gc_set_clip_rectangle (style->white_gc, area);
		      }

		      gdk_draw_line (window, style->white_gc, x+1,y+height-2,x+width-2,y+height-2);
		      gdk_draw_line (window, is_ltr ? style->white_gc : ia_ora_style->gray_gc[1], x+width-2,y+2,x+width-2,y+height-3);
  
		      gdk_draw_line (window, ia_ora_style->gray_gc[1], x+1,y+1,x+width-2,y+1);
		      gdk_draw_line (window, is_ltr ? ia_ora_style->gray_gc[1] : style->white_gc, x+1,y+2,x+1,y+height-3);

		      if (area) {
        		gdk_gc_set_clip_rectangle (style->white_gc, NULL);
		      }
	          }
	      }
	  break;
        case GTK_SHADOW_OUT:
		gc1 = ia_ora_style->gray_gc[2];

	      gdk_draw_line (window, gc1, x+1,y,x+width-2,y);
	      gdk_draw_line (window, gc1, x,y+1,x,y+height-2);
	      gdk_draw_line (window, gc1, x+1,y+height-1,x+width-2,y+height-1);
	      gdk_draw_line (window, gc1, x+width-1,y+1,x+width-1,y+height-2);
	      if (xthick > 1 || ythick > 1) {
		if (area) {
        		gdk_gc_set_clip_rectangle (style->white_gc, area);
		}

		      gdk_draw_line (window, ia_ora_style->gray_gc[1], x+1,y+height-2,x+width-2,y+height-2);
		      gdk_draw_line (window, is_ltr ? ia_ora_style->gray_gc[1] : style->white_gc, x+width-2,y+2,x+width-2,y+height-3);
  
		      gdk_draw_line (window, style->white_gc, x+1,y+1,x+width-2,y+1);
		      gdk_draw_line (window, is_ltr ? style->white_gc :ia_ora_style->gray_gc[1], x+1,y+2,x+1,y+height-3);
		if (area) {
        		gdk_gc_set_clip_rectangle (style->white_gc, NULL);
		}
	      }

            break;
    default:
      break;
    }
    
      if (area) {
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[1], NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[2], NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], NULL);
      }
    }
}

static void ia_ora_draw_scale (GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkOrientation orientation, gboolean inverted, gboolean is_progressbar)
{

  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
	int i;
	GdkGC *color_gc;

      if (is_progressbar) {
	     color_gc = ia_ora_style->blue_gc[2];
      } else {
	     color_gc = ia_ora_style->gray_gc[1];
      }

      if (area) {
	gdk_gc_set_clip_rectangle (color_gc, area);
      }

      if (!inverted) {
	switch (orientation) {
		case GTK_ORIENTATION_HORIZONTAL:
			for (i = x + SCALE_INCREMENT; i < x + width-1 ; i += SCALE_INCREMENT) {
				gdk_draw_line (window, color_gc, i,y+1,i,y+height-2);
			}
		break;
		case GTK_ORIENTATION_VERTICAL:
			for (i = y + SCALE_INCREMENT; i < y + height-1; i += SCALE_INCREMENT) {
				gdk_draw_line (window, color_gc, x+1,i,x+width-2,i);
			}
			break;
	}
      } else {
	switch (orientation) {
		case GTK_ORIENTATION_HORIZONTAL:
			for (i = x + width - SCALE_INCREMENT; i > x+1 ; i -= SCALE_INCREMENT) {
				gdk_draw_line (window, color_gc, i,y+1,i,y+height-2);
			}
		break;
		case GTK_ORIENTATION_VERTICAL:
			for (i = y + height - SCALE_INCREMENT; i > y+1 ; i -= SCALE_INCREMENT) {
				gdk_draw_line (window, color_gc, x+1,i,x+width-2,i);
			}
			break;
	}
      }

      if (area) {
	gdk_gc_set_clip_rectangle (color_gc, NULL);
      }
}

static void draw_flat_box(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{

     ia_ora_parent_class->draw_flat_box (style, window, state_type, shadow_type, area,
			      widget, detail, x, y, width, height);

    if (state_type == GTK_STATE_PRELIGHT) {
      if(area) {
	gdk_gc_set_clip_rectangle (style->bg_gc[GTK_STATE_NORMAL], area);
      }
    gdk_draw_point(window, style->bg_gc[GTK_STATE_NORMAL],x,y);
    gdk_draw_point(window, style->bg_gc[GTK_STATE_NORMAL],x+width-1,y);
    gdk_draw_point(window, style->bg_gc[GTK_STATE_NORMAL],x,y+height-1);
    gdk_draw_point(window, style->bg_gc[GTK_STATE_NORMAL],x+width-1,y+height-1);
      if(area) {
	gdk_gc_set_clip_rectangle (style->bg_gc[GTK_STATE_NORMAL], NULL);
      }
     }

}
  

static void draw_box(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
  
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  int xthick, ythick;

  xthick = style->xthickness;
  ythick = style->ythickness;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    sanitize_size(window, &width, &height);

	if (widget && DETAIL("buttondefault")) {
          if(area) {
	    gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[4], area);
           }
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
			      x+xthick+1, y,x+width-2-xthick,y);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
			      x+xthick+1, y+height-1,x+width-2-xthick,y+height-1);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
			      x,y+ythick+1,x,y-2*ythick+height);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
			      x+width-1,y+ythick+1,x+width-1,y-2*ythick+height);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
				x+1,y+2,x+2,y+1);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
				x+1,y+height-3,x+2,y+height-2);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
				x+width-3,y+1,x+width-2,y+2);
		gdk_draw_line(window, ia_ora_style->blue_gc[4],
				x+width-3,y+height-2,x+width-2,y+height-3);

          if(area) {
	    gdk_gc_set_clip_rectangle(ia_ora_style->blue_gc[4], NULL);
           }
	}
 
	else if (widget && 
	(DETAIL ("button")  ||
	 DETAIL ("togglebutton") ||
	 DETAIL("optionmenu"))) /* Button, ComboBox */ {

	    ia_ora_draw_button (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);
      }
    else if (widget && 
	 (DETAIL ("spinbutton_up") || 
	 DETAIL ("spinbutton_down"))) {
	    ia_ora_draw_spinbutton(style, window, state_type, shadow_type, area, widget, detail, x, y, width-1, height);
    }
    else if (widget && 
	 (DETAIL ("spinbutton"))) {
	    ia_ora_draw_half_round_rectangle (style, window, state_type, shadow_type, area, widget, detail, x, y, width-1, height);
    }
    else if (widget &&  
	     (DETAIL ("menubar") || 
	      DETAIL ("toolbar") || 
	      DETAIL ("dockitem_bin")) ) {
      
        gtk_style_apply_default_background (style, window,
				widget && !GTK_WIDGET_NO_WINDOW (widget),
				state_type, area, x, y, width, height);
	if (shadow_type == GTK_SHADOW_OUT) {
          if(area) {
	    gdk_gc_set_clip_rectangle(style->white_gc, area);
	    gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], area);
          }
	  gdk_draw_line (window,style->white_gc,x,y,x+width-1,y);
	  gdk_draw_line (window,ia_ora_style->gray_gc[2],x,y+height-1,x+width-1,y+height-1);	
          if(area) {
	    gdk_gc_set_clip_rectangle(style->white_gc, NULL);
	    gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[2], NULL);
          }
	}

    }
    else if (widget && (DETAIL ("trough-upper") || DETAIL("trough-lower"))) {
	    int color;

	    if DETAIL ("trough-upper")
		    color = 0;
	    else
		    color = 3;

	    if GTK_IS_HSCALE (widget)
		{
		  y += (height - RANGE_TROUGH_SIZE) / 2;
		  height = RANGE_TROUGH_SIZE;
		}
      
	      if GTK_IS_VSCALE (widget)
		{
		  x += (width - RANGE_TROUGH_SIZE) / 2;
		  width = RANGE_TROUGH_SIZE;
		}


	    if (area) {
		  gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[color], area);
	    }

		gdk_draw_rectangle (window, ia_ora_style->gray_gc[color], TRUE, x+1, y+1, width-2, height -2);
	    if (area) {
		  gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[color], NULL);
	    }
		ia_ora_draw_scale (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height, GTK_RANGE (widget)->orientation,DETAIL ("trough-upper") ^ gtk_range_get_inverted(GTK_RANGE(widget)), FALSE );


	  ia_ora_draw_round_rectangle (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);

    }
    else if (widget && DETAIL ("trough")) {
      /* this is the trough used for scrollbar and slider*/
      if GTK_IS_HSCALE (widget)
	{
	  y += (height - SCALE_SIZE) / 2;
	  height = SCALE_SIZE;
	}
      
      if GTK_IS_VSCALE (widget)
	{
	  x += (width - SCALE_SIZE) / 2;
	  width = SCALE_SIZE;
	}


      if (area) {
	  gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[0], area);
	}

	gdk_draw_rectangle (window, ia_ora_style->gray_gc[0],TRUE,
			    x+1,y+1,width-2,height-2);
	if (area) {
	  gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[0], NULL);
	}
      
      if (!GTK_IS_RANGE (widget)) {
	      GtkOrientation orientation;
	      gboolean inverted = FALSE;

	switch (gtk_progress_bar_get_orientation (GTK_PROGRESS_BAR(widget))) {
	      case GTK_PROGRESS_RIGHT_TO_LEFT:
		      inverted = TRUE;
	      case GTK_PROGRESS_LEFT_TO_RIGHT:
		      orientation = GTK_ORIENTATION_HORIZONTAL;
		break;
	
	        case GTK_PROGRESS_BOTTOM_TO_TOP:
			inverted = TRUE;
		case GTK_PROGRESS_TOP_TO_BOTTOM:
		      orientation = GTK_ORIENTATION_VERTICAL;
		break;

	}
	      ia_ora_draw_scale (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height,orientation, inverted, FALSE);
      }
	  ia_ora_draw_round_rectangle (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);

    } 
    else if (widget && (DETAIL ("hscrollbar") || DETAIL ("vscrollbar"))) {
      /* draw steppers, buttons used for scrollbar */
	 GdkColor top_color, bottom_color;

	 if (area) {
	   gdk_gc_set_clip_rectangle (style->base_gc[state_type], area);
	   gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[4], area);
	   gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[5], area);
	 }

	 switch (state_type) {
		 case GTK_STATE_PRELIGHT:
            bottom_color = ia_ora_style->gray[1];
            top_color = style->white;
	    break;
		 case GTK_STATE_ACTIVE:
            bottom_color = ia_ora_style->gray[0];
            top_color = ia_ora_style->gray[1];
	    break;
		 default:
            bottom_color = ia_ora_style->gray[2];
            top_color = ia_ora_style->gray[0];
	    break;
          }

         if (DETAIL ("hscrollbar")) {
	   int color1, color2;

           ia_ora_draw_vgradient (window, style->base_gc[state_type], style->colormap,
                           x+1, y+1, width-2, height-2,
                           &top_color,
                           &bottom_color);

	   if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR) {
		  color1= 4;
		  color2= 5;
	   }  else {
		  color1= 5;
		  color2= 4;
	   }

	   gdk_draw_line (window, ia_ora_style->gray_gc[color1],x,y+1,x,y+height-2);
	   gdk_draw_line (window, ia_ora_style->gray_gc[color2],x+width-1,y+1,x+width-1,y+height-2);

	 } else {

	  if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL) {
		  /* inverse gradient */
		  GdkColor tmp_color;
		  tmp_color = top_color;
		  top_color = bottom_color;
		  bottom_color = tmp_color;

	  }
          ia_ora_draw_hgradient (window, style->base_gc[state_type], style->colormap,
                           x+1, y+1, width-2, height-2,
                           &top_color,
                           &bottom_color);
	  gdk_draw_line (window, ia_ora_style->gray_gc[4],x+1,y,x+width-2,y);
	  gdk_draw_line (window, ia_ora_style->gray_gc[5],x+1,y+height-1,x+width-2,y+height-1);
         }

	 if (area) {
	   gdk_gc_set_clip_rectangle (style->base_gc[state_type], NULL);
	   gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[4], NULL);
	   gdk_gc_set_clip_rectangle(ia_ora_style->gray_gc[5], NULL);
	 }
       }
    else if (widget && DETAIL ("bar")) { /* progress bar */
      if ((height > 2 ) && (width > 2)) {
	 gboolean round_beginning, round_end;
	 gboolean inversed = FALSE;
	 gboolean is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;
	 GtkOrientation orientation;
	 int processed_width = width, processed_height = height;

	 round_beginning = round_end = GTK_PROGRESS(widget)->activity_mode;

	 switch (gtk_progress_bar_get_orientation (GTK_PROGRESS_BAR (widget))) {
		      case GTK_PROGRESS_RIGHT_TO_LEFT:
			      inversed = TRUE;
		      case GTK_PROGRESS_LEFT_TO_RIGHT:
			      orientation = GTK_ORIENTATION_HORIZONTAL;
			      if (!inversed) {
				      processed_width += (!GTK_PROGRESS(widget)->activity_mode) && (((x + width - 2)  % SCALE_INCREMENT) == 0) && (gtk_progress_get_current_percentage (GTK_PROGRESS (widget)) < 1.0);
			      }
			      break;
		        case GTK_PROGRESS_BOTTOM_TO_TOP:
			      inversed = TRUE;
			case GTK_PROGRESS_TOP_TO_BOTTOM:
			      is_ltr = TRUE; /* LTR doesn't matter for vertical */
			      orientation = GTK_ORIENTATION_VERTICAL;
			      if (!inversed) {
				      processed_height += (!GTK_PROGRESS(widget)->activity_mode) && (((y + height - 2)  % SCALE_INCREMENT) == 0) && (gtk_progress_get_current_percentage (GTK_PROGRESS (widget)) < 1.0);
			      }
			      break;
			
			default: /* never reached */
			      break;
        }

	round_end |= (gtk_progress_get_current_percentage (GTK_PROGRESS (widget)) < 1.0) && !inversed;
	round_beginning |= (gtk_progress_get_current_percentage (GTK_PROGRESS (widget)) < 1.0) && inversed;

	ia_ora_draw_double_gradient_bar (style, window, state_type, shadow_type, area, widget, detail, x+1, y+1, processed_width-2, processed_height-2, orientation, is_ltr ? round_beginning : round_end, is_ltr ? round_end : round_beginning);

	if (!GTK_PROGRESS(widget)->activity_mode) {
	      ia_ora_draw_scale (style, window, state_type, shadow_type, area, widget, detail, x, y, width, height, orientation, inversed, TRUE);
        }
      }
    }  
    else if (widget && DETAIL ("menuitem")) {
      ia_ora_draw_double_gradient_bar (style, window, state_type, shadow_type,area,widget,
		      detail, x, y, width, height,GTK_ORIENTATION_HORIZONTAL, TRUE, TRUE);
    } 
    else { /* everything else */
       ia_ora_parent_class->draw_box (style, window, state_type, shadow_type, area,
			      widget, detail, x, y, width, height);	      
    }
}

static void draw_inconsistent_bits(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
    Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
    GdkGC *lower_gc, *top_gc;

    if (state_type == GTK_STATE_INSENSITIVE) {
	    top_gc = ia_ora_style->gray_gc[3];
	    lower_gc = ia_ora_style->gray_gc[4];
    }
    else {
	    	if (IA_ORA_RC_STYLE(style->rc_style)->black_check) {
		    top_gc = ia_ora_style->gray_gc[5];
		    lower_gc = ia_ora_style->gray_gc[6];
		}
		else {
		    top_gc = ia_ora_style->blue_gc[3];
		    lower_gc = ia_ora_style->blue_gc[4];
			}
    }
    if(area) {
	gdk_gc_set_clip_rectangle(top_gc, area);
	gdk_gc_set_clip_rectangle(lower_gc, area);
    }

    gdk_draw_line (window, top_gc, x+4, y+4, x +PART_SIZE-5, y+4);
    gdk_draw_line (window, top_gc, x+3, y+5,x +PART_SIZE-4, y+5);
    gdk_draw_line (window, lower_gc, x+4, y+6,x +PART_SIZE-5, y+6);
    if(area) {
   	gdk_gc_set_clip_rectangle(top_gc, NULL);
    	gdk_gc_set_clip_rectangle(lower_gc, NULL);
    }
}


static void draw_check(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  static GdkBitmap* button_bmap1 = NULL, *button_bmap2 = NULL;
  static GdkPixbuf* menu_pixbuf= NULL, *menu_active_pixbuf = NULL;
  GdkBitmap* bmap1 = NULL, *bmap2 = NULL;


  x -= (1 + PART_SIZE - width) / 2;
  y -= (PART_SIZE - 2  - height) / 2 ;
  
  if (!DETAIL ("check")) {
  if(area) {
    gdk_gc_set_clip_rectangle(style->base_gc[state_type], area);
  }
    gdk_draw_rectangle(window, style->base_gc[state_type == GTK_STATE_INSENSITIVE ? GTK_STATE_INSENSITIVE : GTK_STATE_NORMAL], TRUE, 
		       x+1, y+1, PART_SIZE-2, PART_SIZE-2-1);
  if(area) {
    gdk_gc_set_clip_rectangle(style->base_gc[state_type], NULL);
  }
    
    ia_ora_draw_round_rectangle (style, window, state_type,shadow_type,area,widget,detail,
		 x,y,PART_SIZE,PART_SIZE-1); 

    if (shadow_type == GTK_SHADOW_IN) {
      if (button_bmap1 == NULL) {
   	if (IA_ORA_RC_STYLE(style->rc_style)->use_cross) {
		button_bmap1 = gdk_bitmap_create_from_data(window, top_half_cross_bits,
							  PART_SIZE, PART_SIZE);
	} else {
		button_bmap1 = gdk_bitmap_create_from_data(window, check_top_bits,
							  PART_SIZE, PART_SIZE);
	}
      }
      bmap1 = button_bmap1;
	if (button_bmap2 == NULL) {
   	   if (IA_ORA_RC_STYLE(style->rc_style)->use_cross) {
		button_bmap2 = gdk_bitmap_create_from_data(window, bottom_half_cross_bits,
							  PART_SIZE, PART_SIZE);
	   }
	   else {
		button_bmap2 = gdk_bitmap_create_from_data(window, check_bottom_bits,
							  PART_SIZE, PART_SIZE);
	   }
	}
      bmap2 = button_bmap2;
      }

  }
  else {
     if (menu_pixbuf == NULL) {
	     menu_pixbuf = gdk_pixbuf_new_from_xpm_data (menu_check_xpm);
     }
     if (menu_active_pixbuf == NULL) {
	     menu_active_pixbuf = gdk_pixbuf_new_from_xpm_data (menu_check_active_xpm);
     }
     y += 2;
  }
  
  if (shadow_type == GTK_SHADOW_IN) {
      
  	if(area) {
 	   	gdk_gc_set_clip_rectangle(ia_ora_style->check_gc, area);
  	}

	  if (DETAIL ("check")) {
		  if ((state_type == GTK_STATE_SELECTED) || (state_type == GTK_STATE_ACTIVE)
		     || (state_type == GTK_STATE_PRELIGHT)) {
			  gdk_draw_pixbuf (window, ia_ora_style->check_gc, menu_active_pixbuf, 0,0, x, y, -1, -1,GDK_RGB_DITHER_MAX,0,0);
		  } else {
		  gdk_draw_pixbuf (window, ia_ora_style->check_gc, menu_pixbuf, 0,0, x, y, -1, -1,GDK_RGB_DITHER_MAX,0,0);
		  }
  	}  else {

		gdk_gc_set_stipple(ia_ora_style->check_gc, bmap1);
		gdk_gc_set_ts_origin(ia_ora_style->check_gc, x, y);
		gdk_gc_set_fill(ia_ora_style->check_gc, GDK_STIPPLED);
		if (state_type == GTK_STATE_INSENSITIVE) {
			gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[3]);
		} else {
			if (IA_ORA_RC_STYLE(style->rc_style)->black_check) {
				gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[5]);
			}
			else {
				gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->blue[3]);
			}
		}

		gdk_draw_rectangle(window, ia_ora_style->check_gc, TRUE, x, y, 
				   PART_SIZE, PART_SIZE);

		if (bmap2) {
			gdk_gc_set_stipple(ia_ora_style->check_gc, bmap2);
			if (state_type == GTK_STATE_INSENSITIVE) {
				gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[4]);
			} else {
				if (IA_ORA_RC_STYLE(style->rc_style)->black_check) {
					gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[6]);
				}
				else {
					gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->blue[4]);
				}
			}
			gdk_draw_rectangle(window, ia_ora_style->check_gc, TRUE, x, y, 
				   PART_SIZE, PART_SIZE);
		}

		gdk_gc_set_fill(ia_ora_style->check_gc, GDK_SOLID);
	  }

  	if(area) {
	    gdk_gc_set_clip_rectangle(ia_ora_style->check_gc, NULL);
	  }
      }
  else if (shadow_type == GTK_SHADOW_ETCHED_IN) {
	  draw_inconsistent_bits(style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);
  }

}

static void 
draw_option (GtkStyle      *style,
	     GdkWindow     *window,
	     GtkStateType   state_type,
	     GtkShadowType  shadow_type,
	     GdkRectangle  *area,
	     GtkWidget     *widget,
	     const gchar   *detail,
	     gint           x,
	     gint           y,
	     gint           width,
	     gint           height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  static GdkBitmap *radio_up_bmap = NULL, *radio_down_bmap = NULL;

  if (DETAIL ("option")) /* Menu item */
    {
      ia_ora_parent_class->draw_option (style, window, state_type, shadow_type, area,
				 widget, detail, x, y, width, height);
      return;
    }
    
  if (area) {
    gdk_gc_set_clip_rectangle (style->base_gc[state_type], area);
  }


  ensure_radio_pixmaps(style, state_type, widget);

  x += (width - RADIO_SIZE)/2;
  y += (height - RADIO_SIZE + 2 ) / 2 ;

  gdk_gc_set_clip_mask (style->base_gc[state_type],
			ia_ora_style->radio_pixmap_mask);
  gdk_gc_set_clip_origin (style->base_gc[state_type], x, y);

  gdk_draw_drawable (window, style->base_gc[state_type],
		     ia_ora_style->radio_pixmap_circle[state_type],
		     0, 0,
		     x, y,
		     RADIO_SIZE, RADIO_SIZE);

  gdk_gc_set_clip_origin (style->base_gc[state_type], 0, 0);
  gdk_gc_set_clip_mask (style->base_gc[state_type], NULL);

  if (area) {
    gdk_gc_set_clip_rectangle (style->base_gc[state_type], NULL);
  }


  if (shadow_type == GTK_SHADOW_IN) { /* active*/
   if (radio_up_bmap == NULL) {
	radio_up_bmap = gdk_bitmap_create_from_data(window, radio_up_bits,
						  RADIO_SIZE, RADIO_SIZE);
      }

   if (radio_down_bmap == NULL) {
	radio_down_bmap = gdk_bitmap_create_from_data(window, radio_down_bits,
						  RADIO_SIZE, RADIO_SIZE);
      }

  if(area) {
    gdk_gc_set_clip_rectangle(ia_ora_style->check_gc, area);
  }

	gdk_gc_set_stipple(ia_ora_style->check_gc, radio_up_bmap);
	gdk_gc_set_ts_origin(ia_ora_style->check_gc, x, y);
	gdk_gc_set_fill(ia_ora_style->check_gc, GDK_STIPPLED);
	if (state_type == GTK_STATE_INSENSITIVE) {
		gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[3]);
	} else {
		if (IA_ORA_RC_STYLE(style->rc_style)->black_check) {
			gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[5]);
		}
		else {
			gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->blue[3]);
		}
	}


       	gdk_draw_rectangle(window, ia_ora_style->check_gc, TRUE, x, y, 
			   RADIO_SIZE, RADIO_SIZE);

	gdk_gc_set_stipple(ia_ora_style->check_gc, radio_down_bmap);
	if (state_type == GTK_STATE_INSENSITIVE) {
		gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[4]);
	} else {
		if (IA_ORA_RC_STYLE(style->rc_style)->black_check) {
			gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->gray[6]);
		}
		else {
			gdk_gc_set_foreground(ia_ora_style->check_gc, &ia_ora_style->blue[4]);
		}
	}
	gdk_draw_rectangle(window, ia_ora_style->check_gc, TRUE, x, y, 
			   RADIO_SIZE, RADIO_SIZE);

	gdk_gc_set_fill(ia_ora_style->check_gc, GDK_SOLID);

  if(area) {
    gdk_gc_set_clip_rectangle(ia_ora_style->check_gc, area);
  }
  } else if (shadow_type == GTK_SHADOW_ETCHED_IN) {
	  draw_inconsistent_bits(style, window, state_type, shadow_type, area, widget, detail, x, y+1, width, height);
  }
}


static void draw_shadow_gap(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkPositionType gap_side, gint gap_x, gint gap_width)
{
    GdkRectangle rect;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    draw_shadow(style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);

    switch (gap_side)
    {
        case GTK_POS_TOP:
            rect.x = x + gap_x;
            rect.y = y;
            rect.width = gap_width;
            rect.height = 2;
            break;
        case GTK_POS_BOTTOM:
            rect.x = x + gap_x;
            rect.y = y + height - 2;
            rect.width = gap_width;
            rect.height = 2;
            break;
        case GTK_POS_LEFT:
            rect.x = x;
            rect.y = y + gap_x;
            rect.width = 2;
            rect.height = gap_width;
            break;
        case GTK_POS_RIGHT:
            rect.x = x + width - 2;
            rect.y = y + gap_x;
            rect.width = 2;
            rect.height = gap_width;
            break;
    }

    gtk_style_apply_default_background(style, window, TRUE, state_type, area, rect.x, rect.y, rect.width, rect.height);
}

static void draw_box_gap(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkPositionType gap_side, gint gap_x, gint gap_width)
{
    Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
    gboolean is_ltr;

    GdkGC *gc1 = NULL;
    GdkGC *gc2 = NULL;
    GdkGC *gc3 = NULL;
    GdkGC *gc4 = NULL;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    gtk_style_apply_default_background(style, window, widget && !GTK_WIDGET_NO_WINDOW(widget), state_type, area, x, y, width, height);

    is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;

    sanitize_size(window, &width, &height);

    switch (shadow_type)
    {
        case GTK_SHADOW_NONE:
            return;
        case GTK_SHADOW_IN:
            gc1 = style->bg_gc[state_type];
            gc2 = style->dark_gc[state_type];;
            gc3 = style->bg_gc[state_type];
            gc4 = style->light_gc[state_type];
            break;
        case GTK_SHADOW_ETCHED_IN:
            gc1 = style->dark_gc[state_type];
            gc2 = style->light_gc[state_type];
            gc3 = style->dark_gc[state_type];
            gc4 = style->light_gc[state_type];
            break;
        case GTK_SHADOW_OUT:
            gc1 = ia_ora_style->gray_gc[4];
            gc2 = style->bg_gc[state_type];
            gc3 = style->bg_gc[state_type];
            gc4 = ia_ora_style->gray_gc[4];
            break;
        case GTK_SHADOW_ETCHED_OUT:
            gc1 = style->light_gc[state_type];
            gc2 = style->dark_gc[state_type];
            gc3 = style->light_gc[state_type];
            gc4 = style->dark_gc[state_type];
            break;
    }

    if(area)
    {
        gdk_gc_set_clip_rectangle(gc1, area);
        gdk_gc_set_clip_rectangle(gc2, area);
        gdk_gc_set_clip_rectangle(gc3, area);
        gdk_gc_set_clip_rectangle(gc4, area);
    }

    switch (shadow_type)
    {
        case GTK_SHADOW_NONE:
        case GTK_SHADOW_IN:
        case GTK_SHADOW_OUT:
        case GTK_SHADOW_ETCHED_IN:
        case GTK_SHADOW_ETCHED_OUT:
            switch (gap_side)
            {
                case GTK_POS_TOP:
                    gdk_draw_line(window, gc1, x, y+!is_ltr, x, y + height - 2);
                    gdk_draw_line(window, gc2, x + 1, y, x + 1, y + height - 2);

                    gdk_draw_line(window, gc3, x + 1, y + height - 2, x + width - 2, y + height - 2);
                    gdk_draw_line(window, gc3, x + width - 2, y, x + width - 2, y + height - 2);
                    gdk_draw_line(window, gc4, x+1, y + height - 1, x + width - 2, y + height - 1);
                    gdk_draw_line(window, gc4, x + width - 1, y +is_ltr, x + width - 1, y + height - 2);
                    if(gap_x > 0)
                    {
                        gdk_draw_line(window, gc1, x + 1, y, x + gap_x - 1, y);
                        gdk_draw_line(window, gc2, x + 1, y + 1, x + gap_x - 1, y + 1);
                        gdk_draw_line(window, gc2, x + gap_x, y, x + gap_x, y);
                    }
                    if((width - (gap_x + gap_width)) > 0)
                    {
                        gdk_draw_line(window, gc1, x + gap_x + gap_width, y, x + width - 2, y);
                        gdk_draw_line(window, gc2, x + gap_x + gap_width, y + 1, x + width - 2, y + 1);
                        gdk_draw_line(window, gc2, x + gap_x + gap_width - 1, y, x + gap_x + gap_width - 1, y);
                    }
                    break;
                case GTK_POS_BOTTOM:
                    gdk_draw_line(window, gc1, x+1, y, x + width - 2, y);
                    gdk_draw_line(window, gc1, x, y+1, x, y + height - 1 - !is_ltr);
                    gdk_draw_line(window, gc2, x + 1, y + 1, x + width - 2, y + 1);
                    gdk_draw_line(window, gc2, x + 1, y + 1, x + 1, y + height - 1);

                    gdk_draw_line(window, gc3, x + width - 2, y + 1, x + width - 2, y + height - 1);
                    gdk_draw_line(window, gc4, x + width - 1, y + 1, x + width - 1, y + height - 1 - is_ltr);
                    if(gap_x > 0)
                    {
                        gdk_draw_line(window, gc4, x+1, y + height - 1, x + gap_x - 1, y + height - 1);
                        gdk_draw_line(window, gc3, x + 1, y + height - 2, x + gap_x - 1, y + height - 2);
                        gdk_draw_line(window, gc3, x + gap_x, y + height - 1, x + gap_x, y + height - 1);
                    }
                    if((width - (gap_x + gap_width)) > 0)
                    {
                        gdk_draw_line(window, gc4, x + gap_x + gap_width, y + height - 1, x + width - 2, y + height - 1);
                        gdk_draw_line(window, gc3, x + gap_x + gap_width, y + height - 2, x + width - 2, y + height - 2);
                        gdk_draw_line(window, gc3, x + gap_x + gap_width - 1, y + height - 1, x + gap_x + gap_width - 1, y + height - 1);
                    }
                    break;
                case GTK_POS_LEFT:
                    gdk_draw_line(window, gc1, x, y, x + width - 2, y);
                    gdk_draw_line(window, gc2, x, y + 1, x + width - 2, y + 1);

                    gdk_draw_line(window, gc3, x, y + height - 2, x + width - 2, y + height - 2);
                    gdk_draw_line(window, gc3, x + width - 2, y + 1, x + width - 2, y + height - 2);
                    gdk_draw_line(window, gc4, x+1, y + height - 1, x + width - 2, y + height - 1);
                    gdk_draw_line(window, gc4, x + width - 1, y+1, x + width - 1, y + height - 2);
                    if(gap_x > 0)
                    {
                        gdk_draw_line(window, gc1, x, y, x, y + gap_x - 1);
                        gdk_draw_line(window, gc2, x + 1, y + 1, x + 1, y + gap_x - 1);
                        gdk_draw_line(window, gc2, x, y + gap_x, x, y + gap_x);
                    }
                    if((width - (gap_x + gap_width)) > 0)
                    {
                        gdk_draw_line(window, gc1, x, y + gap_x + gap_width, x, y + height - 2);
                        gdk_draw_line(window, gc2, x + 1, y + gap_x + gap_width, x + 1, y + height - 2);
                        gdk_draw_line(window, gc2, x, y + gap_x + gap_width - 1, x, y + gap_x + gap_width - 1);
                    }
                    break;
                case GTK_POS_RIGHT:
                    gdk_draw_line(window, gc1, x+1, y, x + width - 1, y);
                    gdk_draw_line(window, gc1, x, y+1, x, y + height - 2);
                    gdk_draw_line(window, gc2, x + 1, y + 1, x + width - 1, y + 1);
                    gdk_draw_line(window, gc2, x + 1, y + 1, x + 1, y + height - 2);

                    gdk_draw_line(window, gc3, x + 1, y + height - 2, x + width - 1, y + height - 2);
                    gdk_draw_line(window, gc4, x+1, y + height - 1, x + width - 2, y + height - 1);
                    if(gap_x > 0)
                    {
                        gdk_draw_line(window, gc4, x + width - 1, y, x + width - 1, y + gap_x - 1);
                        gdk_draw_line(window, gc3, x + width - 2, y + 1, x + width - 2, y + gap_x - 1);
                        gdk_draw_line(window, gc3, x + width - 1, y + gap_x, x + width - 1, y + gap_x);
                    }
                    if((width - (gap_x + gap_width)) > 0)
                    {
                        gdk_draw_line(window, gc4, x + width - 1, y + gap_x + gap_width, x + width - 1, y + height - 2);
                        gdk_draw_line(window, gc3, x + width - 2, y + gap_x + gap_width, x + width - 2, y + height - 2);
                        gdk_draw_line(window, gc3, x + width - 1, y + gap_x + gap_width - 1, x + width - 1, y + gap_x + gap_width - 1);
                    }
                    break;
            }
    }

    if(area)
    {
        gdk_gc_set_clip_rectangle(gc1, NULL);
        gdk_gc_set_clip_rectangle(gc2, NULL);
        gdk_gc_set_clip_rectangle(gc3, NULL);
        gdk_gc_set_clip_rectangle(gc4, NULL);
    }
}

static void draw_extension(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkPositionType gap_side)
{ 
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  
    GdkGC *gc1 = NULL;
    gint dx = 0, dy = 0;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);
    gtk_style_apply_default_background (style, window,
					widget && !GTK_WIDGET_NO_WINDOW (widget),
					GTK_STATE_NORMAL, area, x, y, width, height);
    sanitize_size(window, &width, &height);

    switch (shadow_type) {

    case GTK_SHADOW_NONE:
      return;

    case GTK_SHADOW_IN:
    case GTK_SHADOW_ETCHED_IN:
      gc1 = ia_ora_style->gray_gc[5];
      break;

    case GTK_SHADOW_OUT:
    case GTK_SHADOW_ETCHED_OUT:
      if (state_type == GTK_STATE_NORMAL) { /* we draw current page */
	gc1 = ia_ora_style->gray_gc[4] ;
      } else {
	gc1 = ia_ora_style->gray_gc[5];
      }
      break;
    }

    if(area)
    {
        gdk_gc_set_clip_rectangle(gc1, area);
	gdk_gc_set_clip_rectangle (ia_ora_style->blue_gc[3], area);
	gdk_gc_set_clip_rectangle (ia_ora_style->blue_gc[4], area);
    }

    //Unselected tab should be drawn 2 pixels lower than selected tab
    if (state_type == GTK_STATE_ACTIVE)
    {
      dx = 2;
      dy = 2;
    }
    switch (shadow_type)
    {
        case GTK_SHADOW_NONE:
        case GTK_SHADOW_IN:
        case GTK_SHADOW_OUT:
        case GTK_SHADOW_ETCHED_IN:
        case GTK_SHADOW_ETCHED_OUT:
            switch (gap_side)
            {
                case GTK_POS_TOP:
		  gtk_style_apply_default_background (style, window,
						      widget && !GTK_WIDGET_NO_WINDOW (widget),
						      state_type, area,
						      x + style->xthickness + dx, 
						      y, 
						      width - 1 - style->xthickness - dx, 
						      height - style->ythickness - dy);
						      
                    gdk_draw_line (window, gc1, x + dx, y, x + dx, y + height - 3 - dy);
                    gdk_draw_line (window, gc1, x + width - 1, y, x + width - 1, y + height - 3 - dy);
		    
		    if (state_type == GTK_STATE_NORMAL) {
		  	gdk_draw_line (window,ia_ora_style->blue_gc[4] , x + 2 + dx , y + height + dy-1, x + width - 3, y + dy + height-1);
		  	gdk_draw_line (window,ia_ora_style->blue_gc[4] , x + 1 + dx , y + dy+height - 2, x + width - 2, y + dy + height -2);
		  	gdk_draw_line (window,ia_ora_style->blue_gc[3] , x + dx , y + dy + height - 3, x + width - 1, y + dy + height-3);
		  	gdk_draw_line (window,ia_ora_style->blue_gc[3] , x + dx , y + dy + height - 4, x + width - 1, y + dy + height-4);
		 }
		 else {
		    gdk_draw_point (window, gc1, x+1 + dx, y+height-2 - dy);
                    gdk_draw_line(window, gc1, x + 2 + dx, y + height - 1 - dy, x + width - 3, y + height - 1 - dy);
		    gdk_draw_point (window, gc1, x+width-2,y+height-2 - dy);
		 }

                    break;
                case GTK_POS_BOTTOM:
		  gtk_style_apply_default_background (style, window,
						      widget && !GTK_WIDGET_NO_WINDOW (widget),
						      state_type, area,
						      x + style->ythickness + dx,
						      y + style->ythickness + dy,
						      width - 1 - style->xthickness - dx,
						      height - style->ythickness - dy);
		  gdk_draw_line (window, gc1, x + dx, y + 2 + dy, x + dx, y + height - 1 );

		  gdk_draw_line (window, gc1, x + width - 1, y + 2 + dy, 
				 x + width - 1, y + height - 1);
		 if (state_type == GTK_STATE_NORMAL) {
		  	gdk_draw_line (window,ia_ora_style->blue_gc[3] , x + 2 + dx , y + dy, x + width - 3, y + dy);
		  	gdk_draw_line (window,ia_ora_style->blue_gc[3] , x + 1 + dx , y + dy+1, x + width - 2, y + dy+1);
		  	gdk_draw_line (window,ia_ora_style->blue_gc[4] , x + dx , y + dy+2, x + width - 1, y + dy+2);
		  	gdk_draw_line (window,ia_ora_style->blue_gc[4] , x + dx , y + dy+3, x + width - 1, y + dy+3);


		 }
		 else {

		  gdk_draw_point (window, gc1, x + 1 + dx, y+1+dy);
		  gdk_draw_line (window, gc1, x + 2 + dx , y + dy, x + width - 3, y + dy);
		    
		  gdk_draw_point (window, gc1, x + width-2, y + 1 + dy);
		  }
		  break;
                case GTK_POS_LEFT:
		  gtk_style_apply_default_background (style, window,
						      widget && !GTK_WIDGET_NO_WINDOW (widget),
						      state_type, area,
						      x, 
						      y + style->ythickness + dx,
						      width - style->xthickness - dy,
						      height - style->ythickness - dx);
                    gdk_draw_line (window, gc1, x, y + dx, x + width - 3 - dy, y+ dx);
                    gdk_draw_line (window, gc1, x, y + height - 1, x + width - 3 - dy, y + height - 1);

  			if (state_type == GTK_STATE_NORMAL) {
                    		gdk_draw_line (window, ia_ora_style->blue_gc[4], x + width - 1 - dy, y + 2 + dx, x + width - 1 - dy, y + height - 3);
                    		gdk_draw_line (window, ia_ora_style->blue_gc[4], x + width - 2 - dy, y + 1 + dx, x + width - 2 - dy, y + height - 2);
                    		gdk_draw_line (window, ia_ora_style->blue_gc[3], x + width - 3 - dy, y + dx, x + width - 3 - dy, y + height - 1);
                    		gdk_draw_line (window, ia_ora_style->blue_gc[3], x + width - 4 - dy, y + dx, x + width - 4 - dy, y + height - 1);
			}		    
			else {
		    		gdk_draw_point (window, gc1, x+width-2 - dy,y+1 + dx);
		                gdk_draw_point (window, gc1, x + width - 2 - dy, y + height - 2);
                    		gdk_draw_line (window, gc1, x + width - 1 - dy, y + 2 + dx, x + width - 1 - dy, y + height - 3);
			}
                    break;
                case GTK_POS_RIGHT:
		  gtk_style_apply_default_background (style, window,
						      widget && !GTK_WIDGET_NO_WINDOW (widget),
						      state_type, area,
						      x + style->xthickness + dy,
						      y + style->ythickness + dx,
						      width  - style->xthickness - dy,
						      height - style->ythickness - dx);

                    gdk_draw_line (window, gc1, x + 2 + dy, y + dx, x + width - 1, y + dx);
                    gdk_draw_line (window, gc1, x+ 2 + dy, y + height - 1, x + width - 1, y + height - 1);
 	  	if (state_type == GTK_STATE_NORMAL) {
                    gdk_draw_line (window, ia_ora_style->blue_gc[3], x + dy, y + 2 + dx, x + dy, y + height - 3);
                    gdk_draw_line (window, ia_ora_style->blue_gc[3], x + dy+1, y + 1 + dx, x + dy+1, y + height - 2);
                    gdk_draw_line (window, ia_ora_style->blue_gc[4], x + dy+2, y + dx, x + dy+2, y + height - 1);
                    gdk_draw_line (window, ia_ora_style->blue_gc[4], x + dy+3, y + dx, x + dy+3, y + height - 1);
		} else {
		    gdk_draw_point (window, gc1, x+1 + dy, y+1 + dx);
                    gdk_draw_line (window, gc1, x + dy, y + 2 + dx, x + dy, y + height - 3);
                    gdk_draw_point (window, gc1, x+1 + dy, y +height-2);
		}
                    break;
            }
    }

    if(area)
    {
        gdk_gc_set_clip_rectangle(gc1, NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->blue_gc[3], NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->blue_gc[4], NULL);
    }
}

static void draw_slider(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkOrientation orientation)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  GtkAdjustment *adj; 
  gboolean is_horizontal;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    sanitize_size(window, &width, &height);

    orientation = GTK_ORIENTATION_HORIZONTAL;
    if(height > width)
        orientation = GTK_ORIENTATION_VERTICAL;
    

    /* Make sure stepper and slider outlines "overlap" */
      if (widget && GTK_IS_RANGE (widget))
    {
      adj = GTK_RANGE (widget)->adjustment;

      if (adj->value <= adj->lower &&
	  (GTK_RANGE (widget)->has_stepper_a ||
	   GTK_RANGE (widget)->has_stepper_b))
	{
	  if (GTK_IS_VSCROLLBAR (widget))
	    {
	      height += 1;
	      y -= 1;
	    }
	  else if (GTK_IS_HSCROLLBAR (widget))
	    {
	      width += 1;
	      x -= 1;
	    }
	}
      if (adj->value >= adj->upper - adj->page_size &&
	  (GTK_RANGE (widget)->has_stepper_c ||
	   GTK_RANGE (widget)->has_stepper_d))
	{
	  if (GTK_IS_VSCROLLBAR (widget))
	    {
	      height += 1;
	    }
	  else if (GTK_IS_HSCROLLBAR (widget))
	    {
	      width += 1;
	    }
	}
    }

    is_horizontal = orientation == GTK_ORIENTATION_HORIZONTAL;


    if (DETAIL ("slider")) { /* this is for scrollbar */
     adj = GTK_RANGE (widget)->adjustment;
    	ia_ora_draw_double_gradient_bar(style, window, state_type, shadow_type, area, widget, detail, x+1, y+1, width-1-!is_horizontal, height-1-is_horizontal, orientation, adj->value > adj->lower, adj->value < adj->upper - adj->page_size);
    } 
    else { if (DETAIL("hscale") || DETAIL("vscale")) {
  	GdkGC *top_gc, *bottom_gc;
	gboolean is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;

    	if (state_type == GTK_STATE_PRELIGHT) {
		top_gc = ia_ora_style->blue_gc[1];
		bottom_gc = ia_ora_style->blue_gc[2];
	    }
	    else {
		top_gc = ia_ora_style->blue_gc[2];
		bottom_gc = ia_ora_style->blue_gc[4];
	    }

        if (area) {
	      gdk_gc_set_clip_rectangle (top_gc, area);
	      gdk_gc_set_clip_rectangle (bottom_gc, area);
	}

	    if (is_horizontal) {
		    gdk_draw_rectangle (window, top_gc, TRUE, x+1, y+1, width-1, (height-1)/2);
		    gdk_draw_rectangle (window, bottom_gc, TRUE, x+1, y+(height+1)/2, width-1, (height-1)/2);
	    } else {
		    gdk_draw_rectangle (window, is_ltr ? top_gc : bottom_gc , TRUE, x+1, y+1, (width-1) / 2, height-1);
		    gdk_draw_rectangle (window, is_ltr ? bottom_gc : top_gc, TRUE, x+(width+1)/2, y+1, (width-1)/2, height-1);
	    }

        if (area) {
	      gdk_gc_set_clip_rectangle (top_gc, NULL);
	      gdk_gc_set_clip_rectangle (bottom_gc, NULL);
	}

    if (area) {
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[4],area);
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5],area);
    }
		switch (orientation) {
		      case GTK_ORIENTATION_HORIZONTAL:
			      gdk_draw_point (window, ia_ora_style->gray_gc[4],x+1,y+height-2);
			      gdk_draw_point (window, ia_ora_style->gray_gc[4],x+width-1,y+height-2);
			      gdk_draw_point (window, ia_ora_style->gray_gc[5],x+1,y+1);
			      gdk_draw_point (window, ia_ora_style->gray_gc[5],x+width-1,y+1);
			      break;
		      case GTK_ORIENTATION_VERTICAL:
			      gdk_draw_point (window, ia_ora_style->gray_gc[is_ltr ? 5 : 4],x+1,y+1);
			      gdk_draw_point (window, ia_ora_style->gray_gc[is_ltr ? 4 : 5],x+width-2,y+1);
			      gdk_draw_point (window, ia_ora_style->gray_gc[is_ltr ? 5 : 4],x+1,y+height-1);
			      gdk_draw_point (window, ia_ora_style->gray_gc[is_ltr ? 4 : 5],x+width-2,y+height-1);
			      break;
		}
    if (area) {
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[4],NULL);
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5],NULL);
    }

	    if (DETAIL ("hscale"))  {
     		ia_ora_draw_point(ia_ora_style, window, state_type, area, widget, x+width/2-1,y+2, TRUE, FALSE);
    		}
	    else if (DETAIL ("vscale")) { 
     		ia_ora_draw_point(ia_ora_style, window, state_type, area, widget, x+2,y+height/2-1, TRUE, TRUE);
    		}
    	}
    }

}

static void draw_handle(GtkStyle * style, GdkWindow * window, GtkStateType state_type, GtkShadowType shadow_type, GdkRectangle * area, GtkWidget * widget, const gchar * detail, gint x, gint y, gint width, gint height, GtkOrientation orientation)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  gboolean is_panel_applet_frame;

    g_return_if_fail(style != NULL);
    g_return_if_fail(window != NULL);

    sanitize_size(window, &width, &height);

    is_panel_applet_frame = widget && strcmp (g_type_name (G_TYPE_FROM_INSTANCE (widget)), "PanelAppletFrame") == 0;

    if (DETAIL ("dockitem") || is_panel_applet_frame) {
      /* Work around orientation bugs */
      if (orientation == GTK_ORIENTATION_VERTICAL) {
	orientation = GTK_ORIENTATION_HORIZONTAL;
      }
      else {
	orientation = GTK_ORIENTATION_VERTICAL;
      }
    }

    if (DETAIL ("paned") || is_panel_applet_frame) {
      gtk_style_apply_default_background(style, window, TRUE, state_type, 
					 area, x, 
					 y, 
					 width, 
					 height);
    }
    else if (DETAIL("handlebox")) {
    if(area) {
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[0], area);
      }

	gdk_draw_rectangle (window, ia_ora_style->gray_gc[0], TRUE, x, y, width, height);
    if(area) {
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[0], NULL);
      }

    }
    else {
      }

  if (is_panel_applet_frame) {
	  ia_ora_draw_round_rectangle(style, window, state_type, shadow_type, area, widget, detail, x, y, width, height);
  }
  else
if (!DETAIL ("paned")) {
    if(area) {
	gdk_gc_set_clip_rectangle (style->white_gc, area);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[3], area);
      }
      gdk_draw_line (window, ia_ora_style->gray_gc[3], x,y+height-1,
		     x+width-1,y+height-1);
      gdk_draw_line (window, style->white_gc, x,y,x+width-1,y);

      if(area) {
	gdk_gc_set_clip_rectangle (style->white_gc, NULL);
	gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[3], NULL);
      }
    }
    ia_ora_draw_handlers(style, window, state_type, area, widget, x, y, width, height, orientation);

}


static void
draw_resize_grip (GtkStyle       *style,
		  GdkWindow      *window,
		  GtkStateType    state_type,
		  GdkRectangle   *area,
		  GtkWidget      *widget,
		  const gchar    *detail,
		  GdkWindowEdge   edge,
		  gint            x,
		  gint            y,
		  gint            width,
		  gint            height)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  cairo_t *cr;
  cairo_matrix_t matrix;
  gboolean drawn = FALSE;

  g_return_if_fail (GTK_IS_STYLE (style));
  g_return_if_fail (window != NULL);

  width -= 1;
  height -= 1;


  switch (edge) {
   case GDK_WINDOW_EDGE_NORTH_WEST:
      /* make it square */
      if (width < height)
	{
	  height = width;
	}
      else if (height < width)
	{
	  width = height;
	}
      break;
    case GDK_WINDOW_EDGE_NORTH:
      if (width < height)
	{
	  height = width;
	}
      break;
    case GDK_WINDOW_EDGE_NORTH_EAST:
      /* make it square, aligning to top right */
      if (width < height)
	{
	  height = width;
	}
      else if (height < width)
	{
	  x += (width - height);
	  width = height;
	}
      break;
    case GDK_WINDOW_EDGE_WEST:
      if (height < width)
	{
	   width = height;
	}
      break;
    case GDK_WINDOW_EDGE_EAST:
      /* aligning to right */
      if (height < width)
	{
	  x += (width - height);
	  width = height;
	}
      break;
    case GDK_WINDOW_EDGE_SOUTH_WEST:
      /* make it square, aligning to bottom left */
      if (width < height)
	{
	  y += (height - width);
	  height = width;
	}
      else if (height < width)
	{
	  width = height;
	}
      break;
    case GDK_WINDOW_EDGE_SOUTH:
      /* align to bottom */
      if (width < height)
	{
	  y += (height - width);
	  height = width;
	}
      break;
    case GDK_WINDOW_EDGE_SOUTH_EAST:
      /* make it square, aligning to bottom right */
      if (width < height)
	{
	  y += (height - width);
	  height = width;
	}
      else if (height < width)
	{
	  x += (width - height);
	  width = height;
	}
      x+=3;
      y+=2;
      break;
    default:
      g_assert_not_reached ();
    }


  /* Clear background */
  gtk_style_apply_default_background (style, window, FALSE,
				      state_type, area,
				      x, y, width, height);   

  cr = gdk_cairo_create (window);

   cairo_set_line_width (cr, 1);
  cairo_set_antialias(cr, CAIRO_ANTIALIAS_NONE);


	cairo_matrix_init_translate(&matrix, 0, 0.5);
       cairo_transform (cr, &matrix);


  switch (edge) {
    case GDK_WINDOW_EDGE_WEST:
    case GDK_WINDOW_EDGE_EAST:
      {
	gint xi;

	xi = x;

  if (area)
    {
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], area);
      gdk_gc_set_clip_rectangle (style->white_gc, area);
    }
	while (xi < x + width)
	  {
	    gdk_draw_line (window,
			   style->white_gc,
			   xi, y,
			   xi, y + height);

	    xi++;
	    gdk_draw_line (window,
			   ia_ora_style->gray_gc[5],
			   xi, y,
			   xi, y + height);

	    xi += 2;
	  }
	drawn = TRUE;
      }
      break;
    case GDK_WINDOW_EDGE_NORTH:
    case GDK_WINDOW_EDGE_SOUTH:
      {
	gint yi;

	yi = y;

  if (area)
    {
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], area);
      gdk_gc_set_clip_rectangle (style->white_gc, area);
    }
	while (yi < y + height)
	  {
	    gdk_draw_line (window,
			   style->white_gc,
			   x, yi,
			   x + width, yi);

	    yi++;
	    gdk_draw_line (window,
			   ia_ora_style->gray_gc[5],
			   x, yi,
			   x + width, yi);

	    yi+= 2;
	  }
	drawn = TRUE;
      }
      break;
    case GDK_WINDOW_EDGE_SOUTH_EAST:
      break;
    case GDK_WINDOW_EDGE_SOUTH_WEST:
      cairo_matrix_init(&matrix, -1,0,0,1,2*x+width-0.5-1,2);
      cairo_transform (cr, &matrix);
      break; 
    case GDK_WINDOW_EDGE_NORTH_WEST:
      cairo_matrix_init(&matrix, -1,0,0,-1,2*x+width,2*y+height);
      cairo_transform (cr, &matrix);
      break; 
    case GDK_WINDOW_EDGE_NORTH_EAST:
      cairo_matrix_init(&matrix, 1,0,0,-1,0,2*y+height);
      cairo_transform (cr, &matrix);
      break; 
    default:
      g_assert_not_reached ();
      break;
    }

   if (!drawn) {
  if (area)
    {
      gdk_cairo_rectangle (cr, area);
      cairo_clip (cr);
    }

   gdk_cairo_set_source_color (cr, &ia_ora_style->blue[3]);
      cairo_move_to (cr, x,              y + height);
      cairo_line_to (cr, x + width,      y);
      cairo_line_to (cr, x + width,      y+height);
  cairo_close_path (cr);
  cairo_fill (cr);
   gdk_cairo_set_source_color (cr, &ia_ora_style->blue[2]);
      cairo_move_to (cr, x+width,              y);
      cairo_line_to (cr, x + 3,      y+height-3);
      cairo_line_to (cr, x + 6,      y+height-3);
      cairo_line_to (cr, x+6,              y+height-4);
      cairo_line_to (cr, x + 9,      y+height-4);
      cairo_line_to (cr, x+ 10,              y+height-5);
      cairo_line_to (cr, x + 11,      y+height-5);
      cairo_line_to (cr, x + width - 1,      y+height-9);
      cairo_line_to (cr, x + width - 1,      y+1);
  cairo_fill (cr);
  
      cairo_move_to (cr, x + width-1 , y+1);
      cairo_line_to (cr, x + width , y+1);
      cairo_stroke_preserve(cr);
      cairo_move_to (cr, x+width-3, y+height);
      cairo_line_to (cr, x+width, y+height-3);
      cairo_line_to (cr, x+width, y+height);
      cairo_line_to (cr, x+width-3, y+height);
      cairo_fill (cr);

  cairo_destroy (cr);
    }
   else {

    if (area) {
      gdk_gc_set_clip_rectangle (ia_ora_style->gray_gc[5], NULL);
      gdk_gc_set_clip_rectangle (style->white_gc, NULL);
    }
  }
}

static gboolean
is_combo_box_entry (GtkWidget * widget)
{
	gboolean result = FALSE;
	
	if (widget && widget->parent)
	{
		if (GTK_IS_COMBO_BOX_ENTRY (widget))
			result = TRUE;
		else
			result = is_combo_box_entry(widget->parent);
	}
	
	return result;
}

static gboolean
is_combo_box (GtkWidget * widget)
{
	gboolean result = FALSE;
	
	if (widget && widget->parent)
	{
		if (GTK_IS_COMBO_BOX (widget))
			result = TRUE;
		else
			result = is_combo_box (widget->parent);
	}
	
	return result;
}

static gboolean
is_deprecated_combo(GtkWidget * widget)
{
	gboolean result = FALSE;
	
	if (widget && widget->parent)
	{
		if (GTK_IS_COMBO(widget))
			result = TRUE;
		else
			result = is_deprecated_combo(widget->parent);
	}
	
	return result;
}

static void
calculate_arrow_geometry (GtkArrowType  arrow_type,
			  gint         *x,
			  gint         *y,
			  gint         *width,
			  gint         *height)
{
  gint w = *width;
  gint h = *height;
  
  switch (arrow_type)
    {
    case GTK_ARROW_UP:
    case GTK_ARROW_DOWN:
      w += (w % 2) - 1;
      h = (w / 2 + 1);
      
      if (h > *height)
	{
	  h = *height;
	  w = 2 * h - 1;
	}
      
      if (arrow_type == GTK_ARROW_DOWN)
	{
	  if (*height % 2 == 1 || h % 2 == 0)
	    *height += 1;
	}
      else
	{
	  if (*height % 2 == 0 || h % 2 == 0)
	    *height -= 1;
	}
      break;

    case GTK_ARROW_RIGHT:
    case GTK_ARROW_LEFT:
      h += (h % 2) - 1;
      w = (h / 2 + 1);
      
      if (w > *width)
	{
	  w = *width;
	  h = 2 * w - 1;
	}
      
      if (arrow_type == GTK_ARROW_RIGHT)
	{
	  if (*width % 2 == 1 || w % 2 == 0)
	    *width += 1;
	}
      else
	{
	  if (*width % 2 == 0 || w % 2 == 0)
	    *width -= 1;
	}
      break;
      
    default:
      /* should not be reached */
      break;
    }

  *x += (*width - w) / 2;
  *y += (*height - h) / 2;
  *height = h;
  *width = w;
}

static void draw_arrow (GtkStyle      *style,
			GdkWindow     *window,
			GtkStateType   state,
			GtkShadowType  shadow,
			GdkRectangle  *area,
			GtkWidget     *widget,
			const gchar   *detail,
			GtkArrowType   arrow_type,
			gboolean       fill,
			gint           x,
			gint           y,
			gint           width,
			gint           height) {
  gboolean combo_box, combo_box_entry, is_deprecated;
  gint i, j;
  gboolean is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;

    if (area) {
      gdk_gc_set_clip_rectangle (style->fg_gc[state], area);
    }

  if (DETAIL ("spinbutton")) {
	 width += 1;
	 height += 1;
	 if (arrow_type == GTK_ARROW_DOWN)
 	    y +=1;

  }
  combo_box_entry = is_combo_box_entry(widget);
  combo_box = is_combo_box(widget);
  is_deprecated = is_deprecated_combo (widget);

  if (combo_box_entry) {
	  width += 3;
	  y -= 1;
	  x -= 1;
  }

  if (combo_box || combo_box_entry || is_deprecated) {
    if (combo_box || is_deprecated) {
	  width -= 1;
	  height-= 1;
    }

    if (combo_box && !combo_box_entry) {
	if (is_ltr) {
		x += 1;
	 }
	else {
		x -= 1;
	}
    }
    /* same design as KDE arrow, add a line */
    x += 1;
    y += 2;
    gdk_draw_line (window, style->fg_gc[state],x,y,x+width-(2-width%2),y);
    if (combo_box_entry || combo_box) {
	y += 1;
    }
  }


  sanitize_size (window, &width, &height);

  calculate_arrow_geometry (arrow_type, &x, &y, &width, &height);

  if (detail && strcmp (detail, "menu_scroll_arrow_up") == 0)
    y++;

  if (arrow_type == GTK_ARROW_DOWN)
    {
      for (i = 0, j = 0; i < height; i++, j++)
	gdk_draw_line (window, style->fg_gc[state], x + j, y + i, x + width - j - 1, y + i);
    }
  else if (arrow_type == GTK_ARROW_UP)
    {
      for (i = height - 1, j = 0; i >= 0; i--, j++)
	gdk_draw_line (window, style->fg_gc[state], x + j, y + i, x + width - j - 1, y + i);
    }
  else if (arrow_type == GTK_ARROW_LEFT)
    {
      for (i = width - 1, j = 0; i >= 0; i--, j++)
	gdk_draw_line (window, style->fg_gc[state], x + i, y + j, x + i, y + height - j - 1);
    }
  else if (arrow_type == GTK_ARROW_RIGHT)
    {
      for (i = 0, j = 0; i < width; i++, j++)
	gdk_draw_line (window, style->fg_gc[state], x + i, y + j, x + i, y + height - j - 1);
    }


    if (area) {
      gdk_gc_set_clip_rectangle (style->fg_gc[state], NULL);
    }

}


static void 
draw_tab (GtkStyle      *style,
	     GdkWindow     *window,
	     GtkStateType   state_type,
	     GtkShadowType  shadow_type,
	     GdkRectangle  *area,
	     GtkWidget     *widget,
	     const gchar   *detail,
	     gint           x,
	     gint           y,
	     gint           width,
	     gint           height)
{

  if (DETAIL ("optionmenutab")) {
      y += 1;
  }
  ia_ora_parent_class->draw_tab (style, window, state_type, shadow_type, area,
			  widget, detail, x, y, width, height);
    
}

static void
draw_expander (GtkStyle        *style,
               GdkWindow       *window,
               GtkStateType     state_type,
               GdkRectangle    *area,
               GtkWidget       *widget,
               const gchar     *detail,
               gint             x,
               gint             y,
	       GtkExpanderStyle expander_style) 
{
  GtkStyle *style_copy;
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);

  style_copy = gtk_style_copy (style);

  style_copy->fg[GTK_STATE_NORMAL] = style_copy->bg[GTK_STATE_SELECTED];
  style_copy->fg[GTK_STATE_PRELIGHT] = style_copy->bg[GTK_STATE_SELECTED];
  style_copy->light[GTK_STATE_ACTIVE] = ia_ora_style->blue[4];
  style_copy->fg[GTK_STATE_ACTIVE] = ia_ora_style->blue[4];

  ia_ora_parent_class->draw_expander (style_copy, window, state_type, area,
			  widget, detail, x, y, expander_style);

  gtk_style_unref (style_copy);
    
}



static GdkPixbuf*
scale_or_ref (GdkPixbuf *src,
              int width,
              int height)
{
	if (width == gdk_pixbuf_get_width (src) &&
	    height == gdk_pixbuf_get_height (src)) {
		return g_object_ref (src);
	} else {
		return gdk_pixbuf_scale_simple (src,
						width, height,
						GDK_INTERP_BILINEAR);
	}
}

static GdkPixbuf *
set_transparency (const GdkPixbuf *pixbuf, gdouble alpha_percent)
{
	GdkPixbuf *target;
	guchar *data, *current;
	guint x, y, rowstride, height, width;

	g_return_val_if_fail (pixbuf != NULL, NULL);
	g_return_val_if_fail (GDK_IS_PIXBUF (pixbuf), NULL);

	/* Returns a copy of pixbuf with it's non-completely-transparent pixels to
	   have an alpha level "alpha_percent" of their original value. */

	target = gdk_pixbuf_add_alpha (pixbuf, FALSE, 0, 0, 0);

	if (alpha_percent == 1.0)
		return target;
	width = gdk_pixbuf_get_width (target);
	height = gdk_pixbuf_get_height (target);
	rowstride = gdk_pixbuf_get_rowstride (target);
	data = gdk_pixbuf_get_pixels (target);

	for (y = 0; y < height; y++) {
		for (x = 0; x < width; x++) {
			/* The "4" is the number of chars per pixel, in this case, RGBA,
			   the 3 means "skip to the alpha" */
			current = data + (y * rowstride) + (x * 4) + 3; 
			*(current) = (guchar) (*(current) * alpha_percent);
		}
	}

	return target;
}


static GdkPixbuf *
render_icon (GtkStyle            *style,
	     const GtkIconSource *source,
	     GtkTextDirection     direction,
	     GtkStateType         state,
	     GtkIconSize          size,
	     GtkWidget           *widget,
	     const char          *detail)
{
	int width = 1;
	int height = 1;
	GdkPixbuf *scaled;
	GdkPixbuf *stated;
	GdkPixbuf *base_pixbuf;
	GdkScreen *screen;
	GtkSettings *settings;
	
	/* Oddly, style can be NULL in this function, because
	 * GtkIconSet can be used without a style and if so
	 * it uses this function.
	 */
	
	base_pixbuf = gtk_icon_source_get_pixbuf (source);
	
	g_return_val_if_fail (base_pixbuf != NULL, NULL);
	
	if (widget && gtk_widget_has_screen (widget)) {
		screen = gtk_widget_get_screen (widget);
		settings = gtk_settings_get_for_screen (screen);
	} else if (style->colormap) {
		screen = gdk_colormap_get_screen (style->colormap);
		settings = gtk_settings_get_for_screen (screen);
	} else {
		settings = gtk_settings_get_default ();
		GTK_NOTE (MULTIHEAD,
			  g_warning ("Using the default screen for gtk_default_render_icon()"));
	}
	
  
	if (size != (GtkIconSize) -1 && !gtk_icon_size_lookup_for_settings (settings, size, &width, &height)) {
		g_warning (G_STRLOC ": invalid icon size '%d'", size);
		return NULL;
	}

	/* If the size was wildcarded, and we're allowed to scale, then scale; otherwise,
	 * leave it alone.
	 */
	if (size != (GtkIconSize)-1 && gtk_icon_source_get_size_wildcarded (source))
		scaled = scale_or_ref (base_pixbuf, width, height);
	else
		scaled = g_object_ref (base_pixbuf);
	
	/* If the state was wildcarded, then generate a state. */
	if (gtk_icon_source_get_state_wildcarded (source)) {
		if (state == GTK_STATE_INSENSITIVE) {
			stated = set_transparency (scaled, 0.3);
			gdk_pixbuf_saturate_and_pixelate (stated, stated,
							  0.1, FALSE);
			
			g_object_unref (scaled);
		} else if (state == GTK_STATE_PRELIGHT) {
			stated = gdk_pixbuf_copy (scaled);      
			
			gdk_pixbuf_saturate_and_pixelate (scaled, stated,
							  1.2, FALSE);
			
			g_object_unref (scaled);
		} else {
			stated = scaled;
		}
	}
	else
		stated = scaled;
  
  return stated;
}


static GdkGC *
realize_color (GtkStyle * style,
	       GdkColor * color)
{
  GdkGCValues gc_values;

  gdk_colormap_alloc_color (style->colormap, color,
			    FALSE, TRUE);

  gc_values.foreground = *color;

  return gtk_gc_get (style->depth, style->colormap,
		     &gc_values, GDK_GC_FOREGROUND);
}
static GdkPixmap *
pixbuf_to_pixmap (GtkStyle * style, GdkPixbuf *pixbuf, GdkScreen *screen)
{
  GdkGC *tmp_gc;
  GdkPixmap *pixmap;
  
  pixmap = gdk_pixmap_new (gdk_screen_get_root_window (screen),
			   gdk_pixbuf_get_width (pixbuf),
			   gdk_pixbuf_get_height (pixbuf),
			   style->depth);
  gdk_drawable_set_colormap (pixmap, style->colormap);

  tmp_gc = gdk_gc_new (pixmap);
  
  gdk_pixbuf_render_to_drawable (pixbuf, pixmap, tmp_gc, 0, 0, 0, 0,
				 gdk_pixbuf_get_width (pixbuf),
				 gdk_pixbuf_get_height (pixbuf),
				 GDK_RGB_DITHER_NORMAL, 0, 0);
  g_object_unref (tmp_gc);

  return pixmap;
}

static
void ensure_radio_pixmaps (GtkStyle     *style,
			   GtkStateType  state,
			   GtkWidget *widget)
{ 
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  GdkPixbuf *circle, *outline, *composite;
  GdkColor *composite_color;  
  GdkScreen *screen;
  unsigned int i;
  gboolean is_ltr;

  if (ia_ora_style->radio_pixmap_circle[state] != NULL)
    return;

  screen = gtk_widget_get_screen (widget);
  is_ltr = gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR;

  outline = generate_bit (outline_alpha, &ia_ora_style->gray[4], 1.0);

  if (!is_ltr) {
	  GdkPixbuf *tmp_pixbuf;

	  tmp_pixbuf = gdk_pixbuf_flip (outline, TRUE);
	  g_object_unref (outline);
	  outline = tmp_pixbuf;
  }

  if (ia_ora_style->radio_pixmap_mask == NULL) {
    gdk_pixbuf_render_pixmap_and_mask (outline,
   				       NULL,
				       &ia_ora_style->radio_pixmap_mask,
				       1);
  }
  
  for (i = 0; i < G_N_ELEMENTS(ia_ora_style->radio_pixmap_circle); i++)
    {
      if (i == GTK_STATE_ACTIVE)
	{
	  composite_color = &style->bg[GTK_STATE_PRELIGHT];
	  circle = generate_bit (circle_alpha, &style->bg[i], 1.0);
	}   
      else
	{
	  composite_color = &style->bg[i];
	  circle = generate_bit (circle_alpha, &style->white, 1.0);
	}

      composite = generate_bit (NULL, composite_color, 1.0);
      if (!is_ltr) {
	  GdkPixbuf *tmp_pixbuf;

	  tmp_pixbuf = gdk_pixbuf_flip (circle, TRUE);
	  g_object_unref (circle);
	  circle = tmp_pixbuf;

	  tmp_pixbuf = gdk_pixbuf_flip (composite, TRUE);
	  g_object_unref (composite);
	  composite = tmp_pixbuf;
      }


      gdk_pixbuf_composite (outline, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      gdk_pixbuf_composite (circle, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);

      ia_ora_style->radio_pixmap_circle[i] = pixbuf_to_pixmap (style, composite, screen);
      
      g_object_unref (composite);
      g_object_unref (circle);
    }
  g_object_unref (outline);
	
}

static void ia_ora_style_init_from_rc (GtkStyle* style, GtkRcStyle * rc_style)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  char *colorname;
  unsigned int i;
 /*double shading[] = {1.036875969, 1.0 , 0.893708749, 0.822125139,
		      0.74403433, 0.71800443, 0.646420819,0.587006644};*/
  ia_ora_parent_class->init_from_rc (style, rc_style);

  for (i=0 ; i<G_N_ELEMENTS(ia_ora_style->gray); i++) {
   colorname = g_strdup_printf("gray%d",i);
   gtk_style_lookup_color (style, colorname, &ia_ora_style->gray[i]);
   g_free (colorname);
  }
   

  for (i=0 ; i<G_N_ELEMENTS(ia_ora_style->blue); i++) {
   colorname = g_strdup_printf("blue%d",i);
   gtk_style_lookup_color (style, colorname, &ia_ora_style->blue[i]);
   g_free (colorname);
  }

/*  for (i=0; i<G_N_ELEMENTS (ia_ora_style->gray); i++) {
    shade(&style->bg[GTK_STATE_NORMAL],&ia_ora_style->gray[i],shading[i]);
  }*/



/*
  ia_ora_style->blue[2].red = style->base[GTK_STATE_SELECTED].red;
  ia_ora_style->blue[2].green = style->base[GTK_STATE_SELECTED].green;
  ia_ora_style->blue[2].blue = style->base[GTK_STATE_SELECTED].blue;

  ia_ora_style->blue[0].red = style->base[GTK_STATE_ACTIVE].red;
  ia_ora_style->blue[0].green = style->base[GTK_STATE_ACTIVE].green;
  ia_ora_style->blue[0].blue = style->base[GTK_STATE_ACTIVE].blue;


  shade(&style->base[GTK_STATE_SELECTED],&ia_ora_style->blue[1],1.204945256);*/

  shade(&style->white,&ia_ora_style->check_color,0.2);

}

static void ia_ora_style_realize(GtkStyle * style)
{
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
 
  unsigned int i;

  ia_ora_parent_class->realize (style);

  ia_ora_style->check_gc = realize_color (style, &ia_ora_style->check_color);

  for (i=0; i < G_N_ELEMENTS(ia_ora_style->gray_gc) ; i++) {
    ia_ora_style->gray_gc[i] = realize_color (style, &ia_ora_style->gray[i]);
  }

  for (i=0; i < G_N_ELEMENTS(ia_ora_style->blue_gc) ; i++) {
    ia_ora_style->blue_gc[i] = realize_color (style, &ia_ora_style->blue[i]);
  }
}

static void
ia_ora_style_copy (GtkStyle * style, GtkStyle * src)
{
  Ia_OraStyle * ia_ora_style = IA_ORA_STYLE (style);
  Ia_OraStyle * ia_ora_src = IA_ORA_STYLE (src);
  unsigned int i;


  for (i=0; i < G_N_ELEMENTS(ia_ora_src->gray) ; i++) {
    ia_ora_style->gray[i] = ia_ora_src->gray[i];
  }

  for (i=0; i < G_N_ELEMENTS(ia_ora_src->blue) ; i++) {
    ia_ora_style->blue[i] = ia_ora_src->blue[i];
  }

  ia_ora_style->check_color = ia_ora_src->check_color;

  for (i=0; i < G_N_ELEMENTS(ia_ora_src->gray_gc) ; i++) {
    if (ia_ora_src->gray_gc[i]) {
      ia_ora_style->gray_gc[i] = g_object_ref (ia_ora_src->gray_gc[i]);
    }
  }

  for (i=0; i < G_N_ELEMENTS(ia_ora_src->blue_gc) ; i++) {
    if (ia_ora_src->blue_gc[i]) {
      ia_ora_style->blue_gc[i] = g_object_ref (ia_ora_src->blue_gc[i]);
    }
  }

  if (ia_ora_src->check_gc) {
    ia_ora_style->check_gc = g_object_ref (ia_ora_src->check_gc);
  }

  for (i=0; i < G_N_ELEMENTS(ia_ora_src->radio_pixmap_circle) ; i++) {
	  if (ia_ora_src->radio_pixmap_circle[i])
    		ia_ora_style->radio_pixmap_circle[i] = g_object_ref (ia_ora_src->radio_pixmap_circle[i]);
  }

  if (ia_ora_src->radio_pixmap_mask)
	  ia_ora_style->radio_pixmap_mask = g_object_ref (ia_ora_src->radio_pixmap_mask);

  ia_ora_parent_class->copy (style, src);
}


static void ia_ora_style_unrealize(GtkStyle * style)
{  
  Ia_OraStyle *ia_ora_style = IA_ORA_STYLE (style);
  unsigned int i;

  /* We don't free the colors, because we don't know if
   * gtk_gc_release() actually freed the GC. FIXME - need
   * a way of ref'ing colors explicitely so GtkGC can
   * handle things properly.
   */

  for (i=0; i < G_N_ELEMENTS(ia_ora_style->gray_gc) ; i++) {
    gtk_gc_release (ia_ora_style->gray_gc[i]);
  }

  for (i=0; i < G_N_ELEMENTS(ia_ora_style->blue_gc) ; i++) {
    gtk_gc_release (ia_ora_style->blue_gc[i]);
  }

  for (i = 0; i < G_N_ELEMENTS(ia_ora_style->radio_pixmap_circle); i++) {
    if (ia_ora_style->radio_pixmap_circle[i] != NULL) {
      g_object_unref (ia_ora_style->radio_pixmap_circle[i]);
      ia_ora_style->radio_pixmap_circle[i] = NULL;
    }
  }

  gtk_gc_release (ia_ora_style->check_gc);

  if (ia_ora_style->radio_pixmap_mask != NULL) {
    g_object_unref (ia_ora_style->radio_pixmap_mask);
  }
  ia_ora_style->radio_pixmap_mask = NULL;

  ia_ora_parent_class->unrealize (style);
}

static void ia_ora_style_class_init(Ia_OraStyleClass * klass)
{
    GtkStyleClass *style_class = GTK_STYLE_CLASS(klass);

    ia_ora_parent_class = g_type_class_peek_parent(klass);

    style_class->init_from_rc = ia_ora_style_init_from_rc;
    style_class->realize = ia_ora_style_realize;
    style_class->copy = ia_ora_style_copy;
    style_class->unrealize = ia_ora_style_unrealize;

    style_class->draw_hline = draw_hline;
    style_class->draw_vline = draw_vline;
    style_class->draw_shadow = draw_shadow;
    style_class->draw_box = draw_box;
    style_class->draw_flat_box = draw_flat_box;
    style_class->draw_check = draw_check;
    style_class->draw_option = draw_option;
    style_class->draw_shadow_gap = draw_shadow_gap;
    style_class->draw_box_gap = draw_box_gap;
    style_class->draw_extension = draw_extension;
    style_class->draw_slider = draw_slider;
    style_class->draw_handle = draw_handle;
    style_class->draw_resize_grip = draw_resize_grip;
    style_class->draw_arrow = draw_arrow;
    style_class->draw_tab = draw_tab;
    style_class->draw_expander = draw_expander;
    style_class->render_icon = render_icon;
}


static void
ia_ora_style_init (Ia_OraStyle * style)
{
}

GType ia_ora_type_style = 0;

void ia_ora_style_register_type(GTypeModule * module)
{
    static const GTypeInfo object_info = {
        sizeof(Ia_OraStyleClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) ia_ora_style_class_init,
        NULL,                   /* class_finalize */
        NULL,                   /* class_data */
        sizeof(Ia_OraStyle),
        0,                      /* n_preallocs */
        (GInstanceInitFunc) ia_ora_style_init,
	NULL
    };

    ia_ora_type_style = g_type_module_register_type(module, GTK_TYPE_STYLE, 
						    "Ia_OraStyle", 
						    &object_info, 0);
}

