/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
** Copyright (c) 2001 Phil Thompson <phil@river-bank.demon.co.uk>
** Copyright (c) 2002 Riverbank Computing Limited <info@riverbankcomputing.co.uk>
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "uic.h"
#include <tqfile.h>
#include <tqimage.h>
#include <tqstringlist.h>
#include <tqdatetime.h>
#include <tqfileinfo.h>
#include <stdio.h>
#include <ctype.h>

// on embedded, we do not compress image data. Rationale: by mapping
// the ready-only data directly into memory we are both faster and
// more memory efficient
#if defined(TQ_WS_QWS) && !defined(TQT_NO_IMAGE_COLLECTION_COMPRESSION)
#define TQT_NO_IMAGE_COLLECTION_COMPRESSION
#endif

struct EmbedImage
{
    ~EmbedImage() { delete[] colorTable; }
    int width, height, depth;
    int numColors;
    TQRgb* colorTable;
    TQString name;
    TQString cname;
    bool alpha;
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
    ulong compressed;
#endif
};

static TQString convertToCIdentifier( const char *s )
{
    TQString r = s;
    int len = r.length();
    if ( len > 0 && !isalpha( (char)r[0].latin1() ) )
	r[0] = '_';
    for ( int i=1; i<len; i++ ) {
	if ( !isalnum( (char)r[i].latin1() ) )
	    r[i] = '_';
    }
    return r;
}


static ulong embedData( TQTextStream& out, const uchar* input, int nbytes )
{
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
    TQByteArray bazip( tqCompress( input, nbytes ) );
    uint len = bazip.size();
#else
    uint len = (uint)nbytes;
#endif
    static const char hexdigits[] = "0123456789abcdef";
    TQString s;
    for ( uint i=0; i<len; i++ ) {
	if ( (i%14) == 0 ) {
	    if (i)
		s += '"';

	    s += " \\\n    \"";
	    out << (const char*)s;
	    s.truncate( 0 );
	}
        uint v = (uchar)
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
                 bazip
#else
                 input
#endif
                 [(int)i];
	s += "\\x";
	s += hexdigits[(v >> 4) & 15];
	s += hexdigits[v & 15];
	if ( i == len-1 )
	    s += "\"\n";
    }
    if ( s.length() )
	out << (const char*)s;
    return (ulong)len;
}

static void embedData( TQTextStream& out, const TQRgb* input, int n )
{
    out << hex;
    const TQRgb *v = input;
    for ( int i=0; i<n; i++ ) {
	if ( (i%14) == 0  )
	    out << "\n    ";
	out << "0x";
	out << hex << *v++;
	if ( i < n-1 )
	    out << ',';
    }
    out << dec; // back to decimal mode
}

void Uic::embed( TQTextStream& out, const char* project, const TQStringList& images )
{

    TQString cProject = convertToCIdentifier( project );

    TQStringList::ConstIterator it;
    out << "# Image collection for project '" << project << "'.\n";
    out << "#\n";
    out << "# Generated from reading image files: \n";
    for ( it = images.begin(); it != images.end(); ++it )
	out << "#      " << *it << "\n";
    out << "#\n";
    out << "# Created by: The PyTQt User Interface Compiler (pytquic) " << PYTQT_VERSION << "\n";
    out << "#\n";
    out << "# WARNING! All changes made in this file will be lost!\n";

    out << "\n";
    out << "\n";
    out << "from PyTQt.tqt import TQImage, TQMimeSourceFactory";
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
    out << ", tqUncompress";
#endif
    out << "\n";
    out << "\n";
    out << "\n";


    TQPtrList<EmbedImage> list_image;
    list_image.setAutoDelete( true );
    int image_count = 0;
    for ( it = images.begin(); it != images.end(); ++it ) {
	TQImage img;
	if ( !img.load( *it ) ) {
	    fprintf( stderr, "pytquic: cannot load image file %s\n", (*it).local8Bit().data() );
	    continue;
	}
	EmbedImage *e = new EmbedImage;
	e->width = img.width();
	e->height = img.height();
	e->depth = img.depth();
	e->numColors = img.numColors();
	e->colorTable = new TQRgb[e->numColors];
	e->alpha = img.hasAlphaBuffer();
	memcpy(e->colorTable, img.colorTable(), e->numColors*sizeof(TQRgb));
	TQFileInfo fi( *it );
	e->name = fi.fileName();
	e->cname = TQString("image_%1").arg( image_count++);
	list_image.append( e );
	out << "# " << *it << "\n";
	TQString s;
	if ( e->depth == 1 )
	    img = img.convertBitOrder(TQImage::BigEndian);
	out << s.sprintf( "%s_data =",
			  (const char *)e->cname );
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
        e->compressed =
#endif
	    embedData( out, img.bits(), img.numBytes() );
	out << "\n";
	if ( e->numColors ) {
	    out << s.sprintf( "%s_ctable = [",
			      (const char *)e->cname );
	    embedData( out, e->colorTable, e->numColors );
	    out << "\n]\n\n";
	}
    }

    if ( !list_image.isEmpty() ) {
	out << indent << "embed_image_vec = [\n";
	++indent;
	EmbedImage *e = list_image.first();
	while ( e ) {
	    out << indent << "["
		<< e->width << ","
		<< e->height << ","
		<< e->depth << ","
		<< e->cname << "_data,"
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
                << e->compressed << ","
#endif
		<< e->numColors << ",";
	    if ( e->numColors )
		out << e->cname << "_ctable,";
	    else
		out << "None,";
	    if ( e->alpha )
		out << "1,";
	    else
		out << "0,";
	    out << "\"" << e->name << "\"],\n";
	    e = list_image.next();
	}
	--indent;
	out << indent << "]\n";

	out << "\n"
	    "\n"
	    "def uic_findImage(name):\n";
	++indent;
	out << indent << "global embed_image_vec\n";
	out << "\n";
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
	out << indent << "for (w,h,depth,data,comp,nrCol,colTab,alpha,iname) in embed_image_vec:" << endl;
#else
	out << indent << "for (w,h,depth,data,nrCol,colTab,alpha,iname) in embed_image_vec:" << endl;
#endif
	++indent;
	out << indent << "if iname == str(name):" << endl;
	++indent;
#ifndef TQT_NO_IMAGE_COLLECTION_COMPRESSION
	// Keep the data object alive until we have a deep copy of the image.
	out << indent << "data = tqUncompress(data).data()\n";
#endif
	out << indent << "img = TQImage(data,w,h,depth,colTab,nrCol,TQImage.BigEndian).copy()\n";
	out << "\n";
	out << indent << "if alpha:\n";
	++indent;
	out << indent << "img.setAlphaBuffer(1)\n";
	--indent;
	out << "\n";
	out << indent << "return img\n";
	--indent;
	--indent;
	out << endl;
	out << indent << "return TQImage()\n";
	--indent;
	out << "\n";
	out << "\n";

	out << indent << "class MimeSourceFactory_" << cProject << "(TQMimeSourceFactory):\n";
	++indent;
	out << indent << "def __init__(self):\n";
	++indent;
	out << indent << "TQMimeSourceFactory.__init__(self)\n";
	out << "\n";
	--indent;
	out << indent << "def data(self,abs_name):\n";
	++indent;
	out << indent << "d = TQMimeSourceFactory.data(self,abs_name)\n";
	out << indent << "if d or abs_name.isNull():\n";
	++indent;
	out << indent << "return d\n";
	--indent;
	out << "\n";
	out << indent << "img = uic_findImage(abs_name)\n";
	out << indent << "if not img.isNull():\n";
	++indent;
	out << indent << "self.setImage(abs_name,img)\n";
	--indent;
	out << "\n";
	out << indent << "return TQMimeSourceFactory.data(self,abs_name)\n";
	--indent;
	--indent;
	out << "\n";
	out << "\n";

	out << indent << "factory = None\n";
	out << "\n";

	out << indent << "def tqInitImages_" << cProject << "():\n";
	++indent;
	out << indent << "global factory\n";
	out << "\n";
	out << indent << "if factory is None:\n";
	++indent;
	out << indent << "factory = MimeSourceFactory_" << cProject << "()\n";
	out << indent << "TQMimeSourceFactory.defaultFactory().addFactory(factory)\n";
	--indent;
	--indent;
	out << "\n";
	out << "\n";

	out << indent << "def tqCleanupImages_" << cProject << "():\n";
	++indent;
	out << indent << "global factory\n";
	out << "\n";
	out << indent << "if factory is not None:\n";
	++indent;
	out << indent << "TQMimeSourceFactory.defaultFactory().removeFactory(factory)\n";
	out << indent << "factory = None\n";
	--indent;
	--indent;
	out << "\n";
	out << "\n";

	out << indent << "class StaticInitImages_" << cProject << ":\n";
	++indent;
	out << indent << "def __init__(self):\n";
	++indent;
	out << indent << "self.cleanup = tqCleanupImages_" << cProject << "\n";
	out << indent << "tqInitImages_" << cProject << "()\n";
	--indent;
	out << "\n";
	out << indent << "def __del__(self):\n";
	++indent;
	out << indent << "self.cleanup()\n";
	--indent;
	--indent;
	out << "\n";
	out << "\n";

	out << "staticImages = StaticInitImages_" << cProject << "()\n";
    }
}
