/*
   Copyright (c) 2005 Boudewijn Rempt <boud@valdyas.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <tqbuttongroup.h>
#include <tqnamespace.h>
#include <tqtoolbutton.h>
#include <tqlabel.h>
#include <tqtooltip.h>
#include <tqlayout.h>
#include <tqpixmap.h>
#include <tqtoolbar.h>
#include <tqdockwindow.h>

#include <kdebug.h>
#include <tdeparts/event.h>
#include <tdelocale.h>
#include <tdetoolbar.h>
#include <kiconloader.h>
#include <kseparator.h>
#include <tdeaction.h>
#include <tdeactioncollection.h>
#include <tdeactionclasses.h>

#include <KoMainWindow.h>
#include "KoToolBox.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KoToolBox::KoToolBox( TDEMainWindow *mainWin, const char* name, TDEInstance* instance, int numberOfTooltypes )
    : TDEToolBar( mainWin, TQt::DockLeft, false, name, true, true), m_instance(instance)
{
    setFullSize( false );
    setMargin(2);

    m_buttonGroup = new TQButtonGroup( 0L );
    m_buttonGroup->setExclusive( true );
    connect( m_buttonGroup, TQ_SIGNAL( pressed( int ) ), this, TQ_SLOT( slotButtonPressed( int ) ) );

    m_tools.setAutoDelete( true );
    // Create separate lists for the various sorts of tools
    for (int i = 0; i < numberOfTooltypes ; ++i) {
        ToolList * tl = new ToolList();
        m_tools.append(tl);
    }
}

KoToolBox::~KoToolBox()
{
    delete m_buttonGroup;
}


void KoToolBox::slotPressButton( int id )
{
    m_buttonGroup->setButton( id );
    slotButtonPressed( id );
}

void KoToolBox::slotButtonPressed( int id )
{
    if( id != m_buttonGroup->selectedId() && m_buttonGroup->selected() ) {
        m_buttonGroup->selected()->setDown( false );
    }
    m_idToActionMap.at(id)->activate();

}

void KoToolBox::registerTool( TDEAction *tool, int toolType, TQ_UINT32 priority )
{
    uint prio = priority;
    ToolList * tl = m_tools.at(toolType);
    while( (*tl)[prio] ) prio++;
    (*tl)[prio] = tool;
}

TQToolButton *KoToolBox::createButton(TQWidget * parent,  const char* iconName, TQString tooltip)
{
    TQToolButton *button = new TQToolButton(parent);

    if ( iconName && *iconName ) {
        TQPixmap pixmap = BarIcon( iconName, m_instance );
        button->setPixmap( pixmap );
        button->setToggleButton( true );
    }

    if ( !tooltip.isEmpty() ) {
        TQToolTip::add( button, tooltip );
    }
    return button;
}


void KoToolBox::setupTools()
{
    int id = 0;
    // Loop through tooltypes
    for (uint i = 0; i < m_tools.count(); ++i) {
        ToolList * tl = m_tools.at(i);

        if (!tl) continue;
        if (tl->isEmpty()) continue;

        ToolArea *tools = new ToolArea( this );
        ToolList::Iterator it;
        for ( it = tl->begin(); it != tl->end(); ++it )
        {
            TDEAction *tool = it.data();
            if(! tool)
                continue;
            TQToolButton *bn = createButton(tools->getNextParent(), tool->icon().latin1(), tool->toolTip());
            tools->add(bn);
            m_buttonGroup->insert( bn, id++ );
            m_idToActionMap.append( tool );
        }
        if (i < m_tools.count() -1) addSeparator();
        m_toolBoxes.append(tools);
    }
    // select first (select tool)
    m_buttonGroup->setButton( 0 );
    m_numberOfButtons = id;
}


void KoToolBox::setOrientation ( TQt::Orientation o )
{
    if ( barPos() == Floating ) { // when floating, make it a standing toolbox.
        o = o == TQt::Vertical ? TQt::Horizontal : TQt::Vertical;
    }

    TQDockWindow::setOrientation( o );

    for (uint i = 0; i < m_toolBoxes.count(); ++i) {
        ToolArea *t = m_toolBoxes.at(i);
        t->setOrientation(o);
    }
}


void KoToolBox::enableTools(bool enable)
{
    if (m_tools.isEmpty()) return;
    if (!m_buttonGroup) return;
    if (m_numberOfButtons <= 0) return;

    for (uint i = 0; i < m_tools.count(); ++i) {
        ToolList * tl = m_tools.at(i);

        if (!tl) continue;

        ToolList::Iterator it;
        for ( it = tl->begin(); it != tl->end(); ++it )
            if (it != 0 && it.data())
                it.data()->setEnabled(enable);
    }
    m_buttonGroup->setEnabled(enable);
    for (TQ_UINT32 i = 0; i < m_numberOfButtons; ++i) {
        m_buttonGroup->find( i )->setEnabled( enable );
    }
}

void KoToolBox::slotSetTool(const TQString & toolname)
{
    for (uint i = 0; i < m_idToActionMap.count(); ++i) {
        TDEAction * a = m_idToActionMap.at(i);
        if (!a || a->name() != toolname) continue;

        m_buttonGroup->setButton(i);
        return;

    }
}


// ----------------------------------------------------------------
//                         class ToolArea


ToolArea::ToolArea(TQWidget *parent)
    : TQWidget(parent), m_left(true)
{
    m_layout = new TQBoxLayout(this, TQBoxLayout::LeftToRight, 0, 0, 0);
    TQWidget *w = new TQWidget(this);
    m_layout->addWidget(w);

    TQGridLayout *grid = new TQGridLayout(w, 2, 2);
    m_leftRow = new TQWidget(w);
    grid->addWidget(m_leftRow, 0, 0);
    m_leftLayout = new TQBoxLayout(m_leftRow, TQBoxLayout::TopToBottom, 0, 1, 0);

    w = new TQWidget(this);
    m_layout->addWidget(w);

    grid = new TQGridLayout(w, 2, 2);
    m_rightRow = new TQWidget(w);
    grid->addWidget(m_rightRow, 0, 0);
    m_rightLayout = new TQBoxLayout(m_rightRow, TQBoxLayout::TopToBottom, 0, 1, 0);
}


ToolArea::~ToolArea()
{
}


void ToolArea::add(TQWidget *button)
{
    if (m_left)
        m_leftLayout->addWidget(button);
    else
        m_rightLayout->addWidget(button);
    button->show();
    m_left = !m_left;
}


TQWidget* ToolArea::getNextParent()
{
    if (m_left)
        return m_leftRow;
    return m_rightRow;
}


void ToolArea::setOrientation ( TQt::Orientation o )
{
    TQBoxLayout::Direction  dir = (o != TQt::Horizontal 
				  ? TQBoxLayout::TopToBottom
				  : TQBoxLayout::LeftToRight);
    m_leftLayout->setDirection(dir);
    m_rightLayout->setDirection(dir);

    m_layout->setDirection(o == TQt::Horizontal
			   ? TQBoxLayout::TopToBottom
			   : TQBoxLayout::LeftToRight);
}


#include "KoToolBox.moc"
