/*
Copyright 2003,04 Aris Adamantiadis

This file is part of the SSH Library

The SSH Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The SSH Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the SSH Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
MA 02110-1301, USA. */

#ifndef _LIBSSH_H
#define _LIBSSH_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <sys/select.h> /* for fd_set * */
#include <sys/types.h>
#define LIBSSH_VERSION "libssh-0.11-dev"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct string_struct STRING;
typedef struct buffer_struct BUFFER;
typedef struct public_key_struct PUBLIC_KEY;
typedef struct private_key_struct PRIVATE_KEY;
typedef struct ssh_options_struct SSH_OPTIONS;
typedef struct channel_struct CHANNEL;
typedef struct ssh_session SSH_SESSION;
typedef struct ssh_kbdint SSH_KBDINT;

/* integer values */
typedef u_int32_t u32;
typedef u_int16_t u16;
typedef u_int64_t u64;
typedef u_int8_t u8;

/* the offsets of methods */
#define KEX_ALGO 0
#define KEX_HOSTKEY 1
#define KEX_CRYPT_C_S 2
#define KEX_CRYPT_S_C 3
#define KEX_MAC_C_S 4
#define KEX_MAC_S_C 5
#define KEX_COMP_C_S 6
#define KEX_COMP_S_C 7
#define KEX_LANG_C_S 8
#define KEX_LANG_S_C 9

#define SSH_AUTH_SUCCESS 0
#define SSH_AUTH_DENIED 1
#define SSH_AUTH_PARTIAL 2
#define SSH_AUTH_INFO 3
#define SSH_AUTH_ERROR -1

#define SSH_SERVER_ERROR -1
#define SSH_SERVER_NOT_KNOWN 0
#define SSH_SERVER_KNOWN_OK 1
#define SSH_SERVER_KNOWN_CHANGED 2
#define SSH_SERVER_FOUND_OTHER 3

#ifndef MD5_DIGEST_LEN
    #define MD5_DIGEST_LEN 16
#endif
/* errors */

enum ssh_error {SSH_NO_ERROR, SSH_REQUEST_DENIED, SSH_INVALID_REQUEST, SSH_CONNECTION_LOST,SSH_FATAL,SSH_INVALID_DATA};
char *ssh_get_error(SSH_SESSION *session); /* returns a static char array */
enum ssh_error ssh_error_code(SSH_SESSION *session);
void ssh_say(int priority,char *format,...);
void ssh_set_verbosity(int num);

 /* There is a verbosity level */
 /* 3 : packet level */
 /* 2 : protocol level */
 /* 1 : functions level */
 /* 0 : important messages only */
 /* -1 : no messages */

/* in client.c */

SSH_SESSION *ssh_connect(SSH_OPTIONS *options);
void ssh_disconnect(SSH_SESSION *session);
int ssh_service_request(SSH_SESSION *session,char *service);
char *ssh_get_issue_banner(SSH_SESSION *session);
/* get copyright informations */
const char *ssh_copyright();
/* string.h */

/* You can use these functions, they won't change */
/* makestring returns a newly allocated string from a char * ptr */
STRING *string_from_char(const char *what);
/* it returns the string len in host byte orders. str->size is big endian warning ! */
int string_len(STRING *str);
STRING *string_new(u32 size);
/* string_fill copies the data in the string. it does NOT check for boundary so allocate enough place with new_string */
/* right before */
void string_fill(STRING *str,void *data,int len);
/* returns a newly allocated char array with the str string and a final nul caracter */
char *string_to_char(STRING *str);
STRING *string_copy(STRING *str);

/* deprecated */
void ssh_crypto_init();

/* useful for debug */
void ssh_print_hexa(char *descr,unsigned char *what, int len);
void ssh_get_random(void *,int);

/* this one can be called by the client to see the hash of the public key before accepting it */
int ssh_get_pubkey_hash(SSH_SESSION *session,char hash[MD5_DIGEST_LEN]);
STRING *ssh_get_pubkey(SSH_SESSION *session);

/* deprecated */
int pubkey_get_hash(SSH_SESSION *session,char hash[MD5_DIGEST_LEN]);

/* in connect.c */
int ssh_fd_poll(SSH_SESSION *session);
int ssh_select(CHANNEL **channels,CHANNEL **outchannels, int maxfd, fd_set *readfds, struct timeval *timeout);

void publickey_free(PUBLIC_KEY *key);

/* in keyfiles.c */

PRIVATE_KEY *privatekey_from_file(SSH_SESSION *session,char *filename,int type,char *passphrase);
void private_key_free(PRIVATE_KEY *prv);
STRING *publickey_from_file(char *filename,int *_type);
STRING *publickey_from_next_file(SSH_SESSION *session,char **pub_keys_path,char **keys_path,
                            char **privkeyfile,int *type,int *count);
int ssh_is_server_known(SSH_SESSION *session);
int ssh_write_knownhost(SSH_SESSION *session);

/* in channels.c */

/* this one is deprecated */
CHANNEL *open_session_channel(SSH_SESSION *session,int window,int maxpacket);
CHANNEL *channel_open_forward(SSH_SESSION *session,char *remotehost, int remoteport, char *sourcehost, int localport);
CHANNEL *channel_open_session(SSH_SESSION *session);
void channel_free(CHANNEL *channel);
int channel_request_pty(CHANNEL *channel);
int channel_request_pty_size(CHANNEL *channel, char *term,int cols, int rows);
int channel_change_pty_size(CHANNEL *channel,int cols,int rows);
int channel_request_shell(CHANNEL *channel);
int channel_request_subsystem(CHANNEL *channel, char *system);
int channel_request_env(CHANNEL *channel,char *name, char *value);
int channel_request_exec(CHANNEL *channel, char *cmd);
int channel_request_sftp(CHANNEL *channel);
int channel_write(CHANNEL *channel,void *data,int len);
int channel_set_write_handler(CHANNEL *channel,
    void (*write_fct)(CHANNEL *channel, void *data, int len, void *userdefined),
    void *user);
int channel_set_stderr_write_handler(CHANNEL *channel,
    void (*write_err_fct)(CHANNEL *channel, void *data, int len, void *userdefined),
    void *user);
int channel_send_eof(CHANNEL *channel);
int channel_read(CHANNEL *channel, BUFFER *buffer,int bytes,int is_stderr);
int channel_poll(CHANNEL *channel, int is_stderr);
int channel_close(CHANNEL *channel);
int channel_read_nonblocking(CHANNEL *channel, char *dest, int len, int is_stderr);
int channel_is_open(CHANNEL *channel);
/* in options.c */

SSH_OPTIONS *options_new();
SSH_OPTIONS *options_copy(SSH_OPTIONS *opt);
int options_set_wanted_method(SSH_OPTIONS *opt,int method, char *list);
void options_set_username(SSH_OPTIONS *opt,char *username);
void options_set_port(SSH_OPTIONS *opt, unsigned int port);
SSH_OPTIONS *ssh_getopt(int *argcptr, char **argv);
void options_set_host(SSH_OPTIONS *opt, const char *host);
/* don't connect to host, use fd instead */
void options_set_fd(SSH_OPTIONS *opt, int fd);
void options_set_bindaddr(SSH_OPTIONS *opt, char *bindaddr);
void options_set_identity(SSH_OPTIONS *opt, char *identity);
void options_set_status_callback(SSH_OPTIONS *opt, void (*callback)(void *arg, float status), void *arg);
void options_set_timeout(SSH_OPTIONS *opt, long seconds, long usec);
void options_set_ssh_dir(SSH_OPTIONS *opt, char *dir);
void options_set_known_hosts_file(SSH_OPTIONS *opt, char *dir);
/* buffer.c */

BUFFER *buffer_new();
void buffer_free(BUFFER *buffer);
/* buffer_get returns a pointer to the begining of the buffer. no position is taken into account */
void *buffer_get(BUFFER *buffer);
/* same here */
int buffer_get_len(BUFFER *buffer);


/* in auth.c */
/* these functions returns AUTH_ERROR is some serious error has happened,
  AUTH_SUCCESS if success,
  AUTH_PARTIAL if partial success,
  AUTH_DENIED if refused */
int ssh_userauth_none(SSH_SESSION *session,char *username);
int ssh_userauth_password(SSH_SESSION *session,char *username,char *password);
int ssh_userauth_offer_pubkey(SSH_SESSION *session, char *username,int type, STRING *publickey);
int ssh_userauth_pubkey(SSH_SESSION *session, char *username, STRING *publickey, PRIVATE_KEY *privatekey);
int ssh_userauth_autopubkey(SSH_SESSION *session, const char *passphrase);
int ssh_userauth_kbdint(SSH_SESSION *session, char *user, char *submethods);
int ssh_userauth_kbdint_getnprompts(SSH_SESSION *session);
char *ssh_userauth_kbdint_getname(SSH_SESSION *session);
char *ssh_userauth_kbdint_getinstruction(SSH_SESSION *session);
char *ssh_userauth_kbdint_getprompt(SSH_SESSION *session, int i, char *echo);
void ssh_userauth_kbdint_setanswer(SSH_SESSION *session, unsigned int i, char *answer);

#ifdef __cplusplus
}
#endif
#endif /* _LIBSSH_H */
