/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2006 by the KFTPGrabber developers
 * Copyright (C) 2003-2006 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPWIDGETS_BROWSERDIRLISTER_H
#define KFTPWIDGETS_BROWSERDIRLISTER_H

#include <tqobject.h>

#include <kurl.h>
#include <kdirlister.h>

namespace KFTPSession {
  class Session;
}

namespace KFTPEngine {
  class Event;
}

namespace KFTPWidgets {

namespace Browser {

/**
 * This class is a wrapper around KDirLister to support remote listings
 * via engine sockets.
 *
 * @author Jernej Kos
 */
class DirLister : public TQObject {
TQ_OBJECT
public:
    /**
     * Class constructor.
     *
     * @param parent Parent object
     */
    DirLister(TQObject *parent);
    
    /**
     * Class destructor.
     */
    ~DirLister();
    
    /**
     * Set the remote session. You have to call this method before anything else
     * in this class!
     *
     * @param session A valid session
     */
    void setSession(KFTPSession::Session *session);
    
    /**
     * Changes the "show hidden files" setting.
     *
     * @param value True to enable showing hidden files, false otherwise
     */
    void setShowingDotFiles(bool value) { m_showHidden = value; }
    
    /**
     * Fetch a specific location.
     *
     * @param url The URL to fetch (can be local or remote)
     * @param reload If set to true listing cache will be invalidated
     */
    void fetchLocation(const KURL &url, bool reload);
    
    /**
     * Stop the current listing operation.
     */
    void stop();
    
    /**
     * Returns the items listed for the current URL. You should call this when receiving
     * the finished() signal.
     *
     * @return The items listed for the current URL
     */ 
    KFileItemList items() const;
protected:
    void enableLocal();
    void enableRemote();
    void disableRemote();
private:
    KDirLister *m_localLister;
    KFTPSession::Session *m_remoteSession;
    KFileItemList m_items;
    KURL m_url;
    bool m_showHidden;
private slots:
    void slotRemoteEngineEvent(KFTPEngine::Event *event);
signals:
    /**
     * Emitted when all items should be cleared.
     */
    void clear();
    
    /**
     * Emitted when the listing operation has been completed.
     */
    void completed();
    
    /**
     * Emitted when an item has to be removed.
     */
    void deleteItem(KFileItem *item);
    
    /**
     * Emitted when items should be refreshed.
     */
    void refreshItems();
    
    /**
     * Emitted when site changes from local to remote or vice-versa.
     *
     * @param url New site URL
     */
    void siteChanged(const KURL &url);
};

}

}

#endif
