/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-05-31
 * Description : implementation of item folder 
 * 
 * Copyright (C) 2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqfont.h>
#include <tqfontmetrics.h>
#include <tqpainter.h>
#include <tqpixmap.h>
#include <tqstyle.h>

// Local includes.

#include "thumbnailsize.h"
#include "albumsettings.h"
#include "folderview.h"
#include "folderitem.h"

namespace Digikam
{

FolderItem::FolderItem(TQListView* parent, const TQString& text, bool special)
          : TQListViewItem(parent, text)
{
    m_special = special;
    m_focus   = false;
}

FolderItem::FolderItem(TQListViewItem* parent, const TQString& text, bool special)
          : TQListViewItem(parent, text)
{
    m_special = special;
    m_focus   = false;
}

FolderItem::~FolderItem()
{
}

void FolderItem::setFocus(bool b)
{
    m_focus = b;
}

bool FolderItem::focus() const
{
    return m_focus;
}

void FolderItem::paintCell(TQPainter* p, const TQColorGroup& cg, int column, int width, int)
{
    FolderView *fv = dynamic_cast<FolderView*>(listView());
    if (!fv)
        return;

    TQFontMetrics fm(p->fontMetrics());

    TQString t           = text(column);
    int margin          = fv->itemMargin();
    int r               = margin;
    const TQPixmap* icon = pixmap(column);

    if (isSelected())
    {
        p->drawPixmap(0, 0, fv->itemBasePixmapSelected());
        p->setPen(cg.highlightedText());
    }
    else
    {
        p->drawPixmap(0, 0, fv->itemBasePixmapRegular());
        p->setPen(cg.text());
    }

    if (icon)
    {
        int xo = r;
        int yo = (height() - icon->height())/2;

        p->drawPixmap( xo, yo, *icon );

        r += icon->width() + 5 + fv->itemMargin();
    }

    if (m_special)
    {
        TQFont f(p->font());
        f.setItalic(true);
        p->setFont(f);

        p->setPen(isSelected() ? cg.color(TQColorGroup::LinkVisited) :
                  cg.color(TQColorGroup::Link));
    }

    TQRect br;
    p->drawText(r, 0, width-margin-r, height(), TQt::AlignLeft|TQt::AlignVCenter, t, -1, &br);

    if (m_special)
    {
        p->drawLine(br.right() + 2, height()/2, fv->width(), height()/2);
    }

    if (m_focus)
    {
        p->setPen(cg.link());
        TQRect r = fv->itemRect(this);
        p->drawRect(0, 0, r.width(), r.height());
    }
}

void FolderItem::setup()
{
    widthChanged();

    FolderView *fv = dynamic_cast<FolderView*>(listView());
    int h = fv->itemHeight();
    if (h % 2 > 0)
        h++;

    setHeight(h);
}

int FolderItem::id() const
{
    return 0;
}

// ------------------------------------------------------------------------------------

FolderCheckListItem::FolderCheckListItem(TQListView* parent, const TQString& text,
                                         TQCheckListItem::Type tt)
                   : TQCheckListItem(parent, text, tt)
{
    m_focus = false;
}

FolderCheckListItem::FolderCheckListItem(TQListViewItem* parent, const TQString& text,
                                         TQCheckListItem::Type tt)
                   : TQCheckListItem(parent, text, tt)
{
    m_focus = false;
}

FolderCheckListItem::~FolderCheckListItem()
{
}

void FolderCheckListItem::setFocus(bool b)
{
    m_focus = b;
}

bool FolderCheckListItem::focus() const
{
    return m_focus;
}

void FolderCheckListItem::paintCell(TQPainter* p, const TQColorGroup& cg, int column, int width, int)
{
    FolderView *fv = dynamic_cast<FolderView*>(listView());
    if (!fv)
        return;

    TQFontMetrics fm(p->fontMetrics());

    TQString t           = text(column);
    int margin          = fv->itemMargin();
    int r               = margin;
    const TQPixmap* icon = pixmap(column);

    int styleflags = TQStyle::Style_Default;
    switch (state())
    {
        case(TQCheckListItem::Off):
            styleflags |= TQStyle::Style_Off;
            break;
        case(TQCheckListItem::NoChange):
            styleflags |= TQStyle::Style_NoChange;
            break;
        case(TQCheckListItem::On):
            styleflags |= TQStyle::Style_On;
            break;
    }

    if (isSelected())
        styleflags |= TQStyle::Style_Selected;

    if (isEnabled() && fv->isEnabled())
        styleflags |= TQStyle::Style_Enabled;

    if ((type() == TQCheckListItem::CheckBox) ||
        (type() == TQCheckListItem::CheckBoxController))
    {
        int boxsize = fv->style().pixelMetric(TQStyle::PM_CheckListButtonSize, fv); 
        int x       = 3;
        int y       = (height() - boxsize)/2 + margin;
        r += boxsize + 4;

        p->fillRect(0, 0, r, height(), cg.base());

        fv->style().drawPrimitive(TQStyle::PE_CheckListIndicator, p,
                                  TQRect(x, y, boxsize, height()),
                                  cg, styleflags, TQStyleOption(this));
    }

    if (isSelected())
    {
        p->drawPixmap(r, 0, fv->itemBasePixmapSelected());
        p->setPen(cg.highlightedText());
    }
    else
    {
        p->drawPixmap(r, 0, fv->itemBasePixmapRegular());
        p->setPen(cg.text());
    }

    if (icon)
    {
        int xo = r;
        int yo = (height() - icon->height())/2;

        p->drawPixmap( xo, yo, *icon );

        r += icon->width() + fv->itemMargin();
    }

    p->drawText(r, 0, width-margin-r, height(), TQt::AlignLeft|TQt::AlignVCenter, t);

    if (m_focus)
    {
        p->setPen(cg.link());
        TQRect r = fv->itemRect(this);
        p->drawRect(0, 0, r.width(), r.height());
    }
}

void FolderCheckListItem::setup()
{
    widthChanged();

    FolderView *fv = dynamic_cast<FolderView*>(listView());
    int h = fv->itemHeight();
    if (h % 2 > 0)
        h++;

    setHeight(h);
}

}  // namespace Digikam
