/***************************************************************************
 *   Copyright (C) 2006 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef GENERIC_CONFIG_H
#define GENERIC_CONFIG_H

#include <tqvariant.h>
#include <tqsize.h>

#include "global.h"
#include "common/common/misc.h"

class GenericConfigPrivate;

class GenericConfig
{
public:
  GenericConfig(const TQString &group);
  ~GenericConfig();
  TQString group() const { return _group; }
  void rollback();

  TQString readEntry(const TQString &key, const TQString &def = TQString()) const;
  void writeEntry(const TQString &key, const TQString &value);
  void writeEntry(const TQString &key, const TQCString &value) { writeEntry(key, TQString(value)); }
  void writeEntry(const TQString &key, const char *value) { writeEntry(key, TQString(value)); }
  TQStringList readListEntry(const TQString &key, const TQStringList &defaultValues) const;
  void writeEntry(const TQString &key, const TQStringList &value);
  TQValueList<int> readIntListEntry(const TQString &key) const;
  void writeEntry(const TQString &key, const TQValueList<int> &value);
  TQSize readSizeEntry(const TQString &key, const TQSize *def = 0) const;
  void writeEntry(const TQString &key, const TQSize &value);
  bool readBoolEntry(const TQString &key, bool def) const;
  void writeEntry(const TQString &key, bool value);
  int readIntEntry(const TQString &key, int def = 0) const;
  void writeEntry(const TQString &key, int value);
  uint readUIntEntry(const TQString &key, uint def = 0) const { return qMax(0, readIntEntry(key, def)); }
  void writeEntry(const TQString &key, uint value) { writeEntry(key, int(value)); }
  template <typename Enum>
  Enum readEnumEntry(const TQString &key, Enum def = Enum::Nb_Types) const { return Enum::fromKey(readEntry(key, def.key())); }
  template <typename Enum>
  void writeEnumEntry(const TQString &key, Enum v) { writeEntry(key, v.key()); }
  TQVariant readVariantEntry(const TQString &key, const TQVariant &defValue) const;
  void writeEntry(const TQString &key, const TQVariant &value);

  static void deleteGroup(const TQString &group);

  struct ItemData {
    const char *key, *label;
    TQVariant defValue;
  };
  template <typename Type>
  TQVariant readVariantEntry(Type type) const { return readVariantEntry(type.data().key, type.data().defValue); }
  template <typename Type>
  void writeVariantEntry(Type type, const TQVariant &value) {
    Q_ASSERT( value.type()==type.data().defValue.type() );
    writeEntry(type.data().key, value);
  }

private:
  TQString _group;
  GenericConfigPrivate *_d;
};

#define BEGIN_DECLARE_CONFIG(Type) \
  BEGIN_DECLARE_ENUM(Type)

#define END_DECLARE_CONFIG(Type, group) \
  END_DECLARE_ENUM(Type, GenericConfig::ItemData) \
  inline TQVariant readConfigEntry(Type type) { \
    GenericConfig config(group); \
    return config.readVariantEntry<Type>(type); \
  } \
  inline void writeConfigEntry(Type type, const TQVariant &v) { \
    GenericConfig config(group); \
    config.writeVariantEntry<Type>(type, v); \
  }

#endif
