/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>
 
    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.
 
    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/


#include "PlayList.h"
#include "PlayListView.h"
#include "PlayListViewItem.h"
#include "document/ConfigGroups.h"
#include <tqlayout.h>

#include <tdelocale.h>
#include <tdeconfig.h>
#include <tdefiledialog.h>
#include <tdeglobal.h>
#include <kurl.h>
#include <tqframe.h>
#include <tqpushbutton.h>
#include <tqstring.h>
#include <tqstringlist.h>
#include <tqstrlist.h>
#include <tqvbox.h>
#include <tqwidget.h>
#include <tqdragobject.h>


namespace Rosegarden
{

PlayList::PlayList(TQWidget *parent, const char *name)
        : TQVBox(parent, name),
        m_listView(new PlayListView(this)),
        m_buttonBar(new TQFrame(this)),
        m_barLayout(new TQHBoxLayout(m_buttonBar)),
        m_playButton(0),
        m_moveUpButton(0),
        m_moveDownButton(0),
        m_deleteButton(0),
        m_clearButton(0)
{
    m_openButton = new TQPushButton(m_buttonBar);
    m_playButton = new TQPushButton(m_buttonBar);
    m_moveUpButton = new TQPushButton(m_buttonBar);
    m_moveDownButton = new TQPushButton(m_buttonBar);
    m_deleteButton = new TQPushButton(m_buttonBar);
    m_clearButton = new TQPushButton(m_buttonBar);
    m_barLayout->addWidget(m_openButton);
    m_barLayout->addWidget(m_playButton);
    m_barLayout->addWidget(m_moveUpButton);
    m_barLayout->addWidget(m_moveDownButton);
    m_barLayout->addWidget(m_deleteButton);
    m_barLayout->addWidget(m_clearButton);
    m_barLayout->addStretch();


    m_openButton ->setText(i18n("Add..."));
    m_playButton ->setText(i18n("Play"));
    m_moveUpButton ->setText(i18n("Move Up"));
    m_moveDownButton->setText(i18n("Move Down"));
    m_deleteButton ->setText(i18n("Delete"));
    m_clearButton ->setText(i18n("Clear"));

    connect(m_openButton, TQ_SIGNAL(clicked()),
            TQ_SLOT(slotOpenFiles()));

    connect(m_playButton, TQ_SIGNAL(clicked()),
            TQ_SLOT(slotPlay()));

    connect(m_deleteButton, TQ_SIGNAL(clicked()),
            TQ_SLOT(slotDeleteCurrent()));

    connect(m_clearButton, TQ_SIGNAL(clicked()),
            TQ_SLOT(slotClear()));

    connect(m_moveUpButton, TQ_SIGNAL(clicked()),
            TQ_SLOT(slotMoveUp()));

    connect(m_moveDownButton, TQ_SIGNAL(clicked()),
            TQ_SLOT(slotMoveDown()));

    connect(m_listView, TQ_SIGNAL(currentChanged(TQListViewItem*)),
            TQ_SLOT(slotCurrentItemChanged(TQListViewItem*)));

    connect(m_listView, TQ_SIGNAL(dropped(TQDropEvent*, TQListViewItem*)),
            TQ_SLOT(slotDropped(TQDropEvent*, TQListViewItem*)));

    restore();

    enableButtons(0);

}

PlayList::~PlayList()
{
    save();
}

void PlayList::slotOpenFiles()
{
    KURL::List kurlList =
        KFileDialog::getOpenURLs(":ROSEGARDEN",
                                 "audio/x-rosegarden audio/x-midi audio/x-rosegarden21",
                                 this,
                                 i18n("Select one or more Rosegarden files"));

    KURL::List::iterator it;

    for (it = kurlList.begin(); it != kurlList.end(); ++it) {
        new PlayListViewItem(m_listView, *it);
    }

    enableButtons(m_listView->currentItem());
}

void
PlayList::slotDropped(TQDropEvent *event, TQListViewItem* after)
{
    TQStrList uri;

    // see if we can decode a URI.. if not, just ignore it
    if (TQUriDrag::decode(event, uri)) {

        // okay, we have a URI.. process it
        // weed out non-rg files
        //
        for (TQString url = uri.first(); !url.isNull(); url = uri.next()) {
            if (url.right(3).lower() == ".rg")
                new PlayListViewItem(m_listView, after, KURL(url));

        }
    }

    enableButtons(m_listView->currentItem());
}

void PlayList::slotPlay()
{
    PlayListViewItem *currentItem = dynamic_cast<PlayListViewItem*>(m_listView->currentItem());

    if (currentItem)
        emit play(currentItem->getURL().url());
}

void PlayList::slotMoveUp()
{
    TQListViewItem *currentItem = m_listView->currentItem();
    TQListViewItem *previousItem = m_listView->previousSibling(currentItem);

    if (previousItem)
        previousItem->moveItem(currentItem);

    enableButtons(currentItem);
}

void PlayList::slotMoveDown()
{
    TQListViewItem *currentItem = m_listView->currentItem();
    TQListViewItem *nextItem = currentItem->nextSibling();

    if (nextItem)
        currentItem->moveItem(nextItem);

    enableButtons(currentItem);
}

void PlayList::slotClear()
{
    m_listView->clear();
    enableButtons(0);
}

void PlayList::slotDeleteCurrent()
{
    TQListViewItem* currentItem = m_listView->currentItem();
    if (currentItem)
        delete currentItem;
}

void PlayList::slotCurrentItemChanged(TQListViewItem* currentItem)
{
    enableButtons(currentItem);
}

void PlayList::enableButtons(TQListViewItem* currentItem)
{
    bool enable = (currentItem != 0);

    m_playButton->setEnabled(enable);
    m_deleteButton->setEnabled(enable);

    if (currentItem) {
        m_moveUpButton->setEnabled(currentItem != m_listView->firstChild());
        m_moveDownButton->setEnabled(currentItem != m_listView->lastItem());
    } else {
        m_moveUpButton->setEnabled(false);
        m_moveDownButton->setEnabled(false);
    }

    m_clearButton->setEnabled(m_listView->childCount() > 0);
}

void PlayList::save()
{
    TQStringList urlList;
    PlayListViewItem* item = dynamic_cast<PlayListViewItem*>(getListView()->firstChild());

    while (item) {
        urlList << item->getURL().url();
        item = dynamic_cast<PlayListViewItem*>(item->nextSibling());
    }

    TDEConfig *kc = TDEGlobal::config();
    TDEConfigGroupSaver cs(kc, PlayListConfigGroup);
    kc->writeEntry("Playlist Files", urlList);

    getListView()->saveLayout(kc, PlayListConfigGroup);
}

void PlayList::restore()
{
    TDEConfig *kc = TDEGlobal::config();
    getListView()->restoreLayout(kc, PlayListConfigGroup);

    TDEConfigGroupSaver cs(kc, PlayListConfigGroup);
    TQStringList urlList = kc->readListEntry("Playlist Files");

    for (TQStringList::Iterator it = urlList.begin();
            it != urlList.end(); ++it) {
        new PlayListViewItem(getListView(), KURL(*it));
    }
}

}
#include "PlayList.moc"
