;/***************************************************************************\
;* level1.s                                                                  *
;* Level 1 Debug Monitor Code                                                *
;* Copyright (C) 1991 Advanced RISC Machines Limited. All rights reserved.   *
;\***************************************************************************/

;;; RCS $Revision: 1.19 $
;;; Checkin $Date: 93/11/01 17:31:43 $
;;; Revising $Author: hmeekings $

                INCLUDE level0_h.s
                INCLUDE level1_h.s

                KEEP

                EXPORT  Level1Init

                AREA    |Level1|, CODE, READONLY

;************************************************************************
;*                         Initialisation Entry                         *
;************************************************************************

DefaultHandlers MOV     r0, #0
                ADR     r1, NullHandler
                MOV     r2, #Handlers
InitHandlers    STMIA   r2!,{r0, r1}
                CMP     r2, #Handlers + ErrorV * 2
                BLE     InitHandlers
                MOV     pc, r14

Level1Init      MOV     r5, lr ; save the link register
                MOV     r4, #ROMBase
                ADRL    r0, Level1RDI ; install RDI
                MOV     lr, pc
                LDR     pc, [r4, #InstallRDPV]
                ADRL    r0, Level1SWI ; install SWI
                MOV     r1, #SWIV :SHR: 2
                BL      InstallVector
                BL      BreakPt_Init
                BL      DefaultHandlers
                ADRL    r0, AddrExceptHandler
                MOV     r1, #AddrExceptV :SHR: 2
                BL      InstallVector
                ADRL    r0, DataAbortHandler
                MOV     r1, #DataAbortV :SHR: 2
                BL      InstallVector
                ADRL    r0, PrefAbortHandler
                MOV     r1, #PrefAbortV :SHR: 2
                BL      InstallVector
                ADRL    r0, ErrorHandler
                MOV     r1, #ErrorV :SHR: 2
                BL      InstallVector
                MOV     r0, #WorkBase
                ADD     r0, r0, #RAMBranchThrough0 - WorkBase
                ADRL    r1, BranchThrough0Handler
                MOV     r2, #9
01              LDR     r3, [r1], #+4
                STR     r3, [r0], #+4
                SUBS    r2, r2, #1
                BNE     %B01
                SUB     r0, r0, #9*4
                MOV     r1, #ResetV :SHR: 2
                BL      InstallVector
                MOV     r1, #WorkBase
                MOV     r0, #0
                STR     r0, [r1, #DebuggeeRunning - WorkBase]
                STR     r0, [r1, #FPAbortPossible - WorkBase] ; and SWIState too
                STR     r0, [r1, #FPRegsAddr - WorkBase]
                STR     r0, [r1, #ExpectedAbort - WorkBase]
                LDR     r14, [r1, #14 * 4] ; get lr back
                STR     r14, [r1, #DebuggeePC - WorkBase]
 [ {CONFIG} = 32
                MRS     r0, SPSR
                STR     r0, [r1, #DebuggeeCPSR - WorkBase]
 ]
                MOV     pc, r5 ; and return

NullHandler     MOV     pc, r14

;************************************************************************
;*                      Install a Vector Handler                        *
;************************************************************************
; Pass handlers address in r0, vector address in r1 (0 = reset, 8 = SWI
; etc) and the old handlers address is returned in r0

InstallVector   STMFD   sp!, {r1,r2}
                MOV     r2, #SoftVectors
                ADD     r2, r2, r1, LSL #2
                LDR     r1, [r2]
                STR     r0, [r2]
                MOV     r0, r1
                LDMFD   sp!, {r1,r2}
                MOV     pc, lr

;************************************************************************
;*                      Level 1 RDI Handler Code                        *
;************************************************************************

Level1RDI       STMFD   sp!, {lr}
                CMP     r0, #RDP_LastRequest
                BHI     RDIUnknown
                ADR     r1, RDIJumpTable
                LDR     pc, [r1, r0, LSL #2]

RDIJumpTable    DCD     RDIOpen         ; &0
                DCD     RDIClose        ; &1
                DCD     RDIRead         ; &2
                DCD     RDIWrite        ; &3
                DCD     RDICPURead      ; &4
                DCD     RDICPUWrite     ; &5
                DCD     RDICPRead       ; &6
                DCD     RDICPWrite      ; &7
                DCD     RDIUnknown      ; &8
                DCD     RDIUnknown      ; &9
                DCD     RDISetBreak     ; &a
                DCD     RDIClearBreak   ; &b
                DCD     RDISetWatch     ; &c
                DCD     RDIClearWatch   ; &d
                DCD     RDIUnknown      ; &e
                DCD     RDIUnknown      ; &f
                DCD     RDIExecute      ; &10
                DCD     RDIStep         ; &11
                DCD     RDIInfo         ; &12
                DCD     RDIOSOpReply    ; &13

;***************************************************************************\
;                                 RDIOpen                                   *
;***************************************************************************/

RDIOpen         BL      GetByte
                MOV     r4, r0

                BL      GetWord ; get the initial memory size

                TST     r4, #RDPOpen_ResetLine ; reset channel ?
                BLNE    GetByte ; get the line speed
                MOV     r7, r0

                MOV     r0, #RDP_Return
                BL      PutByte

                MOV     r5, #RDPOpen_SexBig
                MOV     r6, #WorkBase
                STR     r5, [r6, #EndianWord - WorkBase]
                LDRB    r5, [r6, #EndianWord - WorkBase]
                TST     r4, #RDPOpen_SexDontCare
                BNE     WhatSex
                AND     r6, r4, #RDPOpen_SexBig
                CMP     r5, r6
                MOVNE   r0, #RDIError_NoError
                MOVEQ   r0, #RDIError_WrongByteSex
                B       OutputSex

WhatSex         CMP     r5, #RDPOpen_SexBig
                MOVEQ   r0, #RDIError_LittleEndian
                MOVNE   r0, #RDIError_BigEndian

OutputSex       BL      PutByte

                TST     r4, #RDPOpen_ResetLine ; reset channel
                BEQ     %F01

                MOV     r0, r7
                CallI   r1, #ChannelSpeedV

01              BL      DefaultHandlers
                LDMFD   sp!, {pc} ; return

;***************************************************************************\
;                                RDIClose                                   *
;***************************************************************************/

RDIClose        MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                MOV     r0, #1
                CallI   r1, #ChannelSpeedV
 [ {CONFIG} = 26
                TEQP    pc, #NoInt | SVC26Mode
                NOP
 |
                MRS     r0, CPSR
                BIC     r0, r0, #NoInt | ModeMask
                ORR     r0, r0, #NoInt | SVC32Mode
                MSR     CPSR, r0
 ]
                MOV     r14, #WorkBase
                LDR     r0, [r14, #MMUType - WorkBase]
                CMP     r0, #&100

 [ {CONFIG} = 26
                MOVGE   r0, #Config26Bit
 |
                MOVGE   r0, #Config32Bit
 ]
                MCRGE   MMUCP, 0, r0, MMUControlReg, c0 ; configure arm6x0 xx-bit address & data + little endian
                LDMIA   r14, {r0-lr} ; restore registers
 [ {CONFIG} = 26
                TEQP    pc, #SVC26Mode
 |
                MSR     CPSR, #SVC32Mode
 ]
                B       .

;***************************************************************************\
;                                 RDIRead                                   *
;***************************************************************************/

RDIRead         BL      GetWord ; address
                MOV     r4, r0
                BL      GetWord ; number of bytes
                MOV     r5, r0
                MOV     r0, #RDP_Return
                BL      PutByte
                ADR     r0, RDIRead_Fail
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
                ORR     r0, r0, r1
 ]
                MOV     r1, #WorkBase
                STR     r0, [r1, #ExpectedAbort - WorkBase]
                MOV     r6, r5
RDIRead_Loop    LDRB    r0, [r4], #1
                BL      PutByte
                SUBS    r5, r5, #1
                BNE     RDIRead_Loop
                MOV     r1, #WorkBase
                MOV     r0, #0
                STR     r0, [r1, #ExpectedAbort - WorkBase]
             ;  MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDIRead_Fail
                SUB     r6, r6, r5
                MOV     r0, #0
RDIRead_FailLoop
                BL      PutByte
                SUBS    r5, r5, #1
                BNE     RDIRead_FailLoop
                MOV     r0, #RDIError_DataAbort
                BL      PutByte
                MOV     r0, r6
                BL      PutWord
                MOV     r1, #WorkBase
                MOV     r0, #0
                STR     r0, [r1, #ExpectedAbort - WorkBase]
                LDMFD   sp!, {pc}

;**************************************************************************\
;                                  RDIWrite                                 *
;***************************************************************************/

RDIWrite        BL      GetWord ; address
                MOV     r4, r0
                BL      GetWord ; number of bytes
                MOV     r5, r0
                ADR     r0, RDIWrite_Fail
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
                ORR     r0, r0, r1
 ]
                MOV     r1, #WorkBase
                STR     r0, [r1, #ExpectedAbort - WorkBase]
                MOV     r6, r5
WriteLoop       BL      GetByte
                STRB    r0, [r4], #1
                SUBS    r5, r5, #1
                BNE     WriteLoop
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r1, #WorkBase
                MOV     r0, #0
                STR     r0, [r1, #ExpectedAbort - WorkBase]
             ;  MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDIWrite_Fail
                SUB     r6, r6, r5
RDIWrite_FailLoop
                BL      GetByte
                SUBS    r5, r5, #1
                BNE     RDIWrite_FailLoop
                MOV     r0, #RDIError_DataAbort
                BL      PutByte
                MOV     r0, r6
                BL      PutWord
                MOV     r1, #WorkBase
                MOV     r0, #0
                STR     r0, [r1, #ExpectedAbort - WorkBase]
                LDMFD   sp!, {pc}


;***************************************************************************\
;                               RDICPURead                                  *
;***************************************************************************/

RDICPURead      BL      GetByte ; the mode
                MOV     r7, r0
                MOV     r6, #SavedRegs
                CMP     r7, #RDIMode_Curr
                BNE     RDICPURead_RealMode
                ; if we were asked for registers of the current mode, find out
                ; what that is (from one of the two appropriate places).
                LDR     r0, [r6, #DebuggeeRunning - SavedRegs]
                TEQ     r0, #0
 [ {CONFIG} = 26
                ANDNE   r7, lr, #ModeMask
                LDREQ   r7, [r6, #DebuggeePC - SavedRegs]
                ANDEQ   r7, r7, #ModeMask
 |
                MRSNE   r7, SPSR
                LDREQ   r7, [r6, #DebuggeeCPSR - SavedRegs]
                AND     r7, r7, #ModeMask
 ]
RDICPURead_RealMode
 [ {CONFIG} = 26
                MOV     r8, pc
                AND     r8, r8, #ModeMask
                TEQ     r8, #FIQ26Mode
 |
                ORR     r7, r7, #PSR_32Bit
                MRS     r8, CPSR        ; what mode are we in?
                AND     r8, r8, #ModeMask
                TEQ     r8, #FIQ32Mode
 ]

                MOVEQ   r8, #8*4        ; offset of first banked register in dump
                MOVNE   r8, #13*4

                BL      GetWord ; the mask
                MOV     r4, r0
                MOV     r5, #0
                MOV     r0, #RDP_Return
                BL      PutByte
RDICPURead_RegLoop1
                TST     r4, #1          ; get this register ?
                LDRNE   r0, [r6, r5]    ; yes
                BLNE    PutWord ; send it
                MOV     r4, r4, LSR #1  ; next register
                ADD     r5, r5, #4
                CMP     r5, r8          ; upto the first banked register ?
                BNE     RDICPURead_RegLoop1

                ; for modes other than user, we can't store the registers
                ; directly, but must switch to the appropriate mode.

                MOV     r6, #WorkingDumpEnd
                TST     r7, #SubModeMask
                BNE     RDICPURead_NonUserMode

                TEQ     r8, #8 * 4
                STMEQDB r6, {r8 - r14}^
                STMNEDB r6, {r13, r14}^
                B       RDICPURead_UserMode

RDICPURead_NonUserMode
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
 |
                MRS     r1, CPSR
 ]
                BIC     r0, r1, #ModeMask ; transfer interrupt disable state
                ORR     r7, r7, r0      ; to target mode
                MOV     r0, r8
 [ {CONFIG} = 26
                TEQP    pc, r7
 |
                MSR     CPSR, r7
 ]
                TEQ     r0, #8 * 4
                STMEQDB r6, {r8 - r14}
                STMNEDB r6, {r13, r14}
 [ {CONFIG} = 26
                TEQP    pc, r1
 |
                MSR     CPSR, r1
 ]

RDICPURead_UserMode
                SUB     r6, r6, #15*4
RDICPURead_RegLoop2
                TST     r4, #1          ; get this register ?
                LDRNE   r0, [r6, r5]    ; yes
                BLNE    PutWord         ; send it
                MOV     r4, r4, LSR #1  ; next register
                ADD     r5, r5, #4
                CMP     r5, #r15*4      ; up to the pc ?
                BNE     RDICPURead_RegLoop2
                MOV     r6, #SavedRegs

                TST     r4, #1          ; get R15 ?
                LDRNE   r0, [r6, #DebuggeePC - SavedRegs] ; yes
                BLNE    PutWord         ; send it

                TST     r4, #2          ; get the PC ?
                LDRNE   r0, [r6, #DebuggeePC - SavedRegs] ; yes
 [ {CONFIG} = 26
                BICNE   r0, r0, #CPSRMask
 ]
                BLNE    PutWord         ; send it

                TST     r4, #4          ; get CPSR ?
 [ {CONFIG} = 26
                LDRNE   r0, [r6, #DebuggeePC - SavedRegs] ; yes
                ANDNE   r0, r0, #CPSRMask
 |
                LDRNE   r0, [r6, #DebuggeeCPSR - SavedRegs] ; yes
 ]
                BLNE    PutWord         ; send it

RDICPURead_TrySPSR
                TST     r4, #8          ; get SPSR ?
                BEQ     RDICPURead_Finished

                ; in user mode, there's no spsr so give him the cpsr instead
                TST     r7, #ModeMask
 [ {CONFIG} = 26
                LDREQ   r0, [r6, #DebuggeePC - SavedRegs] ; yes
                ANDNE   r0, r0, #CPSRMask
 |
                LDREQ   r0, [r6, #DebuggeeCPSR - SavedRegs] ; yes
 ]
                BLEQ    PutWord ; and send it
                BEQ     RDICPURead_Finished

RDICPURead_DoRealSPSR
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
                TEQP    pc, r7
                NOP
                AND     r0, lr, #CPSRMask
                TEQP    pc, r1
 |
                MRS     r1, CPSR
                BIC     r0, r1, #ModeMask
                ORR     r0, r0, r7
                MSR     CPSR, r0
                MRS     r0, SPSR
                MSR     CPSR, r1
 ]
                BL      PutWord

RDICPURead_Finished
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDICPURead_Running     ; pc and cpsr are in the FIQ handler's save
                TST     r4, #1 ; get R15 ?
                LDRNE   r0, [r6, #r14*4] ; yes
                BLNE    PutWord ; and send it

                TST     r4, #2 ; get the PC ?
                LDRNE   r0, [r6, #r14*4] ; yes
                BLNE    PutWord ; and send it

                TST     r4, #4 ; get CPSR ?
 [ {CONFIG} = 26
                LDRNE   r0, [r6, #14*4] ; yes
                ANDNE   r0, r0, #CPSRMask
 |
                LDRNE   r0, [r6, #15*4] ; yes
 ]
                BLNE    PutWord ; and send it

                TST     r4, #8 ; get SPSR ?
                BEQ     RDICPURead_Finished

                ; in user mode, there's no spsr so give him the cpsr instead
                TST     r7, #ModeMask
                BNE     RDICPURead_DoRealSPSR

 [ {CONFIG} = 26
                LDRNE   r0, [r6, #14*4] ; yes
                ANDNE   r0, r0, #CPSRMask
 |
                LDR     r0, [r6, #15*4] ; yes
 ]
                BL      PutWord ; and send it
                B       RDICPURead_Finished

;***************************************************************************\
;                               RDICPUWrite                                 *
;***************************************************************************/

RDICPUWrite     BL      GetByte ; the mode
                MOV     r7, r0
                MOV     r6, #SavedRegs
                CMP     r7, #RDIMode_Curr
                BNE     RDICPUWrite_RealMode
                ; if we were asked for registers of the current mode, find out
                ; what that is (from one of the two appropriate places).
                LDR     r0, [r6, #DebuggeeRunning - SavedRegs]
                TEQ     r0, #0
 [ {CONFIG} = 26
                ANDNE   r7, lr, #ModeMask
                LDREQ   r7, [r6, #DebuggeePC - SavedRegs]
                ANDEQ   r7, r7, #ModeMask
 |
                MRSNE   r7, SPSR
                LDREQ   r7, [r6, #DebuggeeCPSR - SavedRegs]
                AND     r7, r7, #ModeMask
 ]
RDICPUWrite_RealMode
 [ {CONFIG} = 26
                MOV     r8, pc
                AND     r8, r8, #ModeMask
                TEQ     r8, #FIQ26Mode
 |
                ORR     r7, r7, #PSR_32Bit
                MRS     r8, CPSR        ; what mode are we in?
                AND     r8, r8, #ModeMask
                TEQ     r8, #FIQ32Mode
 ]
                MOVEQ   r8, #8*4        ; offset of first banked register in dump
                MOVNE   r8, #13*4

                BL      GetWord ; the mask
                MOV     r4, r0
                MOV     r5, #0
RDICPUWrite_RegLoop1
                TST     r4, #1          ; put this register ?
                BEQ     RDICPUWrite_Next1
                BL      GetWord
                STR     r0, [r6, r5]    ; yes
RDICPUWrite_Next1
                MOV     r4, r4, LSR #1  ; next register
                ADD     r5, r5, #4
                CMP     r5, r8          ; upto the first banked register ?
                BNE     RDICPUWrite_RegLoop1

                ; for modes other than user, we can't store the registers
                ; directly, but must switch to the appropriate mode.

                MOV     r6, #WorkingDumpEnd
                TST     r7, #SubModeMask
                BNE     RDICPUWrite_NonUserMode

                TEQ     r8, #8 * 4
                STMEQDB r6, {r8 - r14}^
                STMNEDB r6, {r13, r14}^
                B       RDICPUWrite_UserMode

RDICPUWrite_NonUserMode
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
 |
                MRS     r1, CPSR
 ]
                BIC     r0, r1, #ModeMask ; transfer interrupt disable state
                ORR     r7, r7, r0      ; to target mode
                MOV     r0, r8
 [ {CONFIG} = 26
                TEQP    pc, r7
 |
                MSR     CPSR, r7
 ]
                TEQ     r0, #8 * 4
                STMEQDB r6, {r8 - r14}
                STMNEDB r6, {r13, r14}
 [ {CONFIG} = 26
                TEQP    pc, r1
 |
                MSR     CPSR, r1
 ]

RDICPUWrite_UserMode
                SUB     r6, r6, #15*4
RDICPUWrite_RegLoop2
                TST     r4, #1          ; put this register ?
                BEQ     RDICPUWrite_Next2
                BL      GetWord
                STR     r0, [r6, r5]    ; yes
RDICPUWrite_Next2
                MOV     r4, r4, LSR #1  ; next register
                ADD     r5, r5, #4
                CMP     r5, #r15*4      ; up to the pc ?
                BNE     RDICPUWrite_RegLoop2
                ADD     r6, r6, #15*4

                ; now restore the banked registers from our working storage
                TST     r7, #SubModeMask
                BNE     RDICPUWrite_NonUserMode2

                TEQ     r8, #8 * 4
                LDMEQDB r6, {r8 - r14}^
                LDMNEDB r6, {r13, r14}^
                B       RDICPUWrite_TryR15

RDICPUWrite_NonUserMode2
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
 |
                MRS     r1, CPSR
 ]
                MOV     r0, r8
 [ {CONFIG} = 26
                TEQP    pc, r7
 |
                MSR     CPSR, r7
 ]
                TEQ     r0, #8 * 4
                LDMEQDB r6, {r8 - r14}
                LDMNEDB r6, {r13, r14}
 [ {CONFIG} = 26
                TEQP    pc, r1
 |
                MSR     CPSR, r1
 ]

RDICPUWrite_TryR15
                MOV     r6, #SavedRegs

                TST     r4, #1          ; put R15 ?
                BEQ     RDICPUWrite_TryPC
                BL      GetWord
                STR     r0, [r6, #DebuggeePC - SavedRegs] ; yes

RDICPUWrite_TryPC
                TST     r4, #2          ; put the PC ?
                BEQ     RDICPUWrite_TryPSR
                BL      GetWord
                STR     r0, [r6, #DebuggeePC - SavedRegs] ; yes

RDICPUWrite_TryPSR
                TST     r4, #4          ; put CPSR ?
                BEQ     RDICPUWrite_TrySPSR
RDICPUWrite_DoCPSR
                BL      GetWord
 [ {CONFIG} = 26
                LDR     r1, [r6, #DebuggeePC - SavedRegs] ; yes
                BIC     r1, r1, #CPSRMask
                AND     r0, r0, #CPSRMask
                ORR     r0, r0, r1
                STR     r0, [r6, #DebuggeePC - SavedRegs] ; yes
 |
                STR     r0, [r6, #DebuggeeCPSR - SavedRegs] ; yes
 ]

RDICPUWrite_TrySPSR
                TST     r4, #8          ; get SPSR ?
                BEQ     RDICPUWrite_Finished

                ; in user mode, there's no spsr so give him the cpsr instead
                TST     r7, #SubModeMask
                BICEQ   r4, r4, #8
                BEQ     RDICPUWrite_DoCPSR

RDICPUWrite_DoRealSPSR
                BL      GetWord

 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CPSRMask
                TEQP    pc, r7
                NOP
                BIC     lr, lr, #CPSRMask
                AND     r0, r0, #CPSRMask
                AND     lr, lr, r0
                TEQP    pc, r1
 |
                MRS     r1, CPSR
                MSR     CPSR, r7
                MSR     SPSR, r0
                MSR     CPSR, r1
 ]

RDICPUWrite_Finished
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDICPUWrite_Running     ; pc and cpsr are in the FIQ handler's save
                TST     r4, #1          ; get R15 ?
                BEQ     RDICPUWrite_TryPC_2
                BL      GetWord
                STR     r0, [r6, #r14*4] ; yes

RDICPUWrite_TryPC_2
                TST     r4, #2          ; get the PC ?
                BEQ     RDICPUWrite_TryCPSR_2
                BL      GetWord
                STR     r0, [r6, #r14*4] ; yes

RDICPUWrite_TryCPSR_2
                TST     r4, #4          ; get CPSR ?
                BEQ     RDICPUWrite_TrySPSR_2
RDICPUWrite_DoCPSR_2
                BL      GetWord
                STR     r0, [r6, #15*4] ; yes

RDICPUWrite_TrySPSR_2
                TST     r4, #8          ; get SPSR ?
                BEQ     RDICPUWrite_Finished

                ; in user mode, there's no spsr so give him the cpsr instead
                TST     r7, #SubModeMask
                BICEQ   r4, r4, #8
                BEQ     RDICPUWrite_DoCPSR_2

                B       RDICPUWrite_DoRealSPSR

;***************************************************************************\
;                               Find the FPE                                *
;***************************************************************************/

FindFPE         STMFD   sp!, {lr}
                BL      GetByte ; get the co-pro number
                CMP     r0, #1
                CMPNE   r0, #2
                BEQ     %F01
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_UnknownCoPro
                BL      PutByte
                MOV     r0, #0
                BL      PutByte
                ADD     sp, sp, #4
                LDMFD   sp!, {pc}

01              MOV     r6, #WorkBase
                STRB    r0, [r6, #FPAbortPossible - WorkBase] ; 1 or 2
                RFS     r0
                MOV     r1, #0
                STRB    r1, [r6, #FPAbortPossible - WorkBase]
                LDMFD   sp!, {pc}

; Returns with r0 the contents of the fpsr if the is an FPE
; r0 = -1 otherwise

;***************************************************************************\
;                               RDICPRead                                   *
;***************************************************************************/

RDICPRead       BL      FindFPE
                MOV     r4, r0
                BL      GetWord ; the mask
                MOV     r5, r0
                MOV     r0, #RDP_Return
                BL      PutByte
                CMN     r4, #1
                BEQ     RDICPRead_NoFP

                MOV     r6, #WorkingDump
                STR     r4, [r6, #8 * 12]
                MOV     r0, #0
                WFS     r0      ; clear exception masks (else storing a trapping
                                ; NAN will cause trouble).
                STFE    f0, [r6, #0 * 12]
                STFE    f1, [r6, #1 * 12]
                STFE    f2, [r6, #2 * 12]
                STFE    f3, [r6, #3 * 12]
                STFE    f4, [r6, #4 * 12]
                STFE    f5, [r6, #5 * 12]
                STFE    f6, [r6, #6 * 12]
                STFE    f7, [r6, #7 * 12]
                WFS     r4      ; restore fp status

RDICPRead_PutLoop
01              TST     r5, #1
                ADDEQ   r6, r6, #12
                BEQ     %F02
                LDR     r0, [r6], #+4
                BL      PutWord
                LDR     r0, [r6], #+4
                BL      PutWord
                LDR     r0, [r6], #+4
                BL      PutWord
                MOV     r0, #&a0000000  ; Extended format
                BL      PutWord
02              MOV     r5, r5, LSR #1
                CMP     r6, #WorkingDump + 8 * 12
                BNE     %B01

                TST     r5, #1 ; FPSR
                LDRNE   r0, [r6]
                BLNE    PutWord
                TST     r5, #2 ; FPCR
                MOVNE   r0, #0
                BLNE    PutWord

                CMN     r4, #1
                BEQ     %F01
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

01              MOV     r0, #RDIError_BadCoProState
                BL      PutByte
                MOV     r0, #0
                BL      PutByte
                LDMFD   sp!, {pc}

RDICPRead_NoFP  MOV     r0, #0
                MOV     r6, #WorkingDump + 8 * 12
                STR     r0, [r6, #0]
01              STR     r0, [r6, #-4]!
                CMP     r6, #WorkingDump
                BNE     %B01
                B       RDICPRead_PutLoop

;***************************************************************************\
;                               RDICPWrite                                  *
;***************************************************************************/

RDICPWrite      BL      FindFPE
                MOV     r4, r0
                BL      GetWord ; the mask
                MOV     r5, r0
                MOV     r0, #RDP_Return
                BL      PutByte
                CMN     r4, #1
                BEQ     RDICPWrite_NoFP

                MOV     r6, #WorkingDump
                STR     r4, [r6, #8 * 12]
                MOV     r0, #0
                WFS     r0      ; clear exception masks (else storing a trapping
                                ; NAN will cause trouble).
                STFE    f0, [r6, #0 * 12]
                STFE    f1, [r6, #1 * 12]
                STFE    f2, [r6, #2 * 12]
                STFE    f3, [r6, #3 * 12]
                STFE    f4, [r6, #4 * 12]
                STFE    f5, [r6, #5 * 12]
                STFE    f6, [r6, #6 * 12]
                STFE    f7, [r6, #7 * 12]

01              TST     r5, #1
                ADDEQ   r6, r6, #12
                BEQ     %F02
                BL      GetWord
                STR     r0, [r6], #+4
                BL      GetWord
                STR     r0, [r6], #+4
                BL      GetWord
                STR     r0, [r6], #+4
                BL      GetWord         ; format: must be 0xa0000000 (= extended)
02              MOV     r5, r5, LSR #1
                CMP     r6, #WorkingDump + 8 * 12
                BNE     %B01

                TST     r5, #1 ; FPSR
                BEQ     %F01
                BL      GetWord
                STR     r0, [r6]
01              TST     r5, #2 ; FPCR
                BLNE    GetWord

                LDFE    f7, [r6, #-12]!
                LDFE    f6, [r6, #-12]!
                LDFE    f5, [r6, #-12]!
                LDFE    f4, [r6, #-12]!
                LDFE    f3, [r6, #-12]!
                LDFE    f2, [r6, #-12]!
                LDFE    f1, [r6, #-12]!
                LDFE    f0, [r6, #-12]!
                LDR     r0, [r6, #12 * 8]
                WFS     r0

                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDICPWrite_NoFP MOV     r6, #0
01              TST     r5, #1
                BEQ     %F02
                BL      GetWord
                BL      GetWord
                BL      GetWord
                BL      GetWord
02              MOV     r5, r5, LSR #1
                ADD     r6, r6, #1
                CMP     r6, #8
                BNE     %B01
                TST     r5, #1 ; FPSR
                BLNE    GetWord
                TST     r5, #2 ; FPCR
                BLNE    GetWord
                MOV     r0, #RDIError_BadCoProState
                BL      PutByte
                MOV     r0, #0
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
;                          Breakpoint support code                          *
;***************************************************************************/

; Description of a breakpoint.  There are two ways breakpoints are set:
; ones at addresses below 2^26 are set to branch to a small code fragment in
; the breakpoint block (and can be used to debug code in any mode).
; Others are set the illegal instruction Break_Instr, and cannot be used
; to debug code executing in Undef mode.

                ^       0
bp_Instr        #       4
bp_Address      #       4
bp_CodeFragment1 #      4
bp_CodeFragment2 #      4
bp_CodeFragment3 #      4
bp_Size         #       0
BP_UnUsed       EQU     1       ; value for address field to mark unused
Break_Instr     EQU     &ee000000

BreakPt_Find  ; find breakpoint for address r0.
              ; return pointer to breakpoint record in r1
                MOV     r1, #BreakPoints
01              LDR     r2, [r1, #bp_Address]
                CMP     r2, r0
                MOVEQ   pc, lr
                ADD     r1, r1, #bp_Size
                CMP     r1, #BreakPointsEnd
                BNE     %B01
                MOV     r1, #0
                MOV     pc, lr

BreakPt_Init    ; Copy a small code fragment to low RAM (see below)
                MOV     r1, #Page0BreakPtCode
                ADR     r0, ROMPage0BreakPtCode
                MOV     r2, #BreakPoints - Page0BreakPtCode
01              LDR     r3, [r0], #+4
                STR     r3, [r1], #+4
                SUBS    r2, r2, #4
                BNE     %B01
                ; Mark each break point unused, and construct the code
                ; fragment for it.
                MOV     r0, #BP_UnUsed
                LDR     r2, BreakPtProto
                LDR     r3, BreakPtProto + 4
                MOV     r4, #(Page0BreakPtCode - BreakPoints - bp_CodeFragment3 - 8) :SHR: 2
                EOR     r4, r4, #&ea000000 :EOR: &3f000000
01              STR     r0, [r1, #bp_Address]
                STR     r2, [r1, #bp_CodeFragment1]
                STR     r3, [r1, #bp_CodeFragment2]
                STR     r4, [r1, #bp_CodeFragment3]
                ADD     r1, r1, #bp_Size
                SUB     r2, r2, #bp_Size
                SUB     r4, r4, #bp_Size :SHR: 2
                CMP     r1, #BreakPointsEnd
                BNE     %B01
 [ {CONFIG} = 26
                MOV     pc, lr
 |
                ; Install an undefined instruction handler, in case we need to
                ; set breakpoints that way.
                ADR     r0, UndefHandler
                MOV     r1, #UndefInstrV :SHR: 2
                B       InstallVector   ; then return
 ]

                ; Breakpoints inserted as branches:
                ; each breakpoint contains three instructions, whose general shape
                ; is as follows:
BreakPtProto    STR     r0, [pc, #WorkingDump - BreakPoints - bp_CodeFragment1 - 8]
                ADD     r0, pc, #bp_Instr - bp_CodeFragment2 - 8
                ; B       Page0BreakPtCode ; in low RAM

                ; (Leaving the processor mode unchanged, so the branch can't be
                ; to ROM because the mode may be a 26-bit one).  We save
                ; another working register, and set a 32-bit mode, then
                ; branch to ROM to finish the work.
ROMPage0BreakPtCode
                STR     r1, [pc, #WorkingDump+4 - Page0BreakPtCode-8]
 [ {CONFIG} = 26
                MOV     r1, pc                  ; the two instruction above
                AND     r1, r1, #CPSRMask       ; blew away the PSR bits
                ORR     r0, r0, r1              ; so put them back

                MOV     r1, pc
                TST     r1, #SubModeMask
                SWIEQ   SWI_EnterOS     ; to swi mode if in user mode
 |
                MRS     r1, CPSR
                STR     r1, [pc, #WorkingDump+15*4 - Page0BreakPtCode-16]
                ORR     r1, r1, #PSR_32Bit ; force 32 bit
                TST     r1, #SubModeMask
                ORREQ   r1, r1, #SVC26Mode
                SWIEQ   SWI_EnterOS     ; to swi mode if in user mode
                MSR     CPSR, r1
 ]
                LDR     pc, [pc, #-4]
                & ROMBranchBreakPointCode

ROMBranchBreakPointCode
 [ {CONFIG} = 26
                MOV     r1, pc
                AND     r1, r1, #CCMask
                ORR     r1, r1, #NoInt+SVC26Mode
                TEQP    pc, r1
 |
                MRS     r1, CPSR
                BIC     r1, r1, #NoInt | ModeMask
                ORR     r1, r1, #NoInt | SVC32Mode
                MSR     CPSR, r1
 ]
                MOV     r1, #WorkingDump
                ADD     r1, r1, #2*4
                STMIA   r1, {r2, r3, r4, r13, r14}

 [ {CONFIG} = 26
                BIC     r2, r0, #CPSRMask       ; mask out PSR bits
                LDR     r2, [r2, #bp_Address]
                AND     r0, r0, #CPSRMask
                ORR     r2, r2, r0              ; restore PSR bits
                STR     r2, [r1, #DebuggeePC - WorkingDump - 2*4]
 |
                LDR     r2, [r1, #15*4 - 2*4]
                STR     r2, [r1, #DebuggeeCPSR - WorkingDump - 2*4]
                LDR     r2, [r0, #bp_Address]
                STR     r2, [r1, #DebuggeePC - WorkingDump - 2*4]
 ]
                LDR     r3, [r1, #DebuggeeRunning - WorkingDump - 2*4]
                MOV     r2, #0
                STR     r2, [r1, #DebuggeeRunning - WorkingDump - 2*4]
                TST     r3, #Running_Async
                MOVEQ   r0, #RDP_Return
                MOVNE   r0, #RDP_Stopped
                MOV     r13, #BreakPointStackBase
                BL      PutByte
                MOV     r0, #RDIError_BreakpointReached
                BL      PutByte
                MOV     r0, #WorkingDump
                LDMIA   r0, {r0 - r4, r13, r14} ; reload saved registers,
 [ {CONFIG} = 26
                TEQP    pc, #User26Mode         ; enable interrupts,
 |
                MSR     CPSR, #User32Mode       ; enable interrupts,
 ]
                B       .                       ; and idle

 [ {CONFIG} = 32
UndefHandler    ; This wants to rely on as little as possible being correctly
                ; set up, so assumes no valid base register.
                STR     r0, [r0, -r0]
                MOV     r0, #WorkingDump
                STMIB   r0, {r1 - r3}
                LDR     r1, [r0, -r0]
                STR     r1, [r0]
                LDR     r1, [r0, #ResetVectorCopy - WorkingDump]
                STR     r1, [r0, -r0]
                LDRB    r1, [r0, #FPAbortPossible - WorkingDump]
                TEQ     r1, #0
                LDMNEIB r0, {r1 - r3}   ; restore saved registers
                MOVNE   r0, #-1         ; (except r0, which is a failed indicator)
                MOVNES  pc, r14         ; and continue at the next instruction

                LDR     r1, [r14, #-4]! ; faulting instruction
                TEQ     r1, #Break_Instr
                BNE     UndefH_ReallyUndef
                STR     r14, [r0, #DebuggeePC - WorkingDump]
                MRS     r14, SPSR
                STR     r14, [r0, #DebuggeeCPSR - WorkingDump]
                LDR     r1, [r0, #DebuggeeRunning - WorkingDump]
                MOV     r2, #0
                STR     r2, [r0, #DebuggeeRunning - WorkingDump]
                STR     r13, [r0, #4 * 4]
                MRS     r0, CPSR
                BIC     r0, r0, #NoInt | ModeMask
                ORR     r0, r0, #NoInt | Undef32Mode
                MSR     CPSR, r0
                TST     r1, #Running_Async
                MOVEQ   r0, #RDP_Return
                MOVNE   r0, #RDP_Stopped
                MOV     r13, #BreakPointStackBase
                BL      PutByte
                MOV     r0, #RDIError_BreakpointReached
                BL      PutByte
                MOV     r0, #WorkingDump
                LDMIA   r0, {r0 - r3, r13}      ; reload saved registers,
                MSR     CPSR, #User32Mode       ; enable interrupts,
                B       .                       ; and idle
 ]

;***************************************************************************\
;                               RDISetBreak                                 *
;***************************************************************************/

RDISetBreak     BL      GetWord         ; the address
                MOV     r4, r0
                BL      GetByte         ; type
                TEQ     r0, #0          ; only equal breakpoints are supported
                MOVNE   r5, #RDIError_UnimplementedType
                BNE     RDISetBreak_Error
                MOV     r0, r4
                BL      BreakPt_Find
                TEQ     r1, #0
                BNE     RDISetBreak_AlreadyThere
                MOV     r0, #BP_UnUsed
                BL      BreakPt_Find
                TEQ     r1, #0
                MOVEQ   r5, #RDIError_NoMorePoints
                BEQ     RDISetBreak_Error
                LDR     r2, [r4]
                STMIA   r1, {r2, r4}    ; bp_Instr, bp_Address

 [ {CONFIG} = 26
                ; a branch to the breakpoint block.
        ASSERT          bp_CodeFragment1 = 8
                SUB     r2, r1, r4
                MOV     r2, r2, LSR #2
                BIC     r2, r2, #&ff000000
                ORR     r2, r2, #&ea000000
                STR     r2, [r4]
 |
                TST     r4, #&fc000000
                ; if breakpoint is on a non-26 bit address, insert illegal inst
                LDRNE   r2, =Break_Instr
                STRNE   r2, [r4]
                ; else a branch to the breakpoint block.
        ASSERT          bp_CodeFragment1 = 8
                SUBEQ   r2, r1, r4
                MOVEQ   r2, r2, LSR #2
                BICEQ   r2, r2, #&ff000000
                ORREQ   r2, r2, #&ea000000
                STREQ   r2, [r4]
 ]
RDISetBreak_AlreadyThere
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDISetBreak_Error
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, r5
                BL      PutByte
                LDMFD   sp!, {pc}


;***************************************************************************\
;                               RDIClearBreak                               *
;***************************************************************************/

RDIClearBreak   BL      GetWord
                BL      BreakPt_Find
                TEQ     r1, #0
                BEQ     RDIClearBreak_NotFound
                LDR     r2, [r1, #bp_Instr]
                STR     r2, [r0]
                MOV     r2, #BP_UnUsed
                STR     r2, [r1, #bp_Address]
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

RDIClearBreak_NotFound
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoSuchPoint
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
;                               RDISetWatch                                 *
;***************************************************************************/

RDISetWatch     MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_UnimplementedMessage
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
;                               RDIClearWatch                               *
;***************************************************************************/

RDIClearWatch   MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_UnimplementedMessage
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
;                               RDIExecute                                  *
;***************************************************************************/

Running_Async   EQU     1

RDIOSOpReply    BL      GetByte
                MOV     r4, #0
                MOV     r6, #SavedRegs
                TEQ     r0, #0
                BEQ     RDIExecute_1    ; no return value
                TEQ     r0, #a_byte
                BNE     %F01
                BL      GetByte
                STR     r0, [r6, #0]
                B       RDIExecute_1
01              BL      GetWord
                STR     r0, [r6, #0]
                B       RDIExecute_1

RDIExecute      BL      GetByte
                MOV     r4, r0          ; 1 => asynchronous; 0 => synchronous
                MOV     r6, #SavedRegs
RDIExecute_1    LDR     r0, [r6, #DebuggeeRunning - SavedRegs]
                TEQ     r0, #0
                BNE     RDIExecute_AlreadyRunning  ; should this be an error?

                ; (see SignalStop)
                LDRB    r1, [r6, #SWIState - SavedRegs]
                TST     r1, #SWIState_IgnoreGo
                STRB    r0, [r6, #SWIState - SavedRegs]
                LDMNEFD sp!, {pc}

                LDR     r0, [r6, #DebuggeePC - SavedRegs]
                STR     r0, [r6, #r14*4]
 [ {CONFIG} = 32
                LDR     r0, [r6, #DebuggeeCPSR - SavedRegs]
                STR     r0, [r6, #15*4]
 ]
                ORR     r0, r4, #&100   ; remember whether synchronous and
                STR     r0, [r6, #DebuggeeRunning - SavedRegs]
                                        ; non-zero value to mark running
RDIExecute_AlreadyRunning
                TEQ     r4, #0
                LDMEQFD sp!, {pc}       ; synchronous : reply comes when
                                        ; debuggee stops
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
;                               RDIStep                                     *
;***************************************************************************/

RDIStep         MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_UnimplementedMessage
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
;                               RDIInfo                                     *
;***************************************************************************/

RDIInfo         BL      GetWord
                CMP     r0, #RDIInfo_Target
                BEQ     InfoTarget
                CMP     r0, #RDIInfo_Points
                BEQ     InfoPoints
                CMP     r0, #RDIInfo_Step
                BEQ     InfoStep
                CMP     r0, #RDIInfo_MMU
                BEQ     InfoMMU
                CMP     r0, #RDISignal_Stop
                BEQ     SignalStop
                CMP     r0, #RDIVector_Catch
                BEQ     VectorCatch
                CMP     r0, #RDIInfo_Cycles
                BEQ     Cycles
                SUB     r1, r0, #RDIInfo_ErrorP :AND: :NOT: &ff
                CMP     r1, #RDIInfo_ErrorP :AND: &ff
                BEQ     InfoErrorP
                CMP     r0, #RDISet_Cmdline
                BEQ     CmdLine
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_UnimplementedMessage
                BL      PutByte
                LDMFD   sp!, {pc}

InfoTarget      MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #&17
                BL      PutWord
                LDR     r0, IDString
                BL      PutWord
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}
IDString        DCB     "PIE1"

InfoPoints      MOV     r4, #0
                B       InfoReturnWord

InfoStep        MOV     r4, #0
                B       InfoReturnWord

InfoMMU         LDR     r4, IDString
                B       InfoReturnWord

InfoErrorP      MOV     r4, #ErrorP
                LDR     r4, [r4]
;               B       InfoReturnWord

InfoReturnWord  MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, r4
                BL      PutWord
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

SignalStop      MOV     r4, #SavedRegs
                LDR     r0, [r4, #DebuggeeRunning - SavedRegs]
                TEQ     r0, #0
                BEQ     SignalStop_NotRunning
                LDR     r0, [r4, #r14*4]
                STR     r0, [r4, #DebuggeePC - SavedRegs]
 [ {CONFIG} = 32
                LDR     r0, [r4, #15*4]
                STR     r0, [r4, #DebuggeeCPSR - SavedRegs]
 ]
                LDR     r1, [r4, #DebuggeeRunning - SavedRegs]
                MOV     r0, #0
                STR     r0, [r4, #DebuggeeRunning - SavedRegs]
                TST     r1, #Running_Async
                MOVEQ   r0, #RDP_Return
                MOVNE   r0, #RDP_Stopped
                BL      PutByte
                MOV     r0, #RDIError_UserInterrupt
                BL      PutByte
                LDMIA   r4, {r0 - r13}
 [ {CONFIG} = 26
                TEQP    pc, #User26Mode
 |
                MSR     CPSR, #User32Mode
 ]
                B       .

SignalStop_NotRunning
                ; if we were stopped waiting for the response to a SWI,
                ; the debugger will send its 'go' to resume from the SWI
                ; before it gets the response to the stop.  So we must
                ; arrange to ignore that 'go'.
                LDRB    r0, [r4, #SWIState - SavedRegs]
                TST     r0, #SWIState_Active
                BEQ     %F01
                ORRNE   r0, r0, #SWIState_IgnoreGo
                STRB    r0, [r4, #SWIState - SavedRegs]
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_UserInterrupt
                BL      PutByte
01              LDMIA   r4, {r0 - r13}
 [ {CONFIG} = 26
                TEQP    pc, #User26Mode
 |
                MSR     CPSR, #User32Mode
 ]
                B       .

VectorCatch     BL      GetWord
                MOV     r1, #WorkBase
                STR     r0, [r1, #CaughtVectors - WorkBase]
                MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

Cycles          MOV     r0, #RDP_Return
                BL      PutByte
                MOV     r0, #0
                BL      PutWord
                BL      PutWord
                BL      PutWord
                BL      PutWord
                BL      PutWord
                BL      PutWord
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

CmdLine         MOV     r0, #RDP_Return
                BL      PutByte
                LDR     r4, =CommandLine
                MOV     r5, #0
CmdLineStore    BL      GetByte
                STRB    r0, [r4, r5]
                ADD     r5, r5, #1
                CMP     r5, #256
                MOVEQ   r5, #0
                CMP     r0, #0
                BNE     CmdLineStore
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {pc}

;***************************************************************************\
; Unknown RDI handler                                                       *
;***************************************************************************/

RDIUnknown
 [ {CONFIG} = 26
                TEQP    pc, #NoInt | SVC26Mode
                NOP
 |
                MRS     r0, CPSR
                BIC     r0, r0, #NoInt | ModeMask
                ORR     r0, r0, #NoInt | SVC32Mode
                MSR     CPSR, r0
 ]
                MOV     r14, #WorkBase
                LDR     r0, [r14, #MMUType - WorkBase]
                CMP     r0, #&100
 [ {CONFIG} = 26
                MOVGE   r0, #Config26Bit
 |
                MOVGE   r0, #Config32Bit
 ]
                MCRGE   MMUCP, 0, r0, MMUControlReg, c0 ; configure arm6x0 xx-bit address & data + little endian
                MOV     r0, #ROMBase
                LDR     r0, [r0, #ROMResetV] ; get a return place
                STR     r0, [r14, #15*4] ; poke it into the return place
                LDMIA   r14, {r0-lr,pc} ; restore registers

;***************************************************************************\
;                         Fault Handlers                                    *
;***************************************************************************/

UndefH_ReallyUndef
                ; r0 - r3 dumped in WorkingDump. r14 already wound back
                ; to address faulting instruction.
                ; are we intercepting undefined instructions?
                LDR     r1, [r0, #CaughtVectors - WorkingDump]
                TST     r1, #1 :SHL: (UndefInstrV :SHR: 2)
                ADDNE   r0, r0, #r5*4
                STMNEIA r0, {r4 - r13}
                MOVNE   r10, #RDIError_UndefinedInstruction
                BNE     Vector_Caught_2

                ; call a registered handler.  Here we rely on the state of the
                ; world being reasonably sanitary.
                LDMIA   r0, {r0 - r3}
                STMFD   r13!, {r10 - r12, r14}
                MOV     r11, #Handlers
                ADD     r11, r11, #UndefInstrV * 2
                MOV     r14, pc
                LDMIA   r11, {r11, pc}

                ; if the handler returns, send the fault to the debugger anyway
                MOV     r10, #RDIError_UndefinedInstruction
Vector_Caught   MOV     r11, #WorkingDump
                STMIA   r11!, {r0 - r9}
                LDMFD   r13!, {r0 - r2, r14}    ; original r10-r12, r14
                STMIA   r11, {r0 - r2, r13}

Vector_Caught_2 MOV     r0, #WorkingDump
                STR     r14, [r0, #DebuggeePC - WorkingDump]
 [ {CONFIG} = 32
                MRS     r14, SPSR
                STR     r14, [r0, #DebuggeeCPSR - WorkingDump]
 ]
Vector_Caught_3 LDR     r1, [r0, #DebuggeeRunning - WorkingDump]
                MOV     r2, #0
                STR     r2, [r0, #DebuggeeRunning - WorkingDump]
                TST     r1, #Running_Async
                MOVEQ   r0, #RDP_Return
                MOVNE   r0, #RDP_Stopped
 [ {CONFIG} = 26
                MOV     r1, pc                  ; disable interrupts over the
                ORR     r1, r1, #NoInt          ; RDP message
                TEQP    pc, r1
                NOP
 |
                MRS     r1, CPSR                ; disable interrupts over the
                ORR     r1, r1, #NoInt          ; RDP message
                MSR     CPSR, r1
 ]
                MOV     r13, #BreakPointStackBase
                BL      PutByte
                MOV     r0, r10
                BL      PutByte
                MOV     r0, #WorkingDump
                LDMIA   r0, {r0 - r13}          ; reload saved registers,
 [ {CONFIG} = 26
                TEQP    pc, #User26Mode         ; enable interrupts,
 |
                MSR     CPSR_ctl, #User32Mode   ; enable interrupts,
 ]
                B       .                       ; and idle

PrefAbortHandler
                SUB     r14, r14, #4
                STMFD   r13!, {r10 - r12, r14}
                MOV     r11, #WorkingDump
                LDR     r11, [r11, #CaughtVectors - WorkingDump]
                TST     r11, #1 :SHL: (PrefAbortV :SHR: 2)

                MOVEQ   r11, #Handlers
                ADDEQ   r11, r11, #PrefAbortV * 2
                MOVEQ   r14, pc
                LDMEQIA r11, {r11, pc}

                MOV     r10, #RDIError_PrefetchAbort
                B       Vector_Caught

AddrExceptHandler
                SUB     r14, r14, #8
                STMFD   r13!, {r10 - r12, r14}

                MOV     r10, #WorkingDump
                LDR     r11, [r10, #ExpectedAbort - WorkingDump]
                TEQ     r11, #0
                STRNE   r11, [r13, #12]
                LDMNEIA r13!, {r10 - r12, r14}
                MOVNES  pc, r14

                MOV     r11, #WorkingDump
                LDR     r11, [r11, #CaughtVectors - WorkingDump]
                TST     r11, #1 :SHL: (AddrExceptV :SHR: 2)

                MOVEQ   r11, #Handlers
                ADDEQ   r11, r11, #AddrExceptV * 2
                MOVEQ   r14, pc
                LDMEQIA r11, {r11, pc}

                MOV     r10, #RDIError_AddressException
                B       Vector_Caught

ErrorHandler    LDR     r2, [r1, #16*4]
 [ {CONFIG} = 26
                AND     r2, r2, #CTLMask
                BIC     r14, r14, #CTLMask
                ORR     r14, r14, r2
 |
                MSR     SPSR_ctl, r2
 ]
                ADD     r8,r1,#8*4
                TST     r2, #SubModeMask
                LDMEQIA r8, {r8-r14}^
                BEQ     %F01
                ORR     r2,r2,#NoInt
 [ {CONFIG} = 26
                MOV     r3, pc
                TEQP    pc, r2
                NOP
                LDMIA   r8, {r8-r14}
                TEQP    pc, r3
                NOP
 |
                MRS     r3, CPSR
                MSR     CPSR, r2
                LDMIA   r8, {r8-r14}
                MSR     CPSR, r3
 ]
01              LDR     r14, [r1, #r15*4]
                STMFD   r13!, {r10-r12, r14}
                MOV     r10, r0
                LDMIA   r1, {r0 - r7}

                MOV     r11, #WorkingDump
                STR     r10, [r11, #ErrorP - WorkingDump]
                LDR     r11, [r11, #CaughtVectors - WorkingDump]
                TST     r11, #1 :SHL: (ErrorV :SHR: 2)

                MOVEQ   r11, #Handlers + ErrorV * 2
                MOVEQ   r14, pc
                LDMEQIA r11, {r11, pc}

                MOV     r10, #RDIError_Error
                B       Vector_Caught

BranchThrough0Handler
                ; Could be in any mode, so there's a problem preserving the
                ; registers (may not be able to write to 0).  But if we just
                ; go ahead and try, if it fails we land up in the data abort
                ; handler with no user state lost, so provided the data abort
                ; handler is aware all is well.  (Thus the next three
                ; instructions must be identical to those in the data abort
                ; handler).
                ; (This bit of code gets copied into page 0, to be executable
                ;  in 26-bit modes).
                STR     r0, [r0, -r0]
                MOV     r0, #WorkingDump
BranchThrough0AbortPC ; if the STR aborts, the pc points here ...
                STMIB   r0!, {r1 - r7}
 [ {CONFIG} = 26
                MOV     r1, pc
 |
                MRS     r1, CPSR
 ]
                TST     r1, #SubModeMask
                SWIEQ   SWI_EnterOS
BranchThrough0Aborted ; ... and execution resumes here
 [ {CONFIG} = 26
                TEQP    pc, #NoInt+SVC26Mode
 |
                MSR     CPSR, #NoInt+SVC32Mode
 ]
                LDR     pc, [pc, #-4]
                & BranchThrough0HandlerInROM

BranchThrough0HandlerInROM
                LDR     r2, [r0, -r0]
                STR     r2, [r0]
                LDR     r2, [r0, #ResetVectorCopy - WorkingDump]
                STR     r2, [r0, -r0]
                STMFD   r13!, {r10 - r12, r14}
                MOV     r10, r1
                LDMIA   r0, {r0 - r7}

                MOV     r11, #WorkingDump
                LDR     r11, [r11, #CaughtVectors - WorkingDump]
                TST     r11, #1 :SHL: (ResetV :SHR: 2)

                MOVEQ   r11, #Handlers + ResetV * 2
                MOVEQ   r14, pc
                LDMEQIA r11, {r11, pc}

                MOV     r11, #WorkingDump
                STMIA   r11!, {r0 - r9}
                LDMFD   r13!, {r0 - r2, r14}    ; original r10-r12, r14
                STMIA   r11, {r0 - r2, r13}

                MOV     r0, #WorkingDump
                MOV     r1, #0          ; a fairly meaningless value, but we
                                        ; don't know a helpful one.
                STR     r1, [r0, #DebuggeePC - WorkingDump]
 [ {CONFIG} = 32
                STR     r10, [r0, #DebuggeeCPSR - WorkingDump]
 ]
                MOV     r10, #RDIError_BranchThrough0
                B       Vector_Caught_3

DataAbortHandler
                ; The most complicated of the handlers, since there may be work
                ; to do to unwind base register writeback.  First we must get
                ; of the registers of the faulting mode.
                STR     r0, [r0, -r0]
                MOV     r0, #WorkingDump
                STMIB   r0!, {r1 - r7}

                ; Was the abort in the branch through 0 handler ?
                ; (see above).
                CMP     r14, #RAMBranchThrough0 + 8
                ADDEQ   pc, r14, #BranchThrough0Aborted - BranchThrough0AbortPC

                LDR     r1, [r0, -r0]
                STR     r1, [r0, #-r7*4]
                LDR     r1, [r0, #ResetVectorCopy - WorkingDump - r7*4]
                STR     r1, [r0, -r0]
 [ {CONFIG} = 26
                AND     r6, lr, #CPSRMask
                MOV     r7, pc
                AND     r7, r7, #CPSRMask
 |
                MRS     r6, SPSR
                MRS     r7, CPSR
 ]
                TST     r6, #15 ; some sort of user mode ?
                STMEQIB r0, {r8 - r14}^
                BEQ     %F01
                ORR     r6, r6, #NoInt
 [ {CONFIG} = 26
                TEQP    pc, r6
                STMIB   r0, {r8 - r14}
                TEQP    pc, r7
                NOP
 |
                MSR     CPSR, r6
                STMIB   r0, {r8 - r14}
                MSR     CPSR, r7
 ]
01              ; base = pc doesn't get fixed up, so it doesn't need to go in
                ; in the register dump.  But we do need to unwind the pipeline
                ; increment.
 [ {CONFIG} = 26
                BIC     r0, r14, #CPSRMask
                LDR     r0, [r0, #-8]           ; aborted instruction
                SUB     r14, r14, #8            ; fake r14 writeback
 |
                LDR     r0, [r14, #-8]!         ; aborted instruction
 ]
                MOV     r1, #WorkingDump        ; dumped registers

                AND     r2, r0, #&f0000 ; rn * 2^16
                AND     r3, r0, #&f000000
                CMP     r3, #&8000000
                CMPNE   r3, #&9000000
                BEQ     DAH_fixup_LDM_or_STM
                CMP     r3, #&c000000
                BEQ     DAH_fixup_CPDT_post
                CMP     r3, #&d000000
                BNE     DAH_FixedUp
                TST     r0, #&200000            ; CPDT pre
                BEQ     DAH_FixedUp             ; nothing to do if no WB
DAH_fixup_CPDT_post
                ; unlike LDR/STR writeback still happens on an abort
                CMP     r2, #&f0000
                BEQ     DAH_FixedUp             ; base = pc
                AND     r3, r0, #&ff
                B       DAH_UpdateRegister
DAH_fixup_LDM_or_STM
                TST     r0, #&200000
                CMPNE   r2, #&f0000
                BEQ     DAH_FixedUp             ; no WB or base = pc
                MOV     r4, r0, LSL #16
                MOV     r4, r4, LSR #16
                MOV     r3, #0
                B       %F02
01              ADD     r3, r3, #1              ; count number of registers
                RSB     r5, r4, #0
                AND     r5, r4, r5
                EOR     r4, r5, r4
02              CMP     r4, #0
                BNE     %B01
                CMP     r3, #0
                MOVEQ   r3, #&10
DAH_UpdateRegister
                TST     r0, #&800000
                LDR     r0, [r1, r2, LSR #16-2]
                ADDEQ   r0, r0, r3, ASL #2
                SUBNE   r0, r0, r3, ASL #2
                STR     r0, [r1, r2, LSR #16-2]
                ; base is now correct in WorkingDump.  If it's a banked
                ; register we also need to reload it.  For simplicity,
                ; we just reload all banked registers of the faulting
                ; mode, and dump the corresponding registers of abort
                ; mode.
                ADD     r1, r2, #13*4
                ANDS    r0, r6, #15
                LDMEQIA r1, {r13,r14}^
                STMEQIA r1, {r13}
                BEQ     DAH_FixedUp
 [ {CONFIG} = 26
                TEQP    pc, r6
 |
                MSR     CPSR, r6
 ]
                TEQ     r0, #FIQ26Mode
                SUBEQ   r1, r1, #13*4 - 8*4
                LDMEQIA r1, {r8 - r14}
                LDMNEIA r1, {r13, r14}
 [ {CONFIG} = 26
                TEQP    pc, r7
                TEQ     r0, #FIQ26Mode
 |
                MSR     CPSR, r7
 ]
                STMEQIA r1, {r8 - r13}
                STMNEIA r1, {r13}

DAH_FixedUp
                MOV     r0, #WorkingDump
                LDR     r1, [r0, #ExpectedAbort - WorkingDump]
                TEQ     r1, #0
                MOVNE   r14, r1
                LDMNEIA r0, {r0 - r12}
                MOVNES  pc, r14

                LDR     r1, [r0, #CaughtVectors - WorkingDump]
                TST     r1, #1 :SHL: (DataAbortV :SHR: 2)
                BEQ     DAH_CallHandler
                ; If we're sending the fault to the debugger, check whether the
                ; abort was inside the FP emulator (in which case, it should be
                ; reported as happening at the emulated instruction).
                SUBS    r10, r14, #&2000
                RSBHIS  r10, r14, #&8000
                MOV     r10, #RDIError_DataAbort
                BLS     Vector_Caught_2
DAH_AbortInFPE
 [ {CONFIG} = 26
                TEQP    pc, #NoInt+SVC26Mode
                NOP
 |
                MRS     r8, CPSR
                BIC     r8, r8, #NoInt | ModeMask
                ORR     r8, r8, #NoInt | SVC32Mode
                MSR     CPSR, r8
 ]
                MOV     r8, #WorkingDump
                LDMFD   r13!, {r0 - r7}
                STMIA   r8!, {r0 - r7}
                LDMFD   r13!, {r0 - r7}
                STMIA   r8, {r0 - r6}
 [ {CONFIG} = 26
                AND     r8, r14, #CPSRMask
                TEQP    pc, #NoInt+SVC26Mode
                NOP
                SUB     r14, r7, #4
                BIC     r14, r14, #CPSRMask
                ORR     r14, r14, r8
 |
                MRS     r8, SPSR
                MRS     r14, CPSR
                BIC     r14, r14, #NoInt | ModeMask
                ORR     r14, r14, #NoInt | Abort32Mode
                MSR     CPSR, r14
                SUB     r14, r7, #4
                MSR     SPSR, r8
 ]
                B       Vector_Caught_2

                ; call a registered handler.  Here we rely on the state of the
                ; world being reasonably sanitary.
DAH_CallHandler LDMIA   r0, {r0 - r12}
                STMFD   r13!, {r10 - r12, r14}
                MOV     r11, #Handlers + DataAbortV * 2
                MOV     r14, pc
                LDMIA   r11, {r11, pc}

 [ {CONFIG} = 26
                BIC     r10, r14, #CPSRMask
                SUBS    r10, r10, #&2000
                BICHI   r10, r14, #CPSRMask
                RSBHIS  r10, r10, #&8000
 |
                SUBS    r10, r14, #&2000
                RSBHIS  r10, r14, #&8000
 ]
                MOV     r10, #RDIError_DataAbort
                BLS     Vector_Caught
                B       DAH_AbortInFPE


SWIUnknown      MOV     r10, #RDIError_SoftwareInterrupt
                LDR     r14, [r13, #12]
                SUB     r14, r14, #4
                STR     r14, [r13, #12]
                B       Vector_Caught

;************************************************************************
;*                      Level 1 SWI Handler Code                        *
;************************************************************************

Level1SWI
 [ {CONFIG} = 26
                TEQP    pc, #NoInt | SVC26Mode
 |
                MSR     CPSR, #NoInt | SVC32Mode
 ]
                STR     r0, [r0, -r0] ; free up a register
 [ {CONFIG} = 26
                BIC     r0, r14, #CPSRMask
                LDR     r0, [r0, #-4] ; get the SWI instruction
 |
                LDR     r0, [r14, #-4] ; get the SWI instruction
 ]
                BIC     r0, r0, #&ff000000 ; extract the SWI number
                CMP     r0, #SWI_EnterOS
                BEQ     SWIEnterOS

                STMFD   sp!, {r10-r12,r14}
                MOV     r12, #WorkBase
                LDR     r10, [r12, #ResetVectorCopy - WorkBase]
                LDR     r0, [r12, -r12]
                STR     r10, [r12, -r12]
 [ {CONFIG} = 26
                AND     r10, r14, #CPSRMask
                BIC     r10, r10, #ModeMask + NoInt
                ORR     r10, r10, #SVC26Mode
                TEQP    pc, r10
                NOP
 |
                MRS     r10, SPSR
                BIC     r10, r10, #ModeMask + NoInt
                ORR     r10, r10, #SVC32Mode
                MSR     CPSR, r10
 ]
                ; are we intercepting SWIs?
                LDR     r10, [r12, #CaughtVectors - WorkBase]
                TST     r10, #1 :SHL: (SWIV :SHR: 2)
                MOVNE   r10, #RDIError_SoftwareInterrupt
                SUBNE   r14, r14, #4
                STRNE   r14, [sp, #3*4]
                BNE     Vector_Caught

 [ {CONFIG} = 26
                BIC     r10, r14, #CPSRMask
                LDR     r10, [r10, #-4] ; get the SWI instruction
 |
                LDR     r10, [r14, #-4] ; get the SWI instruction
 ]
                BIC     r10, r10, #&ff000000 ; clear all but the SWI number
                MOV     r11, #Handlers + SWIV * 2
                MOV     r14, pc
                LDMIA   r11, {r11, pc}
                LDR     r14, [sp, #12]
                CMP     r10, #MaxSWI1
                BHI     CheckBlock2
                LDR     r11, =SWIJumpTable1
                LDR     pc, [r11, r10, LSL #2]

CheckBlock2     SUB     r12, r10, #MinSWI2
                CMP     r12, #MaxSWI2 - MinSWI2
                BHI     SWIUnknown
                LDR     r11, =SWIJumpTable2
                LDR     pc, [r11, r12, LSL #2]

SWIJumpTable1   DCD     SWIWriteC               ; &0
                DCD     SWIUnknown              ; &1
                DCD     SWIWrite0               ; &2
                DCD     SWINewLine              ; &3
                DCD     SWIReadC                ; &4
                DCD     SWICLI                  ; &5
                DCD     SWIUnknown              ; &6
                DCD     SWIUnknown              ; &7
                DCD     SWIUnknown              ; &8
                DCD     SWIUnknown              ; &9
                DCD     SWIUnknown              ; &a
                DCD     SWIUnknown              ; &b
                DCD     SWIUnknown              ; &c
                DCD     SWIUnknown              ; &d
                DCD     SWIUnknown              ; &e
                DCD     SWIUnknown              ; &f
                DCD     SWIGetEnv               ; &10
                DCD     SWIExit                 ; &11
                DCD     SWIUnknown              ; &12
                DCD     SWIUnknown              ; &13
                DCD     SWIUnknown              ; &14
                DCD     SWIUnknown              ; &15
                DCD     SWIEnterOS              ; &16
                DCD     SWIBreak                ; &17
                DCD     SWIUnknown              ; &18
                DCD     SWIUnknown              ; &19
                DCD     SWIUnknown              ; &1a
                DCD     SWIUnknown              ; &1b
                DCD     SWIUnknown              ; &1c
                DCD     SWIUnknown              ; &1d
                DCD     SWIUnknown              ; &1e
                DCD     SWIUnknown              ; &1f
                DCD     SWIUnknown              ; &20
                DCD     SWIUnknown              ; &21
                DCD     SWIUnknown              ; &22

MaxSWI1         EQU                               &22

MinSWI2         EQU                               &60

SWIJumpTable2   DCD     SWIGetErrno             ; &60
                DCD     SWIClock                ; &61
                DCD     SWIClockInit            ; &62
                DCD     SWITime                 ; &63
                DCD     SWIRemove               ; &64
                DCD     SWIRename               ; &65

                DCD     SWIOpen                 ; &66
                DCD     SWIUnknown              ; &67
                DCD     SWIClose                ; &68
                DCD     SWIWrite                ; &69
                DCD     SWIRead                 ; &6a
                DCD     SWISeek                 ; &6b
                DCD     SWIFlen                 ; &6c

                DCD     SWIUnknown              ; &6d
                DCD     SWIIsTTY                ; &6e
                DCD     SWITmpNam               ; &6f

                DCD     SWIInstallHandler       ; &70
                DCD     SWIGenerateError        ; &71

MaxSWI2         EQU                               &71

SWIBreak                ; &17
                B       SWIExit

SWIInstallHandler       ; &70
                MOV     r10, #Handlers
                ADD     r10, r10, r0, ASL #3
                MOV     r11, r1
                MOV     r12, r2
                CMP     r0, #LastV :SHR: 2
                LDMLTIA r10, {r1, r2}
                STMLTIA r10, {r11, r12}
                LDMFD   sp!, {r10-r12, pc}^

SWIGenerateError        ; &71
                ADD     sp, sp, #4*4    ; pop saved registers
                MOV     r10, #SoftVectors
                LDR     pc, [r10, #ErrorV]

SWIClock                ; &61
                STMFD   sp!, {r1-r3}
                CallI   r1, #ReadTimerV ; read the time in centiseconds
                LDR     r1, =WorkBase
                LDR     r1, [r1, #TimeZero - WorkBase]
                SUB     r0, r0, r1
                LDMFD   sp!, {r1-r3, r10-r12, pc}^

SWIClockInit            ; &62
                STMFD   sp!, {r0-r3}
                CallI   r1, #ReadTimerV ; read the time in centiseconds
                LDR     r1, =WorkBase
                STR     r0, [r1, #TimeZero - WorkBase]
                LDMFD   sp!, {r0-r3, r10-r12, pc}^

SWIGetEnv               ; &10
                LDR     r0, =CommandLine
                LDR     r1, =RAMSize
                LDR     r1, [r1]
                MOV     r11, #a_word
                B       SWI_Send

SWIExit                 ; &11
                STMFD   sp!, {r0-r3}
                LDR     r0, =SavedRegs
                SUB     r14, r14, #4
 [ {CONFIG} = 32
                MRS     r1, SPSR
                STR     r1, [r0, #DebuggeeCPSR - SavedRegs]
 ]
                STR     r14, [r0, #DebuggeePC - SavedRegs]
                LDR     r2, [r0, #DebuggeeRunning - SavedRegs]
                MOV     r1, #0
                STR     r1, [r0, #DebuggeeRunning - SavedRegs]
                TST     r2, #Running_Async
                MOVEQ   r0, #RDP_Return
                MOVNE   r0, #RDP_Stopped
                BL      PutByte
                MOV     r0, #RDIError_NoError
                BL      PutByte
                LDMFD   sp!, {r0-r3, r10-r12, r14}
 [ {CONFIG} = 26
                TEQP    pc, #SVC26Mode
 |
                MSR     CPSR, #SVC32Mode
 ]
                B       .

; special case called from breakpoint code
SWIEnterOS              ; &16 ; tricky code follows, tread carefully
 [ {CONFIG} = 26
                MOV     r0, r14
                BIC     r0, r0, #ModeMask
                ORR     r0, r0, #SVC26Mode
                MOV     r14, r0
 |
                MRS     r0, SPSR
                AND     r0, r0, #IFConfig
                ORR     r0, r0, #SVC26Mode
                MSR     SPSR, r0
 ]
                MOV     r0, #BreakPointStackBase
                STMFD   r0!, {r14}
                LDR     r14, [r14, -r14]
                STMFD   r0!, {r14}
                MOV     r14, #WorkingDump
                LDR     r14, [r14, #ResetVectorCopy - WorkingDump]
                STR     r14, [r14, -r14]
                LDMFD   r0, {r0, pc}^

a_byte          EQU     1
a_word          EQU     2
a_string        EQU     3
a_addrlen       EQU     4

SWIRename       MOV     r11,#(a_string:SHL:3)+a_string  ; &65
                B       SWI_Send

SWIOpen         MOV     r11,#(a_byte:SHL:3)+a_string    ; &66
                B       SWI_Send

SWIWrite        MOV     r11,#(a_addrlen:SHL:3)+a_word   ; &69
                B       SWI_Send

SWIRead         MOV     r11,#(a_word:SHL:6)+(a_word:SHL:3)+a_word ; &6a
                B       SWI_Send

SWISeek         MOV     r11,#(a_word:SHL:3)+a_word      ; &6b
                B       SWI_Send

SWITmpNam       MOV     r11,#(a_word:SHL:3)+a_word      ; &6f
                B       SWI_Send

SWIGetErrno                                             ; &60
SWITime                                                 ; &63
SWINewLine                                              ; &3
SWIReadC        MOV     r11,#0                          ; &4
                B       SWI_Send

SWIWriteC       MOV     r11, #a_byte                    ; &0
                B       SWI_Send

SWIClose                                                ; &68
SWIFlen                                                 ; &6c
SWIIsTTY                                                ; &6e
                MOV     r11,#a_word
                B       SWI_Send

SWIRemove                                               ; &64
SWIWrite0                                               ; &2
SWICLI          MOV     r11,#a_string                   ; &5
                B       SWI_Send

SWI_Send        STMFD   sp!, {r0-r5}
                MOV     r0, #SavedRegs

 [ {CONFIG} = 26
                TEQP    pc, #NoInt+SVC26Mode
 |
                MRS     r1, CPSR
                BIC     r1, r1, #NoInt+SVC32Mode
                ORR     r1, r1, #NoInt+SVC32Mode
                MSR     CPSR, r1
                MRS     r1, SPSR
                STR     r1, [r0, #DebuggeeCPSR - SavedRegs]
 ]
                STR     r14, [r0, #DebuggeePC - SavedRegs]
                MOV     r1, #0
                STR     r1, [r0, #DebuggeeRunning - SavedRegs]
                MOV     r1, #SWIState_Active
                STRB    r1, [r0, #SWIState - SavedRegs]
                ADR     r1, SWI_Inspect_Fail
 [ {CONFIG} = 26
                MOV     r2, pc
                AND     r2, r2, #CPSRMask
                ORR     r1, r1, r2
 ]
                STR     r1, [r0, #ExpectedAbort - SavedRegs]
                MOV     r5, #0
                MOV     r1, r13
                MOV     r2, #SWIArgDesc
SWI_Inspect_Params
                LDR     r3, [r1], #+4
                STR     r3, [r2], #+8
                ANDS    r0, r11, #7
                BEQ     SWI_Inspect_Next
                CMP     r0, #a_string
                BNE     SWI_Inspect_NotString
                STR     r3, [r2, #-4]
                ADD     r12, r3, #1
01              LDRB    r4, [r3], #+1
                TEQ     r4, #0
                BNE     %B01
                SUB     r3, r3, r12
                STR     r3, [r2, #-8]
                B       SWI_Inspect_Next
SWI_Inspect_NotString
                CMP     r0, #a_addrlen
                STREQ   r3, [r2, #-4]
                LDREQ   r3, [r1], #+4
                STREQ   r3, [r2, #-8]
                MOVEQ   r0, #a_string
SWI_Inspect_Next
                ORR     r5, r0, r5, ROR #2
                MOV     r11, r11, LSR #3
                CMP     r2, #SWIArgDesc + 8*4
                BNE     SWI_Inspect_Params

                MOV     r0, #SavedRegs
                MOV     r1, #0
                STR     r1, [r0, #ExpectedAbort - SavedRegs]

                MOV     r0, #RDP_OSOp
                BL      PutByte
                MOV     r0, r10
                BL      PutWord
                MOV     r5, r5, ROR #26
                MOV     r0, r5
                BL      PutByte
                MOV     r11, #SWIArgDesc
SWI_Send_Params
                ANDS    r1, r5, #3
                BEQ     SWI_Params_Sent
                LDR     r0, [r11], #+8
                CMP     r1, #a_byte
                BNE     %F01
                BL      PutByte
                B       SWI_Send_Next

01              CMP     r1, #a_word
                BNE     %F01
                BL      PutWord
                B       SWI_Send_Next

01              CMP     r0, #32
                BGT     SWI_Send_String_L
                MOV     r4, r0
                BL      PutByte
                LDR     r12, [r11, #-4]
02              CMP     r4, #0
                BEQ     SWI_Send_Next
                LDRB    r0, [r12], #+1
                BL      PutByte
                SUBS    r4, r4, #1
                B       %B02
SWI_Send_String_L
                CMP     r0, #255
                BGE     %F01
                BL      PutByte
                B       %F02
01              MOV     r4, r0
                MOV     r0, #255
                BL      PutByte
                MOV     r0, r4
                BL      PutWord
02              LDR     r0, [r11, #-4]
                BL      PutWord
SWI_Send_Next   MOV     r5, r5, LSR #2
                B       SWI_Send_Params

SWI_Params_Sent
                LDMFD   sp!, {r0-r5, r10-r12, r14}
 [ {CONFIG} = 26
                TEQP    pc, #SVC26Mode
 |
                MSR     CPSR, #SVC32Mode
 ]
                B       .

SWI_Inspect_Fail
                MOV     r0, #SavedRegs
                MOV     r1, #0
                STR     r1, [r0, #ExpectedAbort - SavedRegs]
                LDMFD   sp!, {r0-r5, r10-r12}
                MOV     r14, #WorkingDump
                STMIA   r14!, {r0 - r12}
                MOV     r0, r14
                LDMFD   sp!, {r14}
                SUB     r14, r14, #4
                STMIA   r0, {sp, r14}
                B       DAH_FixedUp


;***************************************************************************\
; Get a byte from the debug channel                                         *
; Byte is returned in R0; R1, R2 and R3 are destroyed.                      *
;***************************************************************************/

GetByte         MOV     r1, #ROMBase
                LDR     pc, [r1, #GetByteV]

;***************************************************************************\
; Put a byte to the debug channel                                           *
; Byte is passed in R0; R1, R2 and R3 are destroyed.                        *
;***************************************************************************/

PutByte         MOV     r1, #ROMBase
                LDR     pc, [r1, #PutByteV]

;***************************************************************************\
; Get a word from the debug channel                                         *
; Word is returned in R0; R1, R2 and R3 are destroyed.                      *
;***************************************************************************/

GetWord         STMFD   sp!, {r4, lr}
                BL      GetByte
                MOV     r4, r0
                BL      GetByte
                ORR     r4, r4, r0, LSL #8
                BL      GetByte
                ORR     r4, r4, r0, LSL #16
                BL      GetByte
                ORR     r0, r4, r0, LSL #24
                LDMFD   sp!, {r4, pc}

;***************************************************************************\
; Put a word to the debug channel                                           *
; Word is in R0; R1, R2 and R3 are destroyed.                               *
;***************************************************************************/

PutWord         STMFD   sp!, {r4, lr}
                MOV     r4, r0
                BL      PutByte
                MOV     r0, r4, LSR #8
                BL      PutByte
                MOV     r0, r4, LSR #16
                BL      PutByte
                MOV     r0, r4, LSR #24
                BL      PutByte
                LDMFD   sp!, {r4, pc}

                END
