;       File:           Driver.a
;
;       Contains:       VL16C551 serial driver code
;
;       Written by:     Stephane THOMAS
;
;       Copyright:      1991 by VLSI Technology, Inc., all rights reserved.
;
;       Change History (most recent first):
;       V1.00   16 Jan 92       matches ARM S/W
;       V1.00   17 Dec 91       first draft


                GET     driver_h.s

                KEEP


; ----------------------------------------------
; Here is the serial chip default configuration.
; ----------------------------------------------
; This routine restores the 16C551 registers to the defaults.
; R0 shall not be changed by this routine (cf PutByte & GetByte)

ResetDriver
                LDR     r1, =SerialPortBase

                MOV     r0, #0                  ; for Serial chip : IER <- 0
                STRB    r0, [r1, #IER]          ; disable ITs

                ; enable Divisor Latch Access
                MOV     r0, #&80
                STRB    r0, [r1, #LCR]

                ; sets baud rate
                ; configures DLR (DLL and DLM)
                MOV     r0, #DLLvalue
                STRB    r0, [r1, #DLL]
                MOV     r0, #DLMvalue
                STRB    r0, [r1, #DLM]

                ; Disable Divisor Latch Access
                ; configures the Line Control Register
                MOV     r0, #LCRvalue
                STRB    r0, [r1, #LCR]

                ; Disable both Rx and Tx FIFOs
                ; FCR <- 0
                MOV     r0, #0
                STRB    r0, [r1, #FCR]

                MOV     r0, #MCRValue
                STRB    r0, [r1, #MCR]

                ; enables Interrupts
                MOV     r0, #IERvalue           ; for 16C551
                STRB    r0, [r1, #IER]

                MOV     r0, #1

                MOV     r15, r14


; --------------------------------
; Here are the interrupt handlers.
; --------------------------------
; The IRQ handler is slightly different from the FIQ one.
; The Serial Interrupt code may modify registers in Saved Regs area.
; For this case only, we must leave the IRQ handler with registers from
; Saved Regs area. For other interrupts, registers shall remain unchanged.
; Futhermore, the user has to be able to install his own interrupt handler,
; for any of the different IRQs available.
; That's why, instead of saving registers to the stack, IRQHandler will save
; them to the saved regs area.

IRQHandler
                STR     r14, [r14, -r14]        ; store lr at zero
                LDR     r14, =SavedRegs         ; R14 <- Saved Regs area
                STMIA   r14!, {r0-r13}          ; save R0-R13 in Saved Regs
                LDR     r0, [r0, -r0]           ; get old R14 in R0
                SUB     r0, r0, #4              ; R14 : ready for return
 [ {CONFIG} = 26
                STMIA   r14!, {r0}              ; store it too
 |
                MRS     r1, SPSR                ; get SPSR in R1
                STMIA   r14!, {r0-r1}           ; store them too
 ]              
                LDR     r0, =ResetVectorCopy    ; restore location zero
                LDR     r0, [r0]
                STR     r0, [r0, -r0]           ; store it
                LDR     r0, =IRQVectors         ; get IRQ Vect location in RAM
                LDR     r1, =IOBase             ; get I/O base
                MOV     r2, #WorkBase
                LDRB    r2, [r2, #SavedIRQMask - WorkBase]
                LDRB    r3, [r1, #IRQStatus]    ; R3 <- IRQ status
                AND     r3, r3, r2              ; mask out the invalid bits
                MOV     r2, #0                  ; r2 reset
01              MOVS    r3, r3, LSR #1          ; shift. Set C if IT
                STMDB   r13!, {r0-r3}           ; save R0-R3 to stack
                MOV     r14, r15                ; R14 points to
                                                ; LDMIA R13!,{R0-R3}
                LDRCS   r15, [r0, r2]           ; jumps to appropr. IRQ routine
                LDMIA   r13!, {r0-r3}           ; restore R0-R3 from stack
                ADD     r2, r2, #4              ; next vector
                CMP     r3, #0                  ; finished ?
                BEQ     %F02
                CMP     r2, #&20                ; Other IRQs to try ?
                BNE     %B01                    ; yes : branch
02              LDR     r14, =SavedRegs
 [ {CONFIG} = 32
                LDR     r0, [r14, #&3C]         ; pick up the CPSR
                MSR     SPSR, r0                ; ready to restore
 ]              
                LDMIA   r14, {r0-r12}           ; restore registers
                LDR     r14, [r14, #&38]        ; pick up R14
                MOVS    r15, r14                ; returns from IT now

FIQHandler      STMDB   r13!, {r0-r12,r14}      ; Stack up registers
                LDR     r0, =FIQVectors         ; get FIQ Vect location in RAM
                LDR     r1, =IOBase             ; get I/O base
                MOV     r2, #0                  ; R2 reset
                LDRB    r3, [r1, #FIQStatus]    ; R3 <- FIQ status
01              MOVS    r3, r3, LSR #1          ; shift. Set C if IT
                STMDB   r13!, {r0-r3}           ; save R0-R3 to stack
                MOV     r14, r15                ; R14 points to
                                                ; LDMIA R13!, {R0-R3}
                LDRCS   r15, [r0, r2]           ; jumps to appropr. FIQ routine
                LDMIA   r13!, {r0-r3}           ; restore R0-R3 from stack
                ADD     r2, r2, #4              ; next vector
                CMP     r3, #0                  ; finished ?
                BEQ     %F02
                CMP     r2, #&20                ; Other FIQs to try ?
                BNE     %B01                    ; yes : branch
02              LDMIA   r13!, {r0-r12,r14}      ; no : restore all regs
                SUBS    r15, r14, #4            ; returns from IT now



; ---------------------------------------------
; This routine sets the serial chip speed value
; ---------------------------------------------
; On entry : R0 value means :
;     R0 = 1  ->   9600 baud
;     R0 = 2  ->  19200 baud
;     R0 = 3  ->  38400 baud

DriverSpeed     STMDB   r13!, {r0-r3, r14}      ; save regs in stack
                LDR     r2, =SerialPortBase     ; get 16C551 address
01              LDRB    r1, [r2, #LSR]          ; get line status reg
                TST     r1, #TransmitterEmpty   ; Tx reg emty ?
                BEQ     %B01                    ; no : loop

                CMP     r0, #0                  ; really change speed ?
                BLE     %F02                    ; no : continue

                MOV     r1, #0                  ; disable ITs from serial chip
                STRB    r1, [r2, #IER]

                MOV     r1, #BR9600             ; default baud rate
                CMP     r0, #2                  ; 19200 bds ?
                MOVEQ   r1, #BR19200            ; yes
                CMP     r0, #3                  ; 3/400 bds ?
                MOVEQ   r1, #BR38400            ; yes


                LDRB    r3, [r2, #LCR]          ; get LCR value
                ORR     r3, r3, #&80            ; DLA <- 1(Access Divisor latch)
                STRB    r3, [r2, #LCR]          ; store LCR value

                ; configures DLR (DLL and DLM)
                STRB    r1, [r2, #DLL]
                MOV     r1, #0
                STRB    r1, [r2, #DLM]

                BIC     r3, r3, #&80            ; DLA <- 0
                STRB    r3, [r2, #LCR]          ; store LCR value


                ; re-enables Interrupts
                MOV     r1, #IERvalue           ; for 16C551
                STRB    r1, [r2, #IER]

02              LDMIA   r13!, {r0-r3, r15}      ; returns


; -----------------------------------------
; This is the serial port interrupt handler
; -----------------------------------------
; an IT from the serial port has been detected.

SerialInt
                STMDB   r13!, {r14}             ; save link register to stack
                LDR     r2, =SerialPortBase     ; get 16C551 address
                LDRB    r0, [r2, #IIR]          ; get Int identifcation reg
                TST     r0, #InterruptPending   ; IT Pending, 1=No IT
                BNE     SerialResume            ; No IT : we should never get
                                                ; here since we are in the
                                                ; interrupt handler !!!
                                                ; The only popsible sources off It for now are :
                                                ;       - Rx Line Status     -> b[2:1]=11
                                                ;       - Rx Data Available  -> b[2:1]=10
                AND     r0, r0, #IntTypeMask    ; keeps b[2:1] only
                CMP     r0, #RxLineStatus       ; Line Status IT ?
                BEQ     ROMReset                ; yes : go to routine
                CMP     r0, #RxDataAvailable    ; Data Available IT ?
                BLEQ    NewMessage              ; yes : go to routine



; restore context and returns from the routine
SerialResume
                LDMIA   r13!, {r14}             ; get R14 back from stack
                MOV     r15, r14                ; returns to IRQ handler
                                                ; which will restore the context

; a new character has been received.
; first read it, then pass it to the RDP handler
; the handler should return to SerialResume
NewMessage
                LDRB    r0, [r2, #RBR]          ; get char in R0 + clear int
                AND     r0, r0, #&FF            ; keep bytes only
                LDR     r1, =RDPHandlerV        ; load base address
                LDR     r15, [r1]               ; then call the handler


; --------
; TimerInt
; --------
; Timer will tick every 10 ms (1 centi second)
; donnt forget to unstack r0-r3 before returning
TimerInt        LDR     r2, =TimerVal           ; Get Timer memory location
                LDR     r1, [r2]                ; load the current value
                ADD     r1, r1, #1              ; increment it
                STR     r1, [r2]                ; store it back
                LDR     r2, =IOBase             ; reset Timer Interrupt
                MOV     r1, #&02
                STRB    r1, [r2, #IRQReset]
                MOV     r15, r14                ; returns to handler


; ---------
; ReadTimer
; ---------
ReadTimer       LDR     r0, =TimerVal           ; Get Timer mem location
                LDR     r0, [r0]                ; load current Ticker value
                MOV     r15, r14                ; returns


; --------------------------------------------------
; GetByte returns the next byte from the serial line
; --------------------------------------------------
; on exit : R0 contains the byte that has been received


GetByte         LDR     r2, =SerialPortBase     ; get 16C551 address
GetByteLoop     LDRB    r1, [r2, #IIR]          ; get Int identifcation reg
                TST     r1, #InterruptPending   ; 0DR     pc- Pending, 1=No IT
                BNE     GetByteLoop             ; wait until IT
                AND     r1, r1, #IntTypeMask    ; keeps b[2:1] only
                CMP     r1, #RxLineStatus       ; Line Status IT ?
                BEQ     ROMReset                ; yes : Hard Reset
GetChar         LDRB    r0, [r2, #RBR]          ; get char in R0 + clear int
                MOV     r15, r14                ; returns from GetByte

; ---------------------------------------
; PutByte sends a char to the serial line
; ---------------------------------------
; on entry : R0 contains the byte to be sent

PutByte         LDR     r2, =SerialPortBase     ; get 16C551 address
PutByteLoop     LDRB    r1, [r2, #IIR]          ; get Int identifcation reg
                AND     r1, r1, #IntTypeMask    ; keeps b[2:1] only
                CMP     r1, #RxLineStatus       ; Line Status IT ?
                BEQ     ROMReset                ; yes : Hard reset
SendChar        LDRB    r1, [r2, #LSR]          ; get line status reg
                TST     r1, #TransmHoldRegEmpty ; ready to send next char ?
                BEQ     PutByteLoop             ; no : loop
                STRB    r0,[r2, #THR]           ; send char
                MOV     r15, r14                ; returns from PutByte


; -----------------------------------
; SetLeds switches the Leds on or off
; -----------------------------------
; on entry : the 4 LSB of r0 contains the 4 LEDs description
;       - Bit 0 : =0 means LED1 off; =1 means LED1 on
;       - Bit 1 : =0   "   LED2 off; =1   "   LED2 on
;       - Bit 2 : =0   "   LED3 off; =1   "   LED3 on
;       - Bit 3 : =0   "   LED4 off; =1   "   LED4 on

SetLEDs         STMDB   r13!,{r0-r1,r14}        ; save regs
                LDR     r1, =ParallelPortBase   ; get 16C551 address
                EOR     r0, r0, #&F             ; inverts last 4 bits
                MOV     r0, r0, LSL #4          ; bits[7:4] in GPIO
                STRB    r0, [r1, #GPIO]         ; write GPIO set the leds
                LDMIA   r13!,{r0-r1,r15}        ; returns from SetLeds

                END
