/* vim: set sw=8 ts=8 si : */
/*
* Serial line daemon program for shutdown button and
* kicking of the watchdog. of your server.
* The program uses the serial interface lines
* DTR (output: data terminal ready), RTS (output:request to send)  
* and CD (input: carrier detect)
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License.
* See http://www.gnu.org/copyleft/ for details.
*
* Written by Guido Socher <guido@linuxfocus.org> 
*
*/
#define VERINFO "version 0.5"
#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>


static int fd = 0;
static int opt_t=0;
static int opt_d=0;
enum serialline {rts_power,dtr_kick};

/* Signal handler: TIOCM_DTR on  then exit */
void offandexit(int code)
{
	int state = TIOCM_DTR;
	ioctl(fd, TIOCMSET, &state);
	usleep(10000);
	syslog(LOG_NOTICE,"daemon stopped by signal, watchdog off");
	if (opt_d) printf("watchdog off, exit\n");
	closelog();
	exit(0);
}

/* Switch the RTS/DTS on or off */
int setwd( enum serialline whichline, int fd, int onoff, int *wdlinestate){
	int bitpat,rval;
	if (whichline == rts_power){
		bitpat=TIOCM_RTS;
	}else{
		bitpat=TIOCM_DTR;
	}
	if (onoff){
		/* on */
		*wdlinestate |= bitpat;
	}else{
		*wdlinestate &= ~bitpat;
	}
	rval=ioctl(fd, TIOCMSET, wdlinestate);
	if (rval == -1){
		syslog(LOG_ERR,"setwd ioctl failure");
	}
	return(*wdlinestate);
}

int kickwd(int fd,int *wdlinestate)
{
	setwd( dtr_kick, fd, 1, wdlinestate);
	usleep(10000);
	setwd( dtr_kick, fd, 0, wdlinestate);
	return(*wdlinestate);
}
/* kick the WD and then power it down
 */
int runshutdown(int fd,int *wdlinestate)
{
	kickwd(fd, wdlinestate);
	syslog(LOG_NOTICE,"Shutdown by buttonpress");
	setwd( rts_power, fd, 0, wdlinestate);
	if (opt_d==0){
		system("/sbin/shutdown -t2 -h now");
	}else{
		printf("in non debug mode I would now run \"/sbin/shutdown -t2 -h now\"\n");
	}
	kickwd(fd, wdlinestate);
	closelog();
	exit(0);
}

/* get the current state of the push button 
 * and return it. */
int getpushbutton(int fd)
{
	int state;
	ioctl(fd, TIOCMGET, &state);	/* read interface */
	if (state & TIOCM_CAR) {
		return (1);
	}
	return (0);
}

void help()
{
	printf("linuxwd -- watchdog and shutdown button control daemon\n\
USAGE: linuxwd [-hdt] serial-device\n\
\n\
OPTIONS:\n\
      -h this help\n\
      -d debug mode. Print messages to screen and do not \n\
         fork to background.\n\
      -t like debug mode but stop kicking the watchdog after\n\
         some time to test that hardware reset works.\n\
\n\
EXAMPLE:\n\
         linuxwd /dev/ttyS0\n\
\n\
Use command setserial -g /dev/ttyS* to see available ttyS devices.\n\
This program forks it self to background. It must have permissions\n\
to execute /sbin/shutdown\n\
");
#ifdef VERINFO
	puts(VERINFO);
#endif	 
exit(0); 
}


int main(int argc, char **argv)
{
	int lstate = 0;
	int i = 0;
	int rval;
	pid_t pid;

	/* The following things are used for getopt: */
	extern char *optarg;
	extern int optind;
	extern int opterr;
	int ch;

	opterr = 0;
	while ((ch = getopt(argc, argv, "dht")) != -1) {
		switch (ch) {
			case 'd':
				opt_d=1;
				break;
			case 't':
				opt_d=1;
				opt_t=1;
				break;
			case 'h':
				help(); /*no break, help does not return */
			case '?':
				fprintf(stderr, "ERROR: No such option. -h for help.\n");
				exit(1);
			/*no default action for case */
			}
	}
	if (optind != argc -1){
		/* exactly one argument must be given */
		help();
	}

	/* open device */
	openlog("linuxwd",LOG_PID,LOG_DAEMON);
	fd = open(argv[optind], O_RDWR | O_NDELAY);
	if (fd < 0) {
		fprintf(stderr, "ERROR: can not open \"%s\"\n", argv[optind]);
		exit(2);
	}

	/* catch signals INT and TERM and switch of WD before 
	 * terminating */
	signal(SIGINT, offandexit);
	signal(SIGTERM, offandexit);

	/* first set a defined state, all off */
	rval=ioctl(fd, TIOCMSET, &lstate);
	if (rval == -1){
		perror("linuxwd ioctl");
		syslog(LOG_ERR,"ioctl failed");
		exit(1);
	}
	/* switch on the power and kick the wd */
        setwd( rts_power, fd, 1, &lstate);
	if (opt_d) printf("watchdog started\n");
	if (opt_t) printf("watchdog: testmode, will loop very 4 seconds (normally it is every 12 seconds) \n");
	kickwd(fd, &lstate);
	syslog(LOG_NOTICE,"watchdog started");
	if (opt_d==0){
		/* now we fork to background */
		if ((pid = fork())<0){
			perror("linuxwd server fork");
			exit(1);
		}else if (pid > 0){
			/* parent dies */
			exit(0);
		}
	}
	/* we are the child and run as a daemon */
	rval=0;
	while(1){
		if (getpushbutton(fd)){
			if (opt_d) printf("button press detected\n");
			runshutdown(fd,&lstate);
		}
		rval++;
		if (opt_t){
			sleep(4);
		}else{
			sleep(8);
		}
		if (rval > 1000) rval=0;
		if (opt_t && rval > 5 ){
			printf("stop kickwd\n");
			if (rval==25){
				rval=0;
			}
			continue;
		}
		if (opt_d) printf("kickwd\n");
		kickwd(fd, &lstate);
	}
	/* we never get here */
	return (2);
}
